package ca.tecreations.text.ansi;

import ca.tecreations.StringTool;

import java.util.ArrayList;
import java.util.List;

/**
 *
 * @author Tim
 */
public class ANSICode {

    // ANSI Control Code Constants
    public static final int UNSET = -1;

    public static final int BELL = 0;
    public static final int BACKSPACE = 1;
    public static final int HORIZONTAL_TAB = 2;
    public static final int LINE_FEED = 3;
    public static final int VERTICAL_TAB = 4;
    public static final int FORM_FEED = 5;
    public static final int RETURN = 6;
    public static final int ESCAPE = 7;
    public static final int DELETE = 8;

    public static final int HOME = 9;
    public static final int UP_ONE_LINE_SCROLL = 10;
    public static final int SAVE_CURSOR_DEC = 11;
    public static final int RESTORE_CURSOR_DEC = 12;
    public static final int SAVE_CURSOR_SCO = 13;
    public static final int RESTORE_CURSOR_SCO = 14;
    public static final int REQUEST_CURSOR_POSITION = 15;

    public static final int UP = 16;
    public static final int DOWN = 17;
    public static final int LEFT = 18;
    public static final int RIGHT = 19;
    public static final int BEGIN_NEXT = 20;
    public static final int BEGIN_PREV = 21;
    public static final int COLUMN = 22;
    public static final int MOVE_TO = 23;

    public static final int ERASE_IN_DISPLAY = 24;
    public static final int ERASE_IN_LINE = 25;
    public static final int ERASE_CURSOR_TO_END = 26;
    public static final int ERASE_CURSOR_TO_HOME = 27;
    public static final int CLS = 28;
    public static final int ERASE_CURSOR_TO_END_OF_LINE = 29;
    public static final int ERASE_CURSOR_TO_START_OF_LINE = 30;
    public static final int ERASE_CURRENT_LINE = 31;

    public static final int RESET_ALL = 32;
    public static final int SET_BOLD = 33;
    public static final int SET_DIM = 34;
    public static final int SET_ITALIC = 35;
    public static final int SET_UNDERLINE = 36;
    public static final int SET_BLINK = 37;
    public static final int SET_REVERSE = 38;
    public static final int SET_VISIBLE = 39;
    public static final int SET_STRIKETHROUGH = 40;
    public static final int RESET_TEXT = 41;
    public static final int RESET_ITALIC = 42;
    public static final int RESET_UNDERLINE = 43;
    public static final int RESET_BLINK = 44;
    public static final int RESET_REVERSE = 45;
    public static final int RESET_VISIBLE = 46;
    public static final int RESET_STRIKETHROUGH = 47;

    public static final int GRAPHICS_MODE = 48;

    public static final int SET_SCREEN = 49;
    public static final int RESET_SCREEN = 50;

    public static final int TEXT = 51;
 
    public int id = UNSET;

    public static final int ESC = Integer.parseInt("1b", 16);

    public ANSIReader reader;
    public String block = "";
    public String code = "";
    public String text = "";
    public int num1;
    public int num2;
    List<Integer> codes = null;

    private String remainder = "";

    boolean debug = false;

    // data constructors
    public ANSICode(int id, String code) {
        this.id = id;
        if (id == TEXT) {
            this.text = code;
        } else {
            this.code = code;
        }
    }

    public ANSICode(int id, String code, String text) {
        this(id, code);
        this.text = text;
    }

    public ANSICode(int id, String code, int num1) {
        this(id, code);
        this.num1 = num1;
    }

    public ANSICode(int id, String code, int num1, int num2) {
        this(id, code, num1);
        this.num2 = num2;
    }

    public ANSICode(int id, String code, String text, List<Integer> codes) {
        this(id, code);
        this.text = text;
        this.codes = codes;
    }

    // parsing consttructor
    public ANSICode(ANSIReader reader, String block) {
        this.reader = reader;
        this.block = block;
        this.remainder = block;
        parse();
    }

    public void debugPrint(String tag, String code, String remainder, List<Integer> codes) {
        String codesStr = "";
        for (int i = 0; i < codes.size() - 1; i++) {
            codesStr += codes.get(i) + ",";
        }
        codesStr += codes.get(codes.size() - 1);
        if (debug) {
            printOut(tag + ": '" + StringTool.escape(code) + "', Text: '" + remainder + "'" + ", Codes: " + codesStr);
        }
    }

    public int getANSIId() {
        return id;
    }

    public int getANSIID() {
        return id;
    }

    public List<Integer> getCodes() {
        return codes;
    }

    public String getNextNumber(String text) {
        int index = 0;
        while (isDigit(text.charAt(index))) {
            index++;
        }
        return text.substring(0, index);
    }

    public String getRepresentation() {
        return block;
    }

    public String getText() {
        return text;
    }

    public boolean inList(char c, String list) {
        for (int i = 0; i < list.length(); i++) {
            if (list.charAt(i) == c) {
                return true;
            }
        }
        return false;
    }

    public boolean isDigit(char c) {
        return c >= '0' && c <= '9';
    }

    public boolean isEscape(char c) {
        return (int) c == 27 || c == 033 || c == '\u001b' || c == 0x1B;
    }

    public void parse() {
        if (debug) {
            printOut("Parse: Block: '" + StringTool.escape(block) + "'");
        }
        remainder = SharedCode.trimWhitespaceL(removeEscapeAtZero(block));

        // sanity check?
        if (remainder.length() > 0) {
            char second = remainder.charAt(0);
            // handle the groups
            if (second == 'P' || second == 0x90) {
                parseDCS(remainder);
            } else if (second == '[' || second == 0x9B) {
                parseCSI(remainder);
            } else if (second == ']' || second == 0x9D) {
                parseOSC(remainder);
            } else if (second == '=') {
                parseScreenMode(remainder);
            } else if (second == '?') {
                parsePrivate(remainder);

                // handle the individuals 
            } else if (second == 'M') {
                reader.addParsed(new ANSICode(UP_ONE_LINE_SCROLL, (char) ESC + "[M"));
            } else if (second == '7') {
                reader.addParsed(new ANSICode(SAVE_CURSOR_DEC, (char) ESC + "[7"));
            } else if (second == '8') {
                reader.addParsed(new ANSICode(RESTORE_CURSOR_DEC, (char) ESC + "[8"));

            } else {
                printErr("Parse: ERROR(1): '" + StringTool.escape(block) + "'");
            }
        }
    }

    public void parseCSI(String remainder) {
        // ESC [
        if (debug) {
            printOut("Parse CSI: remainder: '" + StringTool.escape(remainder) + "'");
        }
        remainder = SharedCode.trimWhitespaceL(remainder);
        if (debug) {
            printOut("Remainder: " + StringTool.escape(remainder));
        }
        char next = remainder.charAt(0);
        if (next == '[') {
            remainder = SharedCode.trimWhitespaceL(remainder.substring(1));
            next = remainder.charAt(0);
        } else {
            printErr("ParseCSI: ERROR(1): Next: '" + next + "', " + StringTool.escape(remainder) + "'");
        }
        //printOut("ParseCSI ESC [: remainder: '" + remainder + "'");
        if (next == 'H') {
            reader.addParsed(new ANSICode(HOME, (char) ESC + "[H"));
        } else if (next == 's') {
            reader.addParsed(new ANSICode(SAVE_CURSOR_SCO, (char) ESC + "[s"));
        } else if (next == 'u') {
            reader.addParsed(new ANSICode(RESTORE_CURSOR_SCO, (char) ESC + "[u"));
        } else if (next == 'J') {
            reader.addParsed(new ANSICode(ERASE_IN_DISPLAY, (char) ESC + "[J"));
        } else if (next == 'K') {
            reader.addParsed(new ANSICode(ERASE_IN_LINE, (char) ESC + "[K"));
        } else if (isDigit(next)) {
            String nextNum = getNextNumber(remainder);
            //printOut("NextNum: '" + nextNum + "'");
            int num = Integer.parseInt(nextNum);
            remainder = SharedCode.trimWhitespaceL(remainder.substring(nextNum.length()));
            char nextChar = remainder.charAt(0);
            code += num + nextChar;
            if (num == 6 && nextChar == 'n') {
                reader.addParsed(new ANSICode(REQUEST_CURSOR_POSITION, (char) ESC + "[6n"));
            } else if (nextChar == 'A') {
                reader.addParsed(new ANSICode(UP, (char) ESC + "[" + num + 'A', num));
            } else if (nextChar == 'B') {
                reader.addParsed(new ANSICode(DOWN, (char) ESC + "[" + num + 'B', num));
            } else if (nextChar == 'C') {
                reader.addParsed(new ANSICode(RIGHT, (char) ESC + "[" + num + 'C', num));
            } else if (nextChar == 'D') {
                reader.addParsed(new ANSICode(LEFT, (char) ESC + "[" + num + 'D', num));
            } else if (nextChar == 'E') {
                reader.addParsed(new ANSICode(BEGIN_NEXT, (char) ESC + "[" + num + 'E', num));
            } else if (nextChar == 'F') {
                reader.addParsed(new ANSICode(BEGIN_PREV, (char) ESC + "[" + num + 'F', num));
            } else if (nextChar == 'G') {
                reader.addParsed(new ANSICode(COLUMN, (char) ESC + "[" + num + 'G', num));
            } else if (nextChar == 'J') {
                if (num == 0) {
                    reader.addParsed(new ANSICode(ERASE_CURSOR_TO_END, (char) ESC + "[" + num + 'J'));
                } else if (num == 1) {
                    reader.addParsed(new ANSICode(ERASE_CURSOR_TO_HOME, (char) ESC + "[" + num + 'J'));
                } else if (num == 2) {
                    reader.addParsed(new ANSICode(CLS, (char) ESC + "[" + num + 'J'));
                } else {
                    printErr("ParseCSI:  ERROR(2): " + StringTool.escape(block));
                }
            } else if (nextChar == 'K') {
                if (num == 0) {
                    reader.addParsed(new ANSICode(ERASE_CURSOR_TO_END_OF_LINE, (char) ESC + "[" + num + 'K'));
                } else if (num == 1) {
                    reader.addParsed(new ANSICode(ERASE_CURSOR_TO_START_OF_LINE, (char) ESC + "[" + num + 'K'));
                } else if (num == 2) {
                    reader.addParsed(new ANSICode(ERASE_CURRENT_LINE, (char) ESC + "[" + num + 'J'));
                } else {
                    printErr("ParseCSI:  ERROR(3): " + StringTool.escape(block));
                }
            } else if (nextChar == 'm') {
                List<Integer> codes = new ArrayList<>();
                codes.add(num);
                code = (char) ESC + "[" + num + "m";
                remainder = remainder.substring(1); // for this, keep the whitespace and assume the author is correct in what they want
                // this will preserve any leading spaces/tabs in text, so especially good for code
                if (debug) {
                    debugPrint("ParseCSI: Codes(1)", code, remainder, codes);
                }
                reader.addParsed(new ANSICode(GRAPHICS_MODE, code, remainder, codes));
            } else if (nextChar == ';') { // separating at least 1 more number
                remainder = SharedCode.trimWhitespaceL(remainder.substring(1));
                String nextNum2 = getNextNumber(remainder);
                //printOut("NextNum2: '" + nextNum2 + "'");
                int num2 = Integer.parseInt(nextNum2);
                remainder = SharedCode.trimWhitespaceL(remainder.substring(nextNum2.length()));
                char terminator = remainder.charAt(0);
                code = (char) ESC + "[" + num + ";" + nextNum2;
                if (terminator == 'H' || terminator == 'f') {
                    code += terminator;
                    reader.addParsed(new ANSICode(MOVE_TO, code, num, num2));
                } else if (terminator == 'm') {
                    List<Integer> codes = new ArrayList<>();
                    codes.add(num);
                    codes.add(num2);
                    code += terminator;
                    remainder = SharedCode.trimWhitespaceL(remainder.substring(1));
                    if (debug) {
                        debugPrint("ParseCSI: Codes(2)", code, remainder, codes);
                    }
                    reader.addParsed(new ANSICode(GRAPHICS_MODE, code, remainder, codes));
                } else if (terminator == ';') {
                    List<Integer> codes = new ArrayList<>();
                    codes.add(num);
                    codes.add(num2);
                    code += terminator;
                    remainder = SharedCode.trimWhitespaceL(remainder.substring(1));
                    boolean done = false;
                    int index = 0;
                    String nextNumLocator3;
                    int nextNum3;
                    while (!done) {
                        nextNumLocator3 = getNextNumber(remainder);
                        nextNum3 = Integer.parseInt(nextNumLocator3);
                        codes.add(nextNum3);
                        remainder = SharedCode.trimWhitespaceL(remainder.substring(nextNumLocator3.length()));
                        terminator = remainder.charAt(0);
                        code += nextNumLocator3 + terminator;
                        if (terminator == 'm') {
                            done = true;
                            // remove the terminator
                            remainder = SharedCode.trimWhitespaceL(remainder.substring(1));
                            // no need to reparse, terminator encountered
                            if (debug) {
                                debugPrint("Parse CSI: Codes(3): ", code, remainder, codes);
                            }

                        } else if (terminator == ';') {
                            // continue parsing numbers
                            remainder = SharedCode.trimWhitespaceL(remainder.substring(1)); // skip the ; and re-parse
                        } else {
                            printErr("ParseCSI:  ERROR(4): num: " + num + ", Code: " + StringTool.escape(block));
                            done = true;
                        }
                    }
                    List<String> lines = StringTool.explode(remainder, '\n');
                    for (int i = 0; i < lines.size(); i++) {
                        reader.addParsed(new ANSICode(GRAPHICS_MODE, code, lines.get(i), codes));
                    }
                }
            }
        }
        text = remainder;
    }

    public void parseDCS(String remainder) {
        code += "P";
        // ESC P
        printErr("ParseDCS: ESC P: " + remainder);
    }

    public void parseOSC(String remainder) {
        code += "]";
        // ESC ]
        printErr("ParseOSC: ESC ]: " + remainder);
    }

    public void parsePrivate(String remainder) {
        code += "?";

        // ESC[?####{l|h}
        printErr("ParsePrivate ESC [?: " + remainder);
    }

    public void parseScreenMode(String remainder) {
        code = (char) ESC + "[=";
        printErr("ParseScreen ESC [=: " + remainder);
        String firstNum = getNextNumber(remainder);
        int num = Integer.parseInt(firstNum);
        remainder = SharedCode.trimWhitespaceL(remainder.substring(firstNum.length()));
        char terminator = remainder.charAt(0);
        code += firstNum + terminator;
        if (terminator == 'h') {
            // set the mode
            reader.addParsed(new ANSICode(SET_SCREEN, code, num));
        } else if (terminator == 'l') {
            // reset the mode
            reader.addParsed(new ANSICode(RESET_SCREEN, code, num));
        } else {
            printErr("parseScreenMode ERROR: num: " + num + " Code: " + StringTool.escape(block));
        }
    }

    public void printBlock() {
        System.out.println(block);
    }

    public void printCode() {
        System.out.println(code);
    }

    public void printErr(String s) {
        System.err.println(s);
    }

    public void printOut(String s) {
        System.out.println(s);
    }

    public String removeEscapeAtZero(String src) {
        return src.substring(1);
    }

    public String toString() {
        return "Block: '" + block + "', Code: '" + code + "', Text: '" + text + "'";
    }
    
}
