package ca.tecreations.text;

import ca.tecreations.text.ansi.ANSILookupResolver;
import ca.tecreations.Color;
import ca.tecreations.TecData;
import ca.tecreations.Font;
import ca.tecreations.ImageTool;
import ca.tecreations.Point;
import ca.tecreations.SystemToken;
import ca.tecreations.TextToken;
import ca.tecreations.interfaces.*;
import ca.tecreations.graphics.DrawObject;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JPanel;
import javax.swing.Timer;

/**
 *
 * @author Tim
 */
public class TextTokenPainter extends DrawObject implements TextPainter {
    private int x = 0;
    private int y = 0;
    
    public int BETWEEN = 2;

    public void setBetween(int pixels) {
        BETWEEN = pixels;
    }

    int width;
    int height;
    
    private TextPoints points = TecData.CODE_POINTS;
    TextToken token;

    Color background = Color.DEFAULT_SWING_BG;
    Color foreground = Color.black;

    public boolean bold = false;
    public boolean dim = false;
    public boolean italic = false;
    boolean underline = false;
    boolean drawMidLine = false;
    boolean doubleUnderline = false;
    boolean blink = false;
    private Timer blinkTimer;
    private int blinkSpeed = 500; // cycle once per second
//    Color oldBackground = background;
//    Color oldForeground = foreground;
    Color blinkBackground = Color.white;
    Color blinkForeground = Color.black;
    private boolean drawBlink = false;
    boolean reverse = false;
    boolean hidden = false;
    boolean strikethrough = false;
    boolean highlight = false;
    Color highlightBackground = Color.yellow;
    Color highlightForeground = Color.black;

    boolean drawBaseline = false;
    
    int startColumn = 0;
    int cursorIndex = 0;
    
    boolean[] selected;
    Color SELECTED_COLOR = new Color(48,74,146);
    
    public TextTokenPainter(TextPoints points, TextToken token,java.awt.Color fg) {
        this(points,token,new Color(fg));
    }
    
    public TextTokenPainter(TextPoints points, TextToken token,Color fg) {
        this.points = points;
        this.token = token;
        this.foreground = fg;
        setForeground(fg);
        int style = points.getFont().getStyle();
        if (style == (Font.BOLD & Font.ITALIC)) {
            bold = true;
            italic = true;
        } else if (style == Font.BOLD) {
            bold = true;
        } else if (style == Font.ITALIC) {
            italic = true;
        }
        getComputedSize();
        //JavaLauncher.tspc.out("SystemTokenPainter.size: " + width + "x" + height);
        selected = new boolean[this.token.length()];
    }

    public void applyCodes(List<Integer> codes) {
        //JavaLauncher.tspc.out("Text: " + getText() + " Codes: " + codes);
        if (codes != null) {
            for (int i = 0; i < codes.size(); i++) {
                int code = codes.get(i);
                if (code == 0) {
                    resetAll();
                } else if (code == 1) {
                    if (codes.size() >= i + 1) {
                        if (codes.get(i + 1) == 34) {
                            i++;
                        } else {
                            setBold(true);
                        }
                    } else {
                        setBold(true);
                    }
                } else if (code == 2) {
                    setDim(true);
                } else if (code == 3) {
                    setItalic(true);
                } else if (code == 4) {
                    setUnderline(true);
  //              } else if (code == 5) {
  //                  setBlink(true);
                } else if (code == 7) {
                    setReverse(true);
                } else if (code == 8) {
                    setHidden(true);
                } else if (code == 9) {
                    setStrikethrough(true);
                } else if (code == 21) {
                    setDoubleUnderline(true);
                } else if (code == 22) {
                    resetBoldAndDim();
                } else if (code == 23) {
                    setItalic(false);
                } else if (code == 24) {
                    setDoubleUnderline(false);
//                } else if (code == 25) {
//                    setBlink(false);
                } else if (code == 27) {
                    setReverse(false);
                } else if (code == 28) {
                    setHidden(false);
                } else if (code == 29) {
                    setStrikethrough(false);
                } else if (code == 30) {
                    setForeground(Color.black);
                } else if (code == 31) {
                    setForeground(Color.red);
                } else if (code == 32) {
                    setForeground(Color.green);
                } else if (code == 33) {
                    setForeground(Color.yellow);
                } else if (code == 34) {
                    setForeground(Color.blue);
                } else if (code == 35) {
                    setForeground(Color.magenta);
                } else if (code == 36) {
                    setForeground(Color.cyan);
                } else if (code == 37) {
                    setForeground(Color.white);
                } else if (code == 39) {
                    setForeground(Color.black);
                } else if (code == 40) {
                    setBackground(Color.black);
                } else if (code == 41) {
                    setBackground(Color.red);
                } else if (code == 42) {
                    setBackground(Color.green);
                } else if (code == 43) {
                    setBackground(Color.yellow);
                } else if (code == 44) {
                    setBackground(Color.blue);
                } else if (code == 45) {
                    setBackground(Color.magenta);
                } else if (code == 46) {
                    setBackground(Color.cyan);
                } else if (code == 47) {
                    setBackground(Color.white);
                } else if (code == 49) {
                    background = new Color(new JPanel().getBackground());
                } else if (code == 90) {
                    setForeground(Color.BRIGHT_BLACK);
                } else if (code == 91) {
                    setForeground(Color.BRIGHT_RED);
                } else if (code == 92) {
                    setForeground(Color.BRIGHT_GREEN);
                } else if (code == 93) {
                    setForeground(Color.BRIGHT_YELLOW);
                } else if (code == 94) {
                    setForeground(Color.BRIGHT_BLUE);
                } else if (code == 95) {
                    setForeground(Color.BRIGHT_MAGENTA);
                } else if (code == 96) {
                    setForeground(Color.BRIGHT_CYAN);
                } else if (code == 97) {
                    setForeground(Color.BRIGHT_WHITE);
                } else if (code == 100) {
                    setBackground(Color.BRIGHT_BLACK);
                } else if (code == 101) {
                    setBackground(Color.BRIGHT_RED);
                } else if (code == 102) {
                    setBackground(Color.BRIGHT_GREEN);
                } else if (code == 103) {
                    setBackground(Color.BRIGHT_YELLOW);
                } else if (code == 104) {
                    setBackground(Color.BRIGHT_BLUE);
                } else if (code == 105) {
                    setBackground(Color.BRIGHT_MAGENTA);
                } else if (code == 106) {
                    setBackground(Color.BRIGHT_CYAN);
                } else if (code == 107) {
                    setBackground(Color.BRIGHT_WHITE);
                } else if (code == 38) {
                    // next should be 5 or 2
                    int nextCode = codes.get(i + 1);
                    if (nextCode == 5) {
                        int lookup = codes.get(i + 2);
                        setForeground(ANSILookupResolver.getColor(lookup));
                        i += 2;
                    } else if (nextCode == 2) {
                        setForeground(new Color(codes.get(i + 2), codes.get(i + 3), codes.get(i + 4)));
                    }
                } else if (code == 48) {
                    int nextCode = codes.get(i + 1);
                    if (nextCode == 5) {
                        int lookup = codes.get(i + 2);
                        setBackground(ANSILookupResolver.getColor(lookup));
                        i += 2;
                    } else if (nextCode == 2) {
                        setBackground(new Color(codes.get(i + 2), codes.get(i + 3), codes.get(i + 4)));
                    }
                }
            }
        }
    }
    
    public void backspace() {
        token.backspace();
    }
    public TextTokenPainter clone() {
        TextTokenPainter clone = new TextTokenPainter(points,new TextToken(),foreground);
        clone.setBackground(background);
        return clone;
    }
    
//    public void cursorHide(Graphics g,int index) {
//        paintCursor(index,background);
//        paintNormal(index);
//    }
    
//    public void cursorShow(Graphics g, int index, Color cursorColor) {
//        paintCursor(index, cursorColor);
//    }
    
    public void cut(int index, int count) {
        token.cut(index,count);
    }
    
    public void deselect() {
        for(int i = 0;i < selected.length;i++) {
            selected[i] = false;
        }
    }
    
    public void draw(Graphics g) {
        paintAt(g,getTX(),getTY());
    }
    
    public void draw(Graphics g, int tx, int ty) {
        paintAt(g,tx,ty);
    }
    
    public void draw(Graphics g, int tx, int ty, Color lineColor) {
        setLineColor(lineColor);
        paintAt(g,tx,ty);
    }
    
    public void draw(Graphics g, int tx, int ty, Color lineColor, Color fillColor) {
        setLineColor(lineColor);
        setFillColor(fillColor);
        paintAt(g,tx,ty);
    }
    
    protected int drawMonospaced(String text, Graphics g, int tx, int ty) {
        int width = 0;
        char ch;
        int capDubWidth = this.points.getWidth((int) 'W');
        //System.out.println("TTP.drawMonospaced: points: " + points.getFontSize() + " Text: " + getText());
        for (int i = 0; i < text.length() - 1; i++) {
            ch = text.charAt(i);
            // get the xOffset to center the char in question
            int xOffset = (capDubWidth - points.getWidth(ch)) / 2;
            if (g != null) {
                Point.drawPoints(g, tx + width + xOffset, ty, points.getPoints(ch));
            } 
            width += capDubWidth + BETWEEN;
        }
        if (text.length() > 0) {
            ch = text.charAt(text.length() - 1);
            int xOffset = (capDubWidth - points.getWidth(ch)) / 2;
            if (g != null) {
                Point.drawPoints(g, tx + width + xOffset, ty, points.getPoints(ch));
            }
            width += capDubWidth;
        }
        return width;

    }

    protected int drawNonMonospaced(String text, Graphics g, int tx, int ty) {
        int width = 0;
        char ch;
        if (text.length() > 0) {
            for (int i = 0; i < text.length() - 1; i++) {
                ch = text.charAt(i);
                if (g != null) {
                    Point.drawPoints(g, tx + width, ty, points.getPoints(ch));
                }
                width += points.getWidth(ch) + BETWEEN;
            }
            ch = text.charAt(text.length() - 1);
            if (g != null) {
                Point.drawPoints(g, tx + width, ty, points.getPoints(ch));
            }
            width += points.getWidth(ch);
        }
        return width;
    }

    public int drawString(String text, Graphics g, int tx, int ty) {
        return drawString(text,g,tx,ty,0);
    }

    public int drawString(String text, Graphics g, int tx, int ty, int between) {
        if (points.isMonospaced()) {
            return drawMonospaced(text, g, tx, ty);
        } else {
            return drawNonMonospaced(text, g, tx, ty);
        }
    }

    public Color getBackground() { 
        return background;
    }
    
    public boolean getBlink() {
        return blink;
    }
    
    public Dimension getComputedSize() {
        width = getTextWidth();
        height = points.getMaxHeight();
        return new Dimension(width, height);
    }

    public int getCursorIndex() { return cursorIndex; }
    
    public int getEndX() { return x + getTextWidth(); }
    
    public int getFontSize() {
        return points.getFontSize();
    }

    public Color getForeground() {
        return foreground;
    }
    
    public int getHeight() {
        return getFontSize();
    }
    
    public boolean getHighlight() {
        return highlight;
    }

    public BufferedImage getImage() {
        BufferedImage image = ImageTool.getNewBufferedImage(width, height);
        Graphics g = image.getGraphics();
        paintAt(g,0,0);
        return image;
    }

    public int getInternalIndex(int x) {
        String text = token.getText();
        int consumed = 0;
        for(int i = 0; i < text.length();i++) {
            int ch = text.charAt(i);
            int min = this.x + consumed - (BETWEEN / 2);
            int max = this.x + consumed + points.getWidth(ch) + (BETWEEN / 2);
            if (min <= x && x <= max) {
                int mid = points.getWidth(ch) / 2;
                int left = min + mid;
                if (x <= left) return i;
                else return i + 1;
            } else consumed += points.getWidth(ch) + BETWEEN;
            System.out.println("Consumed: " + consumed + " min: " + min + " max: " + max + " i: " + i + " x: " + x);
        } 
        return length();
    }
    
    public int getLength() {
        return token.length();
    }
    
    public int getMaxDescent() {
        return points.getMaxDescent();
    }
    
    public int getPaintingWidth() {
        return getTextWidth();
    }
    
    public TextPoints getPoints() {
        return points;
    }
    
    public TextTokenPainter getSpace() {
        TextTokenPainter temp = clone();
        temp.setText(" ");
        return temp;
    }
    
    public String getText() {
        return token.getText();
    }

    public TextToken getTextToken() { return token; }

    public int getTextWidth() {
        int width = 0;
        String text = token.getText();
        int dubWidth = points.getWidth('W');
            
        //System.err.println("Text: '" + text + "'");
        if (points.isMonospaced()) {
            if (text.length() > 0) {
                for(int i = 0; i < text.length() - 1;i++) {
                    width += dubWidth + BETWEEN;
                }
                width += dubWidth;
            }
            return width;
        } else {
            if (text.length() > 0) {
                for(int i = 0;i < text.length() - 1;i++) {
                    width += points.getWidth(text.charAt(i)) + BETWEEN;
                }
                width += points.getWidth(text.charAt(text.length() - 1));
            }
        }
        return width;
    }
    
    public int getWidth(int index) {
        String text = getText();
        if (index < text.length()) {
            char ch = text.charAt(index);
            return points.getWidth(ch);
        } else throw new IllegalArgumentException("TTP.getWidth: index(" + index + ") must be less than length(" + text.length() + ")");
    }
    
    public int getWidthTo(int index) {
        int width = 0;
        String text = token.getText();
        //System.err.println("Text: '" + text + "'");
        if (points.isMonospaced()) {
            int i = 0;
            for(; i < index - 1;i++) {
                width += points.getWidth('W') + BETWEEN;
            }
            if (i < index - 1) width += points.getWidth('W');
            return width;
        }
        int i = 0;
        for (; i < index - 1; i++) {
            width += points.getWidth(text.charAt(i)) + BETWEEN;
        }
        if (i < index - 1) width += points.getWidth(text.charAt(index - 1));
        return width;
    }
    
    public int getWidthAtIndex(int index) {
        char ch = token.getText().charAt(index);
        System.out.println("ch: "+ch);
        return points.getWidth(ch);
    }
    
    public int getWidthForString(String s) {
        String oldText = token.getText();
        token.setText(s);
        int width = getTextWidth();
        token.setText(oldText);
        return width;
    }
    
    public int getX() { return x; }
    
    public boolean hasCol(int start, int col) {
        int max = start + length();
        return (col > start && col < max);
    }
    
    public boolean hasFillPoint(Point p) {
        return false;
    }
    
    public boolean hasPoint(Point p) {
        return false;
    }
    
    public void insert(int index, char c) {
        insert(index,"" + c);   
    }
    
    public void insert(int index, String s) {
        token.insert(index,s);   
    }
    
    public boolean isSystemErr() {
        if (token instanceof SystemToken) {
            if (((SystemToken) token).getStreamType() == TecData.SYS_ERR) {
                return true;
            }
        }
        return false;
    }

    public boolean isSystemOut() {
        if (token instanceof SystemToken) {
            if (((SystemToken) token).getStreamType() == TecData.SYS_OUT) {
                return true;
            }
        }
        return false;
    }
    
    public int length() { return token.length(); }
    
    public void paintAt(Graphics g, int x, int y) {
//        System.out.println("paintAt: bold: " + bold + " italic: " + italic + " Points.font: " + points.toString());
        if (g != null) {
            this.x = x;
            this.y = y;
            Color bg;
            Color fg;
            if (reverse) {
                bg = foreground;
                fg = background;
            } else if (highlight) {
                bg = highlightBackground;
                fg = highlightForeground;
            } else if (blink && drawBlink) {
                bg = blinkBackground;
                fg = blinkForeground;
            } else {
                bg = background;
                fg = foreground;
            }
            int x2 = x;
            String text = token.getText();
            for(int i = 0; i < text.length();i++) {
                int w;
                if (points.isMonospaced()) {
                    w = points.getWidth('W');
                } else {
                    w = points.getWidth(text.charAt(i));
                }
                if (selected.length > 0 && selected[i]) {
                    g.setColor(SELECTED_COLOR);
                } else {
                    g.setColor(bg);
                }
                g.fillRect(x2,y,w + BETWEEN,points.getFontSize());
                x2 += w + BETWEEN;
            }
            
            g.setColor(fg);
            this.x = x;
            if (!hidden) {
                drawString(token.getText(), g, x, y);
            }
            // draw to side of 0, not max0
            if (strikethrough && !hidden) g.drawLine(0, height / 2 - 1, width, height / 2 - 1);
            if (underline && !hidden) g.drawLine(0, height + 1, width, height + 1);
            if (drawMidLine && !hidden) g.drawLine(0,height + 2, width, height + 2);
            if (doubleUnderline & !hidden) g.drawLine(0, height + 3, width, height + 3);
            
            int baseline = points.getBaseline();
            if (drawBaseline) {
                g.setColor(Color.TEC_LIGHT_GREEN);
                g.drawLine(0, baseline, width, baseline);
            }
        }
    }

    public void paintChar(Graphics g, int y, int index, Color color) {
        int indexOffset = 0;
        boolean isMono = points.isMonospaced();
        int dubWidth = points.getWidth('W');
        for(int i = 0; i < index;i++) {
            if (isMono) {
                indexOffset += BETWEEN + dubWidth;
            } else {
                indexOffset += BETWEEN + points.getWidth(getText().charAt(i));
            }
        }
        g.setColor(color);
        Point.drawPoints(g, x + indexOffset, y,points.getPoints(getText().charAt(index)));
    }
    
    public void paintIndex(Graphics g, int index, Color lineColor,Color fillColor) {
        int paintX = 0;
        for(int i = 0; i < index;i++) {
            paintX += points.getWidth(token.getText().charAt(i)) + BETWEEN;
        }
        if (fillColor != null) {
            g.setColor(fillColor);
            g.fillRect(paintX - 1, y, points.getWidth(token.getText().charAt(index)) + 2,height);
        }
        List<Point> pixels = points.getPoints(getText().charAt(index));
        g.setColor(lineColor);
        int indexOffset = 0;
        for(int i = 0; i < index;i++) {
            indexOffset += getWidth(getText().charAt(i)) + BETWEEN;
        }
        Point.drawPoints(g, x + indexOffset, y,pixels);
        
    }
    
    public void paintElement(Graphics g) {
        paintAt(g,getTX(),getTY());
    }
    
    //public void paintNormal(int index) {
//        paintAt(g,x,y);
//    }
    
    public void prefix(String s) {
        token.prefix(s);
    }
    
    public void repaint(Graphics g) {
        paintAt(g,x,y);
    }
    
    public void replace(int index, char c) {
        token.replace(index,"" + c);   
    }
    
    public void replace(int index, String s) {
        token.replace(index,s);   
    }
    
    
    public void resetAll() {
        background = new Color(new JPanel().getBackground());
        foreground = Color.black;
        bold = false;
        dim = false;
        italic = false;
        underline = false;
        doubleUnderline = false;
        blink = false;
        reverse = false;
        hidden = false;
        strikethrough = false;
    }

    public void resetBoldAndDim() {
        setBold(false);
        setDim(false);
    }

    public void select() {
        for(int i = 0; i < selected.length;i++) {
            selected[i] = true;
        }
    }
    
    public void selectFirst(int cols) {
        for(int i = 0;i < selected.length;i++) {
            if (i < cols) selected[i] = true;
            else selected[i] = false;
        }
    }
    
    public void selectFrom(int col) {
        for(int i = 0;i < selected.length;i++) {
            if (i >= col) selected[i] = true;
            else selected[i] = false;
        }
    }

    public void selectInclusive(int low, int high) {
        for(int i = 0; i < selected.length;i++) {
            if (i >= low && i <= high) {
                selected[i] = true;
            } else {
                selected[i] = false;
            }
        }
    }
    
    public void setBackground(java.awt.Color color) {
        background = new Color(color);
    }

    public void setBackground(ca.tecreations.Color color) {
        background = color;
    }
    
    public void setBold(boolean state) {
        bold = state;
        setTextPointsStyle();
    }

    public void setColumnSelected(int index, boolean val) {
        selected[index] = val;
    }
    
    public void setCursorIndex(int index) {
        cursorIndex = index;
    }
    
    public void setDim(boolean state) {
        dim = state;
    }

    @Override
    public void setFillColor(Color fillColor) {
        background = fillColor;
    }
    
    public void setDrawMidLine(boolean state) {
        drawMidLine = state;
    }

    public void setDoubleUnderline(boolean state) {
        underline = state;
        doubleUnderline = state;
    }

    public void setFontSize(int size) {
        points = TextPoints.getInstance(points.getFontName(), points.getFontStyle(), size);
        getComputedSize();
    }

    public void setForeground(java.awt.Color color) {
        foreground = new Color(color);
    }

    public void setHidden(boolean state) {
        hidden = state;
    }

    public TextTokenPainter setHighlight(boolean state) {
        highlight = state;
        return this;
    }

    public TextTokenPainter setHighlightBackground(Color bg) {
        highlightBackground = bg;
        return this;
    }

    public TextTokenPainter setHighlightForeground(Color fg) {
        highlightForeground = fg;
        return this;
    }

    public void setItalic(boolean state) {
        italic = state;
        setTextPointsStyle();
    }

    @Override
    public void setLineColor(Color c) {
        foreground = c;
    }
    
    public void setPoints(TextPoints points) {
        //System.out.println("TTP.setPoints: points: " + points.toString());
        this.points = points;
        //System.out.println("TTP.setPoints: this.points: " + points.toString());
        getComputedSize();
    }
    
    public void setReverse(boolean state) {
        reverse = state;
    }

    public void setStartColumn(int col) {
        startColumn = col;
    }
    
    public void setStrikethrough(boolean state) {
        strikethrough = state;
    }

    public void setText(char ch) {
        token.setText(ch + "");
        getComputedSize();
    }

    public void setText(int i) {
        token.setText("" + i);
        getComputedSize();
    }

    public void setText(String text) {
        token.setText(text);
        getComputedSize();
    }

    public void setTextPoints(TextPoints points) {
        this.points = points;
    }
    
    public void setTextPointsStyle() {
        if (bold && italic) {
            points = TextPoints.getInstance(points.getFontName(), Font.BOLD | Font.ITALIC, points.getFontSize());
        } else if (bold && !italic) {
            points = TextPoints.getInstance(points.getFontName(), Font.BOLD, points.getFontSize());
        } else if (!bold && italic) {
            points = TextPoints.getInstance(points.getFontName(), Font.ITALIC, points.getFontSize());
        } else {
            points = TextPoints.getInstance(points.getFontName(), Font.PLAIN, points.getFontSize());
        }
        width = getTextWidth();
        height = points.getFontSize();
    }

    public void setToken(TextToken token) {
        this.token = token;
    }
    
    public void setUnderline(boolean state) {
        underline = state;
        if (!state) {
            doubleUnderline = false;
        }
    }

    public char[] toCharArray() {
        String text = token.getText();
        char[] array = new char[text.length()];
        for(int i = 0; i < text.length();i++) {
            array[i] = text.charAt(i);
        }
        return array;
    }
    
    public String[] toStringArray() {
        String text = token.getText();
        String[] array = new String[text.length()];
        for(int i = 0; i < text.length();i++) {
            array[i] = "" + text.charAt(i);
        }
        return array;
    }
    
    public void toggleDrawBaseline() {
        drawBaseline = !drawBaseline;
    }

    public void toggleItalic() {
        italic = !italic;
        setItalic(italic);
    }

}
