package ca.tecreations.text;

import ca.tecreations.Color;
import ca.tecreations.File;
import ca.tecreations.Font;
import ca.tecreations.ImageTool;
import ca.tecreations.Point;
import ca.tecreations.Properties;
import ca.tecreations.StringTool;
import ca.tecreations.TypeToType;
import ca.tecreations.apps.launcher.Unicode;
import ca.tecreations.apps.launcher.UnicodeData;
import ca.tecreations.components.Magnifier;
import ca.tecreations.text.TextPoints;

import java.awt.Dimension; 
import java.awt.Rectangle;
import java.awt.font.FontRenderContext;
import java.awt.font.LineMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.awt.image.ImageObserver;
import java.awt.image.PixelGrabber;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 *
 * @author Tim
 */
public class PointsGetter {

    BufferedImage image = new BufferedImage(1, 1, BufferedImage.TYPE_INT_RGB);
    Graphics ig = image.getGraphics();
    Graphics2D g2d = (Graphics2D) ig;
    FontRenderContext context = g2d.getFontRenderContext();
    HashMap<String, Integer> unicodeToIndex = new HashMap<>();
    List<UnicodeData> data = new ArrayList<>();
    HashMap<String, Properties> unicodePropertiesMap = new HashMap<>();

    Font font;
    TextPoints textPoints;
    List<List<Point>> points = new ArrayList<>();
    Properties unicodeProps;

    boolean debug = false;
    boolean done = false;

    boolean displayFirstTest = false;
    Magnifier magnifier;

    static List<Object> elements = new ArrayList<>();
    
    public PointsGetter(Font font) {
        this.font = font;
        textPoints = TextPoints.getInstance(font, true);
    }

    /**
     * Gets a set of points for a given text representation. It accepts both
     * Unicode and ASCII characters, however Unicode must be a 4 to 6 digit
     * hexadecimal number in the range 0000 to 10FFFF with a preceding 'x', so
     * for example, "x10FFF" : z == '1', a == '0', b == 'F', c == 'F' and d ==
     * 'F'.
     *
     * z will only be accepted when present.
     *
     * Use getPoints() to get the List<List<Point>>.
     *
     * @param font the font to use
     * @param list the list of blocks, 1 block per
     * @param monospaced whether to monospace the ASCII textPoints for width
     * calcs
     */
    public PointsGetter(Font font, List<String> list, boolean monospaced) {
        this.font = font;
        textPoints = TextPoints.getInstance(font, monospaced);
        process(list);
    }

    public void addPoints(List<Point> points) {
        this.points.add(points);
    }

    public void doProcess(String target) {
//        System.out.println("doProcess: Target: " + target);
        File fontFile = font.getFile();
        String z = "";
        String codePoint = "";
        String codeIndex = "";
        // determine if Unicode or ASCII -- if unicode, only one glyph at a time
        String s = target.substring(1);
        if (target.startsWith("x") && s.length() >= 4 && s.length() <= 6
                && isHex(s)) {
            z = getZ(s);
            codePoint = getCodePoint(s);
            codeIndex = getCodeIndex(s);
            System.out.println("doProcess: Unicode: " + s);
            UnicodeData data = getUnicodeData(z, codePoint, codeIndex);
            this.points.add(data.points);
            elements.add(data);
        } else {
            System.out.println("doProcess: ASCII: " + target);
            int x = 0;
            List<Point> aggregated = new ArrayList<>();
            for (int i = 0; i < target.length(); i++) {
                List<Point> got = getPoints_ASCII(fontFile, target.charAt(i));
                if (got != null) {
                    for (int j = 0; j < got.size(); j++) {
                        Point p = got.get(j);
                        aggregated.add(new Point(x + p.x, p.y));
                    }
                    x += Point.getWidth(got);
                }
            }
            this.points.add(aggregated);
            elements.add(aggregated);
        }
    }

    public static void drawPoints(List<Point> points, Image image, Color c, int y) {
        Graphics g = image.getGraphics();
        g.setColor(c);
        Point p;
        for (int i = 0; i < points.size(); i++) {
            p = points.get(i);
            g.drawLine(p.x, y + p.y, p.x, y + p.y);
        }
    }

    public List<Point> generatePoints(String s) {
        System.out.println("generatePoints: " + s);
        List<Point> points = new ArrayList<>();
        int[] pix = null;

        // start with the size that Java reports
        Dimension size = getTextSize(s);
        if (size.width == 0) {
            size.width = 32;
        }

        // get the image of the String and display it
        BufferedImage image = getStringImage(textPoints, s);
        if (displayFirstTest) {
            magnifier = new Magnifier(image);
            magnifier.setVisible(true);
        }

        // and determine which pixels were actually painted
        pix = new int[size.width * size.height];
        PixelGrabber pixelGrabber = new PixelGrabber(image, 0, 0, size.width, size.height, pix, 0, size.width);
        try {
            pixelGrabber.grabPixels();
        } catch (InterruptedException e) {
            System.err.println("interrupted waiting for pixels!");
            return null;
        }
        if ((pixelGrabber.getStatus() & ImageObserver.ABORT) != 0) {
            System.err.println("image fetch errored or aborted");
            return null;
        }
        for (int j = 0; j < size.height; j++) {
            for (int i = 0; i < size.width; i++) {
                if (isBlack(i, j, pix[j * size.width + i])) {
                    // okay, so we actually want to modify this and drawString to
                    // put the string at 25% of the width, so we can compute negative
                    // x values of the glyph
                    points.add(new Point(i, j));
                }
            }
        }
        return points;
    }

    public List<Point> generatePoints_Unicode(String zHex, String codePoint, String codeIndex) {
        String hex = "";
        if (!zHex.equals("") && !zHex.equals("0")) {
            hex += zHex;
        }
        hex += codePoint + codeIndex;
        System.out.println("generatePoints_Unicode: unicodeEscape: " + hex);
        String unicodeString = Unicode.toString(hex);
        System.out.println("generatePoints_Unicode: unicodeString: " + unicodeString);

        List<Point> points = generatePoints(unicodeString);
        int index = data.size();
        unicodeToIndex.put(hex, index);
        data.add(new UnicodeData(hex, points, getTextSize(unicodeString)));
        return points;
    }

    public int getAscent(String s) {
        return (int) Math.abs(getLineMetrics(s).getAscent());
    }

    public int getBaseline(String s) {
        return font.getSize() - getMaxDescent(s);
    }

    public String getCodeIndex(String s) {
        if (s.length() == 6) {
            return "" + s.charAt(4) + s.charAt(5);
        } else if (s.length() == 5) {
            return "" + s.charAt(3) + s.charAt(4);
        } else if (s.length() == 4) {
            return "" + s.charAt(2) + s.charAt(3);
        } else {
            throw new IllegalArgumentException("getCodePointFull: illegal size: '" + s + "'");
        }
    }

    public String getCodePoint(String s) {
        if (s.length() == 6) {
            return "" + s.charAt(2) + s.charAt(3);
        } else if (s.length() == 5) {
            return "" + s.charAt(1) + s.charAt(2);
        } else if (s.length() == 4) {
            return "" + s.charAt(0) + s.charAt(1);
        } else {
            throw new IllegalArgumentException("getCodePointFull: illegal size: '" + s + "'");
        }
    }

    public int getFontSize() {
        return font.getSize();
    }

    public LineMetrics getLineMetrics(String s) {
        return font.getJavaFont().getLineMetrics(s, context);
    }

    public int getMaxDescent(String s) {
        return (int) Math.abs(getLineMetrics(s).getDescent());
    }

    public List<List<Point>> getPoints() {
        return points;
    }

    public List<Point> getPoints_ASCII(File file, char ch) {
        return textPoints.getPoints(ch);
    }

    public BufferedImage getMainTestImage(List<List<Point>> points) {
        int w = 0;
        int h = 0;
        for (int i = 0; i < points.size(); i++) {
            w = Math.max(w, Point.getWidth(points.get(i)));
            h += Point.getHeight(points.get(i));
        }

        BufferedImage image = ImageTool.getNewBufferedImage(w, h);
        Graphics ig = image.getGraphics();
        ig.setColor(Color.WHITE);
        ig.fillRect(0, 0, w, h);
        ig.setColor(Color.black);
        List<Point> list;
        Point p;
        int y = 0;
        for (int i = 0; i < points.size(); i++) {
            list = points.get(i);
            for (int j = 0; j < list.size(); j++) {
                p = list.get(j);
                ig.drawLine(p.x, y + p.y, p.x, y + p.y);
            }
            y += Point.getHeight(list);
        }
        return image;
    }

    public BufferedImage getStringImage(TextPoints tp, String s) {
        Dimension size = getTextSize(s);

        System.out.println("getStringImage: " + size);
        image = ImageTool.getNewBufferedImage(size.width, size.height);
        ig = image.getGraphics();

        ImageTool.fill(image, Color.white);

        LineMetrics lm = getLineMetrics(s);
        int y = (int) (size.height - lm.getDescent());

        ig.setColor(Color.black);
        ig.setFont(font.getJavaFont());
        ig.drawString(s, 0, y); // if the user needs kerning along the X axis,
        // do that in the paint methodology, so for example, you could have a 
        // List<Glyph> tied to a List<Kerning>, draw whichever way they want, up,
        // down, leftward, rightward. Just Let them specifiy for what we support
        // ideally, the user would have a stroke with a variable-enabled brush
        return image;
    }

    public Dimension getTextSize(String s) {
        Rectangle bounds = font.getJavaFont().getStringBounds(s, context).getBounds();
        return new Dimension((int) bounds.getWidth(), (int) bounds.getHeight());
    }

    public File getUnicodeFile(String unicode) {
        String z = getZ(unicode);
        String codePoint = getCodePoint(unicode);
        File unicodeFile = font.getUnicodeFilePath(z, codePoint);
        return unicodeFile;
    }

    public UnicodeData getUnicodeData(String zHex, String codePoint, String codeIndex) {
        List<Point> points = new ArrayList<>(); // default to no data
        File unicodeFile = getUnicodeFile(zHex + codePoint + codeIndex);
        System.out.println("UnicodeFile: " + unicodeFile.getAbsolutePath());
        String unicode = zHex + codePoint + codeIndex;
        if (unicodeToIndex.containsKey(unicode)) {
            return data.get(unicodeToIndex.get(unicode));
        } else {
            if (unicodeFile.exists()) {
                // first determine if stored, if so, get that,
                if (unicodePropertiesMap.containsKey(zHex + codePoint)) {
                    unicodeProps = unicodePropertiesMap.get(zHex + codePoint);
                } else {
                    unicodeProps = new Properties(unicodeFile.getAbsolutePath());
                }
            } else {
                unicodeProps = new Properties(unicodeFile.getAbsolutePath());
            }
            String stored = unicodeProps.get(unicode);
            if (stored != null) {
                return UnicodeData.fromStored(stored);
            } else {
                // generate and save
                points = generatePoints_Unicode(zHex, codePoint, codeIndex);
                UnicodeData unicodeData = new UnicodeData(unicode, points, getTextSize(Unicode.toString(unicode)));
                unicodeData.write(unicodeProps);
                data.add(unicodeData);
                // and update our in-memory copy for when we want to reuse it
                unicodePropertiesMap.put(zHex + codePoint, unicodeProps);
                return unicodeData;
            }
        }
    }

    public int getUnicodeWidth(String unicode) { // hex, but no preceding x
        return data.get(unicodeToIndex.get(unicode)).size.width;
    }

    public String getZ(String unicode) {
        String z = "0";
        String remainder = unicode;
        if (unicode.startsWith("10") && unicode.length() == 6) {
            z = "10";
        } else if (unicode.length() == 5) {
            z = "" + unicode.charAt(0);
        }
        return z;
    }

    public boolean isBlack(int x, int y, int pixel) {
        int alpha = (pixel >> 24) & 0xff;
        int maxX = 0;
        int red = (pixel >> 16) & 0xff;
        int green = (pixel >> 8) & 0xff;
        int blue = (pixel) & 0xff;
        if (red == 0 && green == 0 && blue == 0) {
            return true;
        }
        return false;
    }

    public boolean isDone() {
        return done;
    }

    public boolean isHex(String s) {
        if (s.startsWith("10") && s.length() == 6) {
            return isHex(s.substring(2));
        } else if (s.length() == 5) {
            return isHex(s.charAt(0)) && isHex(s.substring(1));
        } else {
            for (int i = 0; i < s.length(); i++) {
                if (!isHex(s.charAt(i))) {
                    return false;
                }
            }
            return true;
        }
    }

    public boolean isHex(char c) {
        if (c >= '0' && c <= '9') {
            return true;
        }
        if (c == 'A' || c == 'a') {
            return true;
        }
        if (c == 'B' || c == 'b') {
            return true;
        }
        if (c == 'C' || c == 'c') {
            return true;
        }
        if (c == 'D' || c == 'd') {
            return true;
        }
        if (c == 'E' || c == 'e') {
            return true;
        }
        if (c == 'F' || c == 'f') {
            return true;
        }
        return false;
    }

    public static void main(String[] args) {
        String[] fontNames = Font.getNames();
        for(int i = 0; i < fontNames.length;i++) {
            System.out.println(i + ": " + fontNames[i]);
        }
        //System.exit(0);
        
        PointsGetter getter = new PointsGetter(new Font("Dialog",  18,"bold"));
        Magnifier magnifier = new Magnifier();

        List<String> glyphs = new ArrayList<>();
        glyphs.add("(AAA)");
        glyphs.add("A+_+A");
        glyphs.add("x10FFFF");
        glyphs.add("x20206");
        glyphs.add("x00206");
        glyphs.add("x02653");

        getter.process(glyphs);

        List<List<Point>> points = getter.getPoints();
        int w = 0;
        int h = 0;
        BufferedImage image;

        for (int i = 0; i < points.size(); i++) {
            List<Point> pts = points.get(i);
            w = Math.max(w, Point.getWidth(pts));
            if (elements.get(i) instanceof UnicodeData) {
                h += ((UnicodeData)elements.get(i)).size.height;
            } else {
                h += getter.getFontSize();
            }
        }
        image = ImageTool.getNewBufferedImage(w, h);
        ImageTool.fill(image, Color.YELLOW);
        int y = 0;
        for (int i = 0; i < elements.size(); i++) {
            if (elements.get(i) instanceof UnicodeData) {
                UnicodeData data = (UnicodeData)elements.get(i);
                drawPoints(data.points,image,Color.BLUE,y);
                y += data.size.height;
            } else {
                List<Point> pts = points.get(i);
                drawPoints(pts, image, Color.BLUE, y);
                y += getter.getFontSize();
            }
        }

        magnifier.setImage(image);
        magnifier.setVisible(true);

        //Magnifier m2 = new Magnifier(getter.getMainTestImage(getter.getPoints()));
        //m2.setVisible(true);

    }

    /**
     * Processes all of the elements within targets. Targets must consist of
     * either hex unicode components, z, codePoint and codeIndex. Size must be 5
     * to 7 and preceded by an 'x' if unicode, otherwise, ASCII is assumed. @See
     * ca.tecreations.TextPoints, ca.tecreations.*Token, ca.tecreations.Font,
     * the (xyz)Painter classes and GUITextTokenPainter
     */
    public void process(List<String> targets) {
        for (int i = 0; i < targets.size(); i++) {
            doProcess(targets.get(i));
        }
    }

    @Override
    public String toString() {
        return "PointsGetter[Font: " + font.getName() + "," + font.getStyleCode() + ", size: " + font.getSize() + "]";
    }
}
