package ca.tecreations.text;

import ca.tecreations.Color;
import ca.tecreations.Point;
import ca.tecreations.StringTool;
import ca.tecreations.TecData;
import ca.tecreations.TextToken;

import java.awt.Graphics;
import java.util.ArrayList;
import java.util.List;

import ca.tecreations.graphics.DrawObject;
import ca.tecreations.interfaces.TextPainter;
/**
 *
 * @author Tim
 */
public class LineOfTextTokenPainter extends DrawObject implements TextPainter {
    List<TextTokenPainter> painters = new ArrayList<>();
    public int BETWEEN = 2;
    public TextPoints points = TecData.ARIAL_B_12;
    
    private int x;
    private int y;
    
    public Color DEMARCATION_COLOR = null;
    
    public int paintedWidth = 0;
    
    
    public LineOfTextTokenPainter(TextPoints points) {
        this.points = points;
    }
    
    public LineOfTextTokenPainter(TextPoints points, Color bg, Color fg) {
        this(points);
        TextTokenPainter painter = new TextTokenPainter(points,new TextToken(),fg);
        painter.setBackground(bg);
        painters.add(painter);
    }
    
    public LineOfTextTokenPainter(List<TextTokenPainter> painters) {
        this.painters = painters;
        this.points = painters.get(0).getPoints();
    }
    
    public LineOfTextTokenPainter(List<TextTokenPainter> painters, TextPoints points) {
        this(points);
        this.painters = painters;
    }
    
    public void addPainter(String text) {
        painters.add(new TextTokenPainter(points,new TextToken(text),TecData.DEFAULT_COLOR));
    }
    
    public void addPainter(TextTokenPainter painter) {
        painters.add(painter);
    }
    
    public void backspace() {
        
    }
    
    public char charAt(int index) {
        int ch = 0;
        String text = getText();
        return text.charAt(index);
    }
    
    public int charAt_Width(int colNum) {
        return charWidthAt(colNum);
    }
    
    public int charWidthAt(int colNum) {
        if (colNum > length()) throw new IllegalArgumentException("column number: " + colNum + " : cannot be past length of text: " + getText());
        return points.getWidth(getText().charAt(colNum));
    }
    
    public void deselect() {
        for(int i = 0; i < painters.size();i++) {
            painters.get(i).deselect();
        }
    }
    
    public void draw(Graphics g) {
        draw(g,getTX(),getTY());
    }
    
    public void draw(Graphics g, int x, int y) {
        Color oldColor = new Color(g.getColor());
        this.x = x;
        this.y = y;
        TextTokenPainter painter = null;
        for(int i = 0;i < painters.size();i++) {
            painter = painters.get(i);
            painter.paintAt(g,x,y);
            x += painter.getTextWidth() + BETWEEN;
        }
        paintedWidth = x - BETWEEN;
        g.setColor(oldColor);
    }
    
    public void draw(Graphics g, int tx, int ty, Color lineColor) {
        setLineColor(lineColor);
        draw(g,tx,ty);
    }
    
    public void draw(Graphics g, int tx, int ty, Color lineColor,Color fillColor) {
        setLineColor(lineColor);
        setFillColor(fillColor);
        draw(g,tx,ty);
    }
    
    public int getColumn(int mouseX) {
        String text = getText();
        boolean isMono = points.isMonospaced();
        int dubWidth = points.getWidth('W');
        int x = 0;
        int lowX = 0;
        int highX = 0;
        int index = 0;
        while (x < mouseX && index < text.length()) {
            lowX = x;
            if (isMono) {
                x += BETWEEN + dubWidth;
                highX = x;
            } else {
                x += BETWEEN + points.getWidth(text.charAt(index));
                highX = x;
            }
            index++;
        }
        int left = mouseX - lowX;
        int right = highX - mouseX;
        if (left < right) return index - 1;
        else return index;
    }
    
    public int getColumnX(int colNum) {
        // This will not be 0, colNum should have already been validated to be positive.
        if (colNum == 0) return 0;
        if (colNum >= getText().length()){
            return getTextWidth();
        }
        int x = 0;
        String text = getText();
        boolean isMono = points.isMonospaced();
        int dubWidth = points.getWidth('W');
        int index = 0;
        while (index < colNum && index < text.length()) {
            if (isMono) {
                x += BETWEEN + dubWidth;
            } else {
                x += BETWEEN + points.getWidth(text.charAt(index));
            }
            index++;
        }
        return x;
    }
    
    public int getHeight() {
        if (painters.size() > 0) {
            return painters.get(0).getHeight();
        }
        return -1;
    }
    
    public int getLength() {
        return length();
    }
    
    public int getMaxColumn() { return length(); }
    
    public int getPaintedWidth() { return paintedWidth; }
    
    public TextTokenPainter getPainter(int col) {
        int start = 0;
        TextTokenPainter painter;
        for(int i = 0; i < painters.size();i++) {
            painter = painters.get(i);
            if (painter.length() >= col - start) {
                return painter;
            }
        }
        return null;
    }
    
    // gets the left hand edge of where the cursor should be at this column number
    public int getX_Low(int column) {
        int lowX = 0;
        
        // process column 0
        if (column == 0) return BETWEEN;
        String text = getText();
        if (column >= text.length()) {
            return BETWEEN + getTextWidth();
        }

        // process monospaced, if so
        boolean isMono = points.isMonospaced();
        if (isMono) {
            int dubWidth = points.getWidth('W');
            int most = ((column - 1) * dubWidth) + ((column - 1) * BETWEEN);
            return most + dubWidth;
        } else {
            for(int i = 0; i < column - 1;i++) {
                lowX += points.getWidth(text.charAt(i)) + BETWEEN;
            } 
            lowX += points.getWidth(text.charAt(column - 1));
        }
        return lowX;
    }

    // gets the right hand edge of where the cursor should be for the column number
    // -- equivalent to get low X for the next column, but should not throw
    //    an index out of bounds exception?
    public int getX_High(int column) {
        int highX = 2;
        if (column == 0) return highX;
        
        String text = getText();
        if (text.length() < column) {
            return highX + getTextWidth();
        }
        int dubWidth = points.getWidth('W');
        boolean isMono = points.isMonospaced();
        for(int i = 0; i < column - 1;i++) {
            if (isMono) {
                highX += BETWEEN + dubWidth;
            } else {
                highX += BETWEEN + points.getWidth(text.charAt(i));
            }
        }
        highX += points.getWidth(text.charAt(column - 1));
        return highX;
    }
    
    public int getXToColumn(int numsWidth, int column) {
        int x = numsWidth + 2;
        String text = getText();
        int dubWidth = points.getWidth('W');
        boolean isMono = points.isMonospaced();
        for(int i = 0;i < column;i++) {
            if (isMono) {
                x += BETWEEN + dubWidth;
            } else {
                x += BETWEEN + points.getWidth(text.charAt(i));
            }
        }
        return x;
    }
    
    // do a test
    public static void main(String[] args) {
        LineOfTextTokenPainter line = new LineOfTextTokenPainter(TecData.CODE_POINTS);
        
        line.addPainter("This");
        line.addPainter(" ");
        line.addPainter("is");
        line.addPainter(" ");
        line.addPainter("a");
        line.addPainter(" ");
        line.addPainter("test.");
        System.out.println("Cols: " + line.getLength() + " Text: '" + line.getText() + "' Col 8 Letter: " + line.charAt(8) + " Col 8 X: " + line.getX_Low(8) + " Col 9 X: " + line.getX_High(8));
    }
    
    public Color getDemarcationColor() { return DEMARCATION_COLOR; }
    
    public int getFontSize() {
        return points.getFontSize();
    }
    
    public int getMaxDescent() {
        return points.getMaxDescent();
    }
    
    public int getPaintingWidth() {
        int width = BETWEEN;
        if (painters.size() > 0) {
            for(int i = 0; i < painters.size() - 1;i++) {
                width += painters.get(i).getPaintingWidth() + BETWEEN;
            }
            width += painters.get(painters.size() - 1).getPaintingWidth() + BETWEEN;
        }
        return width;
    }
    
    public String getText() {
        String s = "";
        if (painters.size() > 0) {
            for(int i = 0;i < painters.size() - 1;i++) {
                s += painters.get(i).getText();
            }
            s += painters.get(painters.size() - 1).getText();
        }
        return s;
    }
    
    public int getTextWidth() {
        return getPaintingWidth();
    }
    
    public boolean hasFillPoint(Point p) {
        return false;
    }
    
    public boolean hasPoint(Point p) {
        return false;
    }
    
    public int length() {
        int count = 0;
        for(int i = 0; i < painters.size();i++) count += painters.get(i).length();
        return count;
    }
    
    public void paintAt(Graphics g, int x, int y) {
        draw(g,x,y);
    }
    
    public void paintChar(Graphics g, int y, int colNum, Color color) {
        if (colNum > length()) {
            throw new IllegalArgumentException("LofTTP.paintChar: colNum(" + colNum + ") must be less than length(" + length() + ")");
        }
        int col = 0;
        TextTokenPainter painter;
        for(int i = 0; i < painters.size();i++) {
            painter = painters.get(i);
            if (col + painter.length() > colNum) { 
                painter.paintChar(g,y,colNum - col,color);
                return;
            } else {
                col += painter.length();
            }
        }
    }
    
    public void paintCol(Graphics g, int colNum, Color lineColor, Color fillColor) {
        int col = 0;
        for(int i = 0; i < painters.size();i++) {
            if (col + painters.get(i).length() > colNum) {
                painters.get(i).paintIndex(g,colNum - col,lineColor,fillColor);
            }
            col += painters.get(i).length();
        }
    }
    
    public void paintElement(Graphics g) {
        draw(g,getTX(),getTY());
    }
    
    public void select() {
        for(int i = 0; i < painters.size();i++) {
            painters.get(i).select();
        }
    }
    
    public void selectFrom(int col) {
        int idx = 0;
        int painterIndex = 0;
        int start = 0;
        TextTokenPainter painter;
        for(int i = 0;i < painters.size();i++) {
            painter = painters.get(painterIndex++);
            for(int j = 0; j < painter.length();j++) {
                if (idx >= col) {
                    painter.setColumnSelected(j,true);
                } else {
                    painter.setColumnSelected(j,false);
                }
                idx++;
            }
        }
    }
    
    public void selectInclusive(int low, int high) {
        int start = 0;
        int col = 0;
        TextTokenPainter painter;
        for(int i = 0; i < painters.size();i++) {
            painter = painters.get(i);
            for(int j = 0; j < painter.length();j++) {
                if (start + j >= low && start + j <= high) {
                    painter.setColumnSelected(j,true);
                } else {
                    painter.setColumnSelected(j,false);
                }
                col++;
            }
            start += painter.length();
            
        }
    }
    
    public void selectTo(int col) {
        int index = 0;
        int lastIndex = 0;
        TextTokenPainter painter = null;
        for(int i = 0;i < painters.size();i++) {
            painter = painters.get(i);
            if (index + painter.length() < col) {
                painter.select();
                index += painter.length();
                lastIndex++;
            } else {
                i = painters.size();
            }
        }
        int cols = col - index;
        if (lastIndex < painters.size()) {
            painter = painters.get(lastIndex++);
            painter.selectFirst(cols);
        }
        for(int i = lastIndex;i < painters.size();i++) {
            painters.get(i).deselect();
        }
    }
    
    public void setBackground(Color c) {
        for(int i = 0; i < painters.size();i++) {
            painters.get(i).setBackground(c);
        }
    }

    public void setBackground(java.awt.Color c) {
        for(int i = 0; i < painters.size();i++) {
            painters.get(i).setBackground(c);
        }
    }
    
    public void setDemarcationColor(java.awt.Color c) {
        DEMARCATION_COLOR = new Color(c);
    }
    
    public void setDemarcationColor(Color c) {
        DEMARCATION_COLOR = new Color(c);
    }
    
    public void setDrawMidLine(boolean state) {
        for(int i = 0; i < painters.size();i++) {
            painters.get(i).setDrawMidLine(true);
        }
    }
    
    public void setDoubleUnderline(boolean state) {
        for(int i = 0; i < painters.size();i++) {
            painters.get(i).setDoubleUnderline(true);
        }
    }
    
    public void setFillColor(Color c) {
        for(int i = 0; i < painters.size();i++) {
            painters.get(i).setFillColor(c);
        }
    }
    
    public void setForeground(Color c) {
        for(int i = 0; i < painters.size();i++) {
            painters.get(i).setForeground(c);
        }
    }

    public void setForeground(java.awt.Color c) {
        for(int i = 0; i < painters.size();i++) {
            painters.get(i).setForeground(c);
        }
    }
    
    public void setLineColor(Color c) {
        for(int i = 0; i < painters.size();i++) {
            painters.get(i).setLineColor(c);
        }
    }
    
    public void setPoints(TextPoints points) {
        this.points = points;
        for(int i = 0; i < painters.size();i++) {
            painters.get(i).setPoints(points);
        }
    }
    
    public void setText(String s) {
        List<String> parts = StringTool.explode(s,' ');
        List<TextTokenPainter> list = new ArrayList<>();
        for(int i = 0; i < parts.size();i++) {
            list.add(new TextTokenPainter(points,new TextToken(parts.get(i)),Color.black));
        }
        painters = list;
    }
    
    public char[] toCharArray() {
        int count = 0;
        for(int i = 0; i < painters.size();i++) count += painters.get(i).length();
        char[] array = new char[count];
        int index = 0;
        TextTokenPainter painter;
        for(int i = 0; i < painters.size();i++) {
            painter = painters.get(i);
            char[] charArray = painter.toCharArray();
            for(int j = 0; j < charArray.length;j++) {
                array[index++] = charArray[j];
            }
        }
        return array;
    }
    
    public String[] toStringArray() {
        int count = 0;
        for(int i = 0; i < painters.size();i++) count += painters.get(i).length();
        String[] array = new String[count];
        int index = 0;
        TextTokenPainter painter;
        for(int i = 0; i < painters.size();i++) {
            painter = painters.get(i);
            String[] stringArray = painter.toStringArray();
            for(int j = 0; j < stringArray.length;j++) {
                array[index++] = stringArray[j];
            }
        }
        return array;
    }
}
