package ca.tecreations.net.bc;

import ca.tecreations.File;
import ca.tecreations.net.ExceptionHandler;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.math.BigInteger;
import java.security.Security;
import java.util.Date;
import java.util.Enumeration;

import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.X500NameBuilder;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x509.BasicConstraints;
import org.bouncycastle.asn1.x509.ExtendedKeyUsage;
import org.bouncycastle.asn1.x509.KeyUsage;
import org.bouncycastle.cert.X509v1CertificateBuilder;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509v1CertificateBuilder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.jcajce.JcaX509CertificateHolder;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jsse.provider.BouncyCastleJsseProvider;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;

public class SecurityTool {
    public static SecurityTool instance = new SecurityTool();

    // Define our constants to reduce errors and access within IDE's
    public static final String BC = "BC";
    public static final String BCJSSE = "BCJSSE";
    public static final String JKS = "JKS";
    public static final String PKCS12 = "PKCS12";
    public static final String PKIX = "PKIX";
    public static final String TLS = "TLS";
    
    private static long serialNumberBase = System.currentTimeMillis();

    public SecurityTool() {
        Security.addProvider(new BouncyCastleProvider());
        Security.addProvider(new BouncyCastleJsseProvider());
    }
    
    /**
     * Calculate a date in seconds (suitable for the PKIX profile - RFC 5280)
     *
     * @param hoursInFuture hours ahead of now, may be negative.
     * @return a Date set to now + (hoursInFuture * 60 * 60) seconds
     */
    // Chapter 8: X.509 Certificates and Attribute Certificates 206
    public static Date calculateDate(int hoursInFuture) {
        long secs = System.currentTimeMillis() / 1000;

        return new Date((secs + (hoursInFuture * 60 * 60)) * 1000);
    }
    
    /**
     * Calculate a serial number using a monotonically increasing value.
     *
     * @return a BigInteger representing the next serial number in the sequence.
    */
    // Chapter 8: X.509 Certificates and Attribute Certificates 206
    public static synchronized BigInteger calculateSerialNumber() {
        return BigInteger.valueOf(serialNumberBase++);
    }
    
    public JcaX509CertificateHolder certToJcaX509Holder(X509Certificate cert) {
        JcaX509CertificateHolder certHolder = null;
        try {
            certHolder = new JcaX509CertificateHolder(cert);
        } catch (Exception e) {
            ExceptionHandler.handle("certToJcaX509Holder","couldn't encode", e);
        }
        return certHolder;
    }
    
    public X509CertificateHolder certToX509Holder(X509Certificate cert) {
        X509CertificateHolder certHolder = null;
        try {
            certHolder = new JcaX509CertificateHolder(cert);
        } catch (Exception e) {
            ExceptionHandler.handle("certToX509Holder","couldn't encode",e);
        }
        return certHolder;
    }
    
    /**
     * Simple method to convert an X509CertificateHolder to an X509Certificate
     * using the java.security.cert.CertificateFactory class.
     */
    // Chapter 8: X.509 Certificates and Attribute Certificates 206
    public static X509Certificate convertX509CertificateHolder(
    X509CertificateHolder certHolder)
    throws Exception
    {
        CertificateFactory cFact = CertificateFactory.getInstance("X.509", "BC");

        return (X509Certificate)cFact.generateCertificate(
            new ByteArrayInputStream(
                certHolder.getEncoded()));
    }
    
    /**
     * Build a sample self-signed V1 certificate to use as a trust anchor, or
     * root certificate.
     *
     * @param keyPair the key pair to use for signing and providing the
     * public key.
     * @param sigAlg the signature algorithm to sign the certificate with.
     * @return an X509CertificateHolder containing the V1 certificate.
     */
    // Chapter 8: X.509 Certificates and Attribute Certificates 206
    public static X509CertificateHolder createBCTrustAnchor(KeyPair keyPair, String sigAlg)
    throws Exception
    {
        X500NameBuilder x500NameBld = new X500NameBuilder(BCStyle.INSTANCE)
            .addRDN(BCStyle.C, "AU")
            .addRDN(BCStyle.ST, "Victoria")
            .addRDN(BCStyle.L, "Melbourne")
            .addRDN(BCStyle.O, "The Legion of the Bouncy Castle")
            .addRDN(BCStyle.CN, "Demo Root Certificate");

        X500Name name = x500NameBld.build();

        X509v1CertificateBuilder certBldr = new JcaX509v1CertificateBuilder(
            name,
            calculateSerialNumber(),
            calculateDate(0),
            calculateDate(24 * 31),
            name,
            keyPair.getPublic());
        ContentSigner signer = new JcaContentSignerBuilder(sigAlg)
            .setProvider("BC").build(keyPair.getPrivate());

         return certBldr.build(signer);
    }

    /**
     * Create a KeyStore containing a single key with a self-signed certificate.
     *
     * @param type Keystore type
     * @param alias Key owner
     * @param keyPass Key pass
     * @return a KeyStore containing a single key with a self-signed
     * certificate.
     * @see createKeyStroeFor2, createTrustStoreFrom
     */
    public static KeyStore createKeyStore(String type,String alias, char[] keyPass) {
        PrivateCredential cred = null;
        try {
            cred = createSelfSignedCredentials();
        } catch (Exception e) {
            ExceptionHandler.handle("createKeyStore: " + type + ": " + alias,"createSelfSignedCertificate",e);
            return null;
        }
        KeyStore store = null;
        try {
            store = KeyStore.getInstance(type);
        } catch (Exception e) {
            ExceptionHandler.handle("createKeyStore " + type + ": " + alias,"getting instance",e);
            return null;
        }  
        try {
            store.load(null, null);
        } catch (Exception e) {
            if (e instanceof FileNotFoundException || e instanceof IOException) {
                ExceptionHandler.handleIO("createKeyStore " + type + ": " + alias,"loading keystore",e);
            } else {
                ExceptionHandler.handleIO("createKeyStoreFor: " + alias,"security", e);
            }
            return null;
        }
        try {
            store.setKeyEntry(alias, cred.getPrivateKey(), keyPass,
                new Certificate[]{cred.getCertificate()});
        } catch (Exception e) {
            ExceptionHandler.handleIO("createKeyStore " + type + ": " + alias,"security", e);
        }
        return store;
    }

    /** 
     * Creates a keystore with a key and self-signed cert.
     * 
     */
    public static KeyStore createKeyStore2(String type,String alias, char[] keyPass,String path,char[] storePass) {
        KeyStore ks = createKeyStore(type,alias,keyPass);
        FileOutputStream out = null;
        try {
            File f = new File(path);
            File deepest = f.getDeepestDirectory();
            deepest.mkdirs();
            out = new FileOutputStream(f.getJavas());
        } catch (FileNotFoundException fnfe) {
            System.out.println("createKeyStore2: unable to open output: " + fnfe);
            return null;
        }
        try {
            ks.store(out,storePass);
        } catch (IOException ioe) {
            ExceptionHandler.handleIO("createKeyStore2: " + alias,"writing keystore",ioe);
            return null;
        } catch (Exception e) {
            ExceptionHandler.handle("createKeyStore2: " + alias,"security",e);
            return null;
        }
        try {
            out.flush();
            out.close();
        } catch (IOException ioe) {
            ExceptionHandler.handleIO("createKeyStore2: " + alias,"flush/close",ioe);
        }
        return ks;
    }

    /**
     * Create a private key with an associated self-signed certificate
     * returning them wrapped in an X500PrivateCredential
     *
     * Note: We use generateECKeyPair() from chapter6.EcDsaUtils and
     * createTrustAnchor() from chapter8.JcaX509Certificate. -- 
     * Java Cryptography - Tools and Techniques
     *
     * @return an X500PrivateCredential containing the key and its certificate.
     * @throws Exception
     * @see ExceptionHandler.handle 
     */
    public static PrivateCredential createSelfSignedCredentials() 
             throws Exception {
        JcaX509CertificateConverter certConverter =
            new JcaX509CertificateConverter().setProvider("BC");

        KeyPair selfSignedKp = EcDsaUtils.generateECKeyPair();

        X509CertificateHolder selfSignedHldr = createBCTrustAnchor(selfSignedKp, "SHA256withECDSA");

        X509Certificate selfSignedCert = certConverter.getCertificate(selfSignedHldr);

        return new PrivateCredential(selfSignedCert, selfSignedKp.getPrivate());
    }
    
    public static X509CertificateHolder createTecreationsTrustAnchor(KeyPair keyPair, String sigAlg)
    throws Exception
    {
        X500NameBuilder x500NameBld = new X500NameBuilder(BCStyle.INSTANCE)
            .addRDN(BCStyle.C, "CA")
            .addRDN(BCStyle.ST, "Alberta")
            .addRDN(BCStyle.L, "Heisler")
            .addRDN(BCStyle.O, "tecreations")
            .addRDN(BCStyle.CN, "Default Root Certificate");

        X500Name name = x500NameBld.build();

        X509v1CertificateBuilder certBldr = new JcaX509v1CertificateBuilder(
            name,
            calculateSerialNumber(),
            calculateDate(0),
            calculateDate(24 * 365 * 10),
            name,
            keyPair.getPublic());
        ContentSigner signer = new JcaContentSignerBuilder(sigAlg)
            .setProvider("BC").build(keyPair.getPrivate());

         return certBldr.build(signer);
    }

    /**
     * Create a key store suitable for use as a trust store, containing only the
     * certificates associated with each alias in the passed in credentialStore.
     *
     * @param type KeyStore type
     * @param credentialStore key store containing public/private credentials.
     * @return a key store containing only certificates.
     */
    public static KeyStore createTrustStore(String type, KeyStore credentialStore) {
        KeyStore store = null;
        try {
            store = KeyStore.getInstance(type);
        } catch (Exception kse) {
            ExceptionHandler.handle("createTrustStore","couldn't create keystore",kse);
            return null;
        }
        try {
            store.load(null, null);
        } catch (Exception e) {
            if (e instanceof IOException) {
                ExceptionHandler.handleIO("createTrustStore","reading keystore",e);
            } else {
                ExceptionHandler.handle("createTrustStore","security",e);
            }
            return null;
        }
        try {
            for (Enumeration<String> en = credentialStore.aliases(); en.hasMoreElements();) {
                String alias = en.nextElement();
                store.setCertificateEntry(alias, credentialStore.getCertificate(alias));
            }
        } catch (KeyStoreException kse) {
            System.err.println("createTrustStore(5): unable to retrieve aliases: " + kse);
        }
        return store;
    }
    
    public static KeyStore createTrustStoreFrom(String type, KeyStore credentialStore, String filename, char[] storePass) {
        KeyStore ks = createTrustStore(type,credentialStore);
        FileOutputStream out = null;
        File file = new File(filename);
        try {
            File deepest = file.getDeepestDirectory();
            deepest.mkdirs();
            out = new FileOutputStream(file.getJavas());
        } catch (FileNotFoundException fnfe) {
            System.err.println("createTrustStoreFrom: file not found: " + filename);
            return null;
        }
        try {
            ks.store(out,storePass);
        } catch (Exception e) {
           if (e instanceof IOException) {
                ExceptionHandler.handleIO("createTrustStoreFrom","writing keystore",e);
            } else {
                ExceptionHandler.handle("createTrustStoreFrom","security",e);
           }
           return null;
        }
        try {
            out.flush();
            out.close();
        } catch (IOException ioe) {
            ExceptionHandler.handleIO("createTrustStoreFrom","flushing/closing",ioe);
        }
        return ks;
    }

    public BasicConstraints getBasicConstraints(X509CertificateHolder holder) {
        BasicConstraints basicConstraints = BasicConstraints.fromExtensions(holder.getExtensions());
        return basicConstraints;
    }
    
    public ExtendedKeyUsage getExtendedKeyUsage(X509CertificateHolder holder) {
        ExtendedKeyUsage extKeyUsage = ExtendedKeyUsage.fromExtensions(
            holder.getExtensions());
        return extKeyUsage;
    }
    
    public KeyUsage getKeyUsage(X509CertificateHolder holder) {
        KeyUsage keyUsage = KeyUsage.fromExtensions(holder.getExtensions());
        return keyUsage;
    }

    // for eventually, after everything has been tested and snapshotted
    public X509Certificate getTecSigned() {
        KeyPairGenerator kpGen = null;
        try {
            kpGen = KeyPairGenerator.getInstance("EC", "BC");
        } catch (NoSuchAlgorithmException nsae) {
            System.err.println("getTecSigned: no such algorithm: " + nsae);
        } catch (NoSuchProviderException nspe) {
            System.err.println("getTecSigned: no such provider: BC : " + nspe);
        }
        KeyPair trustKp = kpGen.generateKeyPair();
        X509CertificateHolder certHolder = null;
        try {
            certHolder = createTecreationsTrustAnchor(trustKp, "SHA256withECDSA");
        } catch (Exception e) {
            ExceptionHandler.handle("getTecSigned","couldn't create trustCert" ,e);
        }
        return holderToCert(certHolder);
    }
    
    public X509Certificate holderToCert(X509CertificateHolder holder) {
        X509Certificate cert = null;
        try {
            cert = new JcaX509CertificateConverter()
            .setProvider("BC")
            .getCertificate(holder);       
        } catch (Exception e) {
            ExceptionHandler.handle("holderToCert",e);
        }
        return cert;
    }

    public static KeyStore openKeyStore(String type, String path, char[] storePass)
    throws UnrecoverableKeyException {
        if ((path == null | path.equals("")) | !new File(path).exists()) {
            throw new IllegalArgumentException("invalid path: '" + path + "'");
        }
        FileInputStream fis = null;
        KeyStore keystore = null;
        try {
            fis = new FileInputStream(path);
        } catch (IOException ioe) {
            ExceptionHandler.handleIO("openKeyStore","open: " + path,ioe);
            return null;
        }
        try {
            keystore = KeyStore.getInstance(type);
        } catch (KeyStoreException kse) {
            ExceptionHandler.handle("SecurityTool.openKeyStore","getting keystore instance", kse);
        }
        try {
            keystore.load(fis, null);
        } catch (IOException ioe) {
            ExceptionHandler.handleIO("SecurityTool.openKeyStore", "loading keystore", ioe);
        } catch (NoSuchAlgorithmException nsae) {
            ExceptionHandler.handle("SecurityTool.openKeyStore","loading keystore", nsae);
        } catch (CertificateException ce) {
            ExceptionHandler.handle("SecurityTool.openKeyStore","loading keystore", ce);
        }
        try {
            fis.close();
        } catch (IOException ioe) {
            ExceptionHandler.handleIO("openKeyStore","writing: closing",ioe);
        }
        return keystore;
    }
    
    public static KeyStore openTrustStore(String type, String path) {
        FileInputStream fis = null;
        KeyStore truststore = null;
        try {
            fis = new FileInputStream(path);
        } catch (IOException ioe) {
            ExceptionHandler.handleIO("openTrustStore","opening",ioe);
            return null;
        }
        try {
            truststore = KeyStore.getInstance(type);
            truststore.load(fis, null);
        } catch (FileNotFoundException e) {
            ExceptionHandler.handleIO("openKeyStore","not found: " + path,e);
            return null;
        } catch (IOException e) {
            ExceptionHandler.handleIO("openKeyStore","opening", e);
            return null;
        } catch (Exception e) {
            ExceptionHandler.handle("openKeyStore","security", e);
            return null;
        }
        try {
            fis.close();
        } catch (IOException ioe) {
            ExceptionHandler.handleIO("openKeyStore","closing",ioe);
        }
        return truststore;
    }
}