package ca.tecreations.components;

import ca.tecreations.Color;
import ca.tecreations.TecData;
import ca.tecreations.File;
import ca.tecreations.ImageTool;
import ca.tecreations.Platform;
import ca.tecreations.Point;
import ca.tecreations.ProjectPath;
import ca.tecreations.TextToken;
import ca.tecreations.text.GUITextTokenPainter;

import java.awt.BorderLayout;
import java.awt.Graphics;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import javax.swing.JMenuBar;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

/**
 *
 * @author Tim
 */
public class Magnifier extends TFrame  implements ActionListener, 
        MouseMotionListener, MouseWheelListener {
    
    // testing code
//    static BufferedImage desktop = ImageTool.getImage(ProjectPath.getDocumentsPath() + "desktop.png");
//    static BufferedImage purple = ImageTool.getImage(ProjectPath.getDocumentsPath() + "ADarkerPurple.png");
//    static BufferedImage box = ImageTool.getImage(ProjectPath.getDocumentsPath() + "box.png");
//    static BufferedImage systemTrayIcon = ImageTool.getImageAsResource(TecImageSupportForSysTray.class.getResource("tec.png"));
//    static BufferedImage[] images = new BufferedImage[] { desktop, purple, box, systemTrayIcon };
//    static int imageNum = 0;
    
    public static Magnifier instance;
    public JMenuBar menubar = new JMenuBar();
    public JMenu operations = new JMenu("Operations");
    public JMenuItem colorSelector = new JMenuItem("Select Color");
    public JMenuItem setScale = new JMenuItem("Set Scale 1");
    
    int centerWidth;
    int centerHeight;
    
    JPanel jPanel = new JPanel(false);
    ImagePanel imagePanel;

    BufferedImage image = null;
    BufferedImage scaled = null;
    Point previousLoc;
    int difference;
    int scale = 1;

    boolean firstRun = true;

    MagnifierStatus status;
    
    public Magnifier() {
        super(ProjectPath.getTecPropsPath() + "Magnifier.properties","Magnifier");
        imagePanel = new ImagePanel();
        setupGUI();
        Color bg = getProperties().getTecColor("background.color");
        if (bg == null) {
            bg = Color.TEC_ORANGE;
            getProperties().set("background.color",bg);
        }
        imagePanel.addMouseMotionListener(this);
    }
    
    public Magnifier(BufferedImage img) {
        super(ProjectPath.getTecreationsPath() + "properties" + File.separator + "Magnifier.properties","Magnifier");
        image = img;
        scaled = img;
        imagePanel = new ImagePanel(image);
        // testing code
        // imagePanel.setImage(images[imageNum]);
        imagePanel.setImage(img,true);
        setupGUI();
        Color bg = getProperties().getTecColor("background.color");
        if (bg == null) {
            bg = Color.TEC_ORANGE;
            getProperties().set("background.color",bg);
        }
        jPanel.setBackground(bg);
        imagePanel.addMouseMotionListener(this);
    }
    
    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == colorSelector) {
            Color color = Platform.requestColor(new Color(jPanel.getBackground()), "Select a background color...");
            jPanel.setBackground(color);
            getProperties().set("background.color",color);
            repaint();
        } else if (e.getSource() == setScale) {
            setScale(1);
            doResizeImage();
            if (shouldCenter()) centerImage();
            else imagePanel.setLocation(15,15);
        }
    }
    
    public void centerImage() {
        imagePanel.setLocation(centerWidth / 2 - imagePanel.getWidth() / 2, centerHeight / 2 - imagePanel.getHeight() / 2);
    }
    
    public void componentResized(ComponentEvent e) {
        System.out.println("Magnifier.ComponentResized");
        super.componentResized(e);
        if (jPanel != null) {
            centerWidth = jPanel.getSize().width;
            centerHeight = jPanel.getSize().height;
        }
    }
    
    private void doResizeImage() {
        try {
            scaled = getScaled();
            imagePanel.setImage(scaled,true);
            repaint();
        } catch (OutOfMemoryError err) {
            System.err.println("Scaled image requires more memory. Scale: " + scale);
            GUITextTokenPainter text = new GUITextTokenPainter(TecData.LG_CODE_POINTS, new TextToken("Scaled image requires more memory: Scale: " + scale));
            text.setForeground(Color.red);
            text.setBackground(getBackground());
            BufferedImage jPanelImg = ImageTool.getNewBufferedImage(centerWidth,centerHeight);
            int xOff = centerWidth / 2 - text.getSize().width / 2;
            int yOff = centerHeight / 2 - text.getSize().height / 2;
            Graphics g = jPanelImg.getGraphics();
            g.setColor(getBackground());
            g.fillRect(0,0,jPanelImg.getWidth(),jPanelImg.getHeight() - 1);
            text.paintAt(g,xOff,yOff);
            setScale(1);
            setImage(jPanelImg); 
            imagePanel.setLocation(0,0);
        }
    }
    
    private void doSetImage(BufferedImage img) {
        image = img;
        scaled = getScaled();
        imagePanel.setImage(scaled,true);
        repaint();
    }
    
    private void doSetLocation(int mouseX, int mouseY) {
        int startX = previousLoc.x;
        int startY = previousLoc.y;
        int normalMouseX = mouseX / (scale - difference);
        int normalMouseY = mouseY / (scale - difference);
        int x = startX + mouseX - normalMouseX * scale;
        int y = startY + mouseY - normalMouseY * scale;
        imagePanel.setLocation(x,y);
    }

    private void extract(BufferedImage image,int x, int y) {
        int width = imagePanel.getSize().width / scale;
        int height = imagePanel.getSize().height / scale;
        int wHalf = width / 2;
        int hHalf = height / 2;
        int newX = x - wHalf;
        int newY = y - hHalf;
        int[] pixels = null;
        // extract rectangle
        image = ImageTool.getRegion(image,newX,newY,width,height);
        
        // scale and paint
        if (scale > 1) {
            scaled = ImageTool.getResized2(image,imagePanel.getSize().width,imagePanel.getSize().height);
        }
        repaint();
    }
    
    
    public BufferedImage getImage() { return image; }
    
    public BufferedImage getScaled() {
        int width = image.getWidth() * scale;
        int height = image.getHeight() * scale;
        if (width > 0 && height > 0) {
            return ImageTool.getResized2(image, width, height);
        } else {
            throw new IllegalArgumentException("Image size must be greater than 0x0: Size: " + image.getWidth() + "x" + image.getHeight());
        }
    }
    
    public static void launch(final BufferedImage img) {
        SwingUtilities.invokeLater(() -> {
            instance = new Magnifier(img);
            instance.standalone = true;
            instance.setVisible(true);
            instance.setTitle("Magnifier");
        });
    }
    
    public void magnify(BufferedImage img,int x, int y) {
        extract(img,x,y);
    }
    
    public void magnify(BufferedImage img, Point p) {
        extract(img,p.x,p.y);
    }
    
    public static void main(String[] args) {
        String darkerPurple = "D:\\ADarkerPurple.png";
        String ansiLookup = "F:\\ANSI-256-lookup-table.png";
        final BufferedImage purple = ImageTool.getImage(darkerPurple);
        if (purple != null) {
            SwingUtilities.invokeLater(() -> {
                launch(purple);
            });
        }
    }
    
    public void mouseDragged(MouseEvent e) {
    }
    
    public void mouseMoved(MouseEvent e) {
        status.setColor(imagePanel.extract(new Point(e.getX(),e.getY())));
        status.setMouseLocation(e.getX(),e.getY());
    }
    
    @Override
    public void mouseWheelMoved(MouseWheelEvent e) {
        int notches = e.getWheelRotation();
        boolean ctrl = e.isControlDown();
        boolean shift = e.isShiftDown();
        boolean set = false;
        if (ctrl) {
            previousLoc = imagePanel.getTecLocation(); 
            if (notches < 0) {
                if (shift) {
                    difference = 10;
                    scale += 10; 
                    set = true;
                } else {
                    difference = 1;
                    scale++;
                    set = true;
                }
            } else {
                if (shift) {
                    if (scale > 10) { 
                        difference = -10;
                        scale += difference;
                        set = true;
                    }
                } else {
                    if (scale > 1) {
                        difference = -1;
                        scale--;
                        set = true;
                    }
                }
            }
            if (set) { 
                setScale(scale);
                doResizeImage();
                doSetLocation(e.getX(),e.getY());
                repaint();
            }
        }
    }

    @Override
    public void paint(Graphics g) {
        while (!isVisible()) Platform.sleep(100);
        super.paint(g);
        if (firstRun) {
            centerWidth = jPanel.getSize().width;
            centerHeight = jPanel.getSize().height;
            if (shouldCenter()) {
                centerImage();
            } else {
                imagePanel.setLocation(15,15);
            }
            firstRun = false;
        }
    }
    
    public Magnifier setImage(BufferedImage img) {
        doSetImage(img);
        return this;
    }
    
    public Magnifier setScale(int scale) {
        this.scale = scale;
        return this;
    }
    
    public void setupGUI() {
        menubar.add(operations);
        operations.add(colorSelector);
        operations.add(setScale);
        colorSelector.addActionListener(this);
        setScale.addActionListener(this);
        setJMenuBar(menubar);
        setLayout(new BorderLayout());
        add(jPanel,BorderLayout.CENTER);
        jPanel.setLayout(null);
        jPanel.add(imagePanel);
        status = new MagnifierStatus(this);
        add(status,BorderLayout.SOUTH);
        validate();
        imagePanel.addMouseMotionListener(this);
        imagePanel.addMouseWheelListener(this);
        if (getProperties().wasCreated()) {
            setLocationRelativeTo(null);
        }
    }

    public boolean shouldCenter() {
        return (imagePanel.getWidth() < centerWidth || imagePanel.getHeight() < centerHeight);
    }
    
}