package ca.tecreations.apps.launcher;

import ca.tecreations.File;
import ca.tecreations.JarReader;
import ca.tecreations.Platform;
import ca.tecreations.Point;
import ca.tecreations.ProjectPath;
import ca.tecreations.Properties;
import ca.tecreations.StringTool;
import ca.tecreations.TecData;
import ca.tecreations.components.ProgressDialog;
import ca.tecreations.components.TFrame;
import ca.tecreations.net.TecStreamPrinterClient;
import ca.tecreations.net.TecStreamPrinterServer;

import java.awt.BorderLayout;
import java.awt.event.*;
import java.util.ArrayList; 
import java.util.List;

import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;

/**
 *
 * @author Tim
 */
public class Launcher extends TFrame implements ActionListener, MouseListener,
        ProcessListener {
    public static Launcher instance;
    public static Properties properties = new Properties(ProjectPath.getTecPropsPath() + Launcher.class.getSimpleName() + ".properties");
    public static ProgressDialog progress = null;
    public static TSPC_Output tspcOutput;
    
    String path;
    JButton selectJar = new JButton("Select .jar");
    JButton selectPath = new JButton("Select Path");

    JLabel mainsLabel = new JLabel("Main Classes:");
    DefaultListModel<String> mainsModel = new DefaultListModel<>();
    JList<String> mainsList = new JList<>(mainsModel);
    JScrollPane mainsScroller;
    JSplitPane split1;
    JLabel targetsLabel = new JLabel("Targets:");
    ProcessesTable targetsTable = new ProcessesTable();
    JScrollPane targetsScroller;
    JTextField args = new JTextField(256);

    JPopupMenu mainsMenu = new JPopupMenu();
    JMenuItem addTarget = new JMenuItem("Add Target");

    JPopupMenu controlMenu = new JPopupMenu();
    JMenuItem start = new JMenuItem("Start");
    JMenuItem startWithArgs = new JMenuItem("Start With Args...");
    JMenuItem stop = new JMenuItem("Stop");
    JMenuItem restart = new JMenuItem("Restart");
    JMenuItem startMany = new JMenuItem("Start Many");
    JMenuItem stopInstances = new JMenuItem("Stop Instances...");
    JMenuItem viewConsole = new JMenuItem("View Console");
    JMenuItem remove = new JMenuItem("Remove");

    List<Runtime> runningApps = new ArrayList<>();
    List<ProcessWatcher> watchers = new ArrayList<>();
    TecStreamPrinterClient tspc = TecData.TSPC;
    
    public Launcher(String initialPath) {
        super(properties, "NewLauncher");
        tspcOutput = new TSPC_Output(properties);
        path = initialPath;
        // headless or gui?
        setupGUI();
        setExitOnClose(true);
        if (properties.wasCreated()) {
            setLocationRelativeTo(null);
            setSize(720,480);
            tspcOutput.setLocation(new Point(0,Platform.getDesktopSize().height - 480));
        }
        tspcOutput.setVisible(true);
    }

    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == selectJar) {
            String name = Platform.requestFile(this, new File(ProjectPath.getDownloadsPath()), "Select a .jar file.");
            if (name != null && new File(name).getExtension().equals("jar")) {
                getMainsJar(name);
            }
        } else if (e.getSource() == selectPath) {
            String path = Platform.requestDirectory(this, new File(ProjectPath.getProjectsHome()), "Select a class path.");
            if (path != null) {
                File parent = new File(path).getDeepestDirectory().getDeepestDirectory();
                getMainsClassPath(parent.getUnwrapped(),path);
            }
        } else if (e.getSource() == addTarget) {
            String fqcn = mainsList.getSelectedValue();
            String key = getID(path, fqcn, args.getText());
            if (!properties.hasKey(key)) {
                targetsTable.addRow(new Runtime(path, fqcn, args.getText()));
                properties.set(key,"-1");
            }
        } else if (e.getSource() == start) {
            start();
        } else if (e.getSource() == startWithArgs) {
        } else if (e.getSource() == stop) {
            stop();
        } else if (e.getSource() == restart) {
            ProcessesTableModel model = targetsTable.getTableModel();
            Runtime runtime = model.getSelectedRuntime();
            removePID(runtime.getKey(TecData.TEC_SEP), runtime.getPID());
            removeWatcher(runtime);
            runtime.restart();
            addWatcher(runtime);
            model.setRuntime(model.indexOf(runtime), runtime);
            model.fireTableDataChanged();
            addPID(runtime.getKey(TecData.TEC_SEP), runtime.getPID());
        } else if (e.getSource() == startMany) {
        } else if (e.getSource() == stopInstances) {
            /// for 1 to list... stop(item)
        } else if (e.getSource() == viewConsole) {
            ProcessesTableModel model = targetsTable.getTableModel();
            Runtime runtime = model.getSelectedRuntime();
            runtime.setConsoleVisible(!runtime.isConsoleVisible());
        } else if (e.getSource() == remove) {
        }
    }

    public void addPID(String id, Long pid) {
        List<String> pids = properties.getList(id);
        if (pids.contains("-1")) {
            properties.set(id,pid);
        } else if (!pids.contains(pid.toString())) {
            pids.add(pid.toString());
            properties.set(id, pids);
        }
    }

    public void addWatcher(Runtime runtime) {
        ProcessWatcher watcher = new ProcessWatcher(runtime);
        watcher.addProcessListener(this);
        watcher.start();
        watchers.add(watcher);
    }
    
    public void close() {
        super.close();
        for(int i = 0; i < runningApps.size();i++) {
            stop(runningApps.get(i));
        }
        if (TecData.TSPS != null) {
            TecData.TSPS.stopRunning();
        }
    }

    public static void createAndShowGUI(final String runtimePath) {
        instance = new Launcher(runtimePath);
        instance.setVisible(true);
        while (!instance.isVisible()) Platform.sleep(125);
        instance.parseProperties();
        progress = new ProgressDialog(instance);
        if (runtimePath.toLowerCase().endsWith(".jar")) {
            instance.getMainsJar(runtimePath);
        } else {
            instance.getMainsClassPath(ProjectPath.getProjectsHome(),
                                       ProjectPath.getProjectsHome() + TecData.TEC_VERSION + File.separator);
        }
    }
 
    public void doLogAction(String s) {
        tspcOutput.addLine(s);
        System.out.println(s);
    }

    public List<File> getFiles(String path) {
        List<File> files = new ArrayList<>();
        File[] entries = new File(path).listFiles();
        for(int i = 0; i < entries.length;i++) {
            if (entries[i].isFile()) {
                if (entries[i].getExtension().equals("java")) {
                    files.add(entries[i]);
                }
            }
        }
        for(int i = 0; i < entries.length;i++) {
            if (entries[i].isDirectory()) {
                List<File> subFiles = getFiles(entries[i].getAbsolutePath());
                for(int j = 0;j  < subFiles.size();j++) {
                    files.add(subFiles.get(j));
                }
            }
        }
        return files;
    }
    
    public String getID(String classPath, String fqcn, String args) {
        return classPath + TecData.TEC_SEP + fqcn + TecData.TEC_SEP + args;
    }

    public void getMainsJar(String path) {
        this.path = path;
        doLogAction("getMainsJar: path: " + path);
        progress.setLocationRelativeTo(null);
        progress.setVisible(true);
        mainsModel.removeAllElements();
        JarReader reader = new JarReader(path);
        // we need to unpack to get the data, retrieve state, return yes/no.
        List<String> fqcns = reader.getJavaMainsFQCNsForJar(progress);
        doLogAction("fqcns: " + fqcns.size());
        for(int i = 0; i < fqcns.size();i++) {
            mainsModel.addElement(fqcns.get(i));
            int percent = (int)((double)(i + 1) / (double)fqcns.size() * (double)100);
            progress.setTitle("Working (" + percent + "%)...");
            progress.setPercent(percent);
            doLogAction("fqcn: " + fqcns.get(i));
        }
        progress.setVisible(false);
    }

    public void getMainsClassPath(String root, String path) {
        File[] entries = new File(path).listFiles();
        List<File> files = getFiles(root);
        progress.setLocationRelativeTo(null);
        progress.setVisible(true);
        mainsModel.removeAllElements();
        for (int i = 0; i < files.size(); i++) {
            System.out.println("FQCN: " + files.get(i).getFQCN(root));
            mainsModel.addElement(files.get(i).getFQCN(root));
            int percent = (int)((double)(i + 1) / (double)files.size() * (double)100);
            progress.setTitle("Working (" + percent + "%)...");
            progress.setPercent(percent);
        }
        progress.setVisible(false);
    }

    public static void launch(final String runtimePath) {
        SwingUtilities.invokeLater(() -> {
            createAndShowGUI(runtimePath);
        });
    }

    public static void main(String[] args) {
        TecData.TSPS = new TecStreamPrinterServer();
        TecData.TSPC = new TecStreamPrinterClient("NewLauncher");
        String runtimePath = ProjectPath.getRuntimePath(Launcher.class.getProtectionDomain());
        if (runtimePath.contains("NetBeansProjects")) {
            runtimePath = ProjectPath.instance.getProjectPath();
        }
        launch(runtimePath);
    }

    public void mouseClicked(MouseEvent e) {
        if (e.getSource() == targetsTable) {
            List<Runtime> runtimes = targetsTable.getTableModel().getRuntimes();
            int row = targetsTable.rowAtPoint(e.getPoint());
            for (int i = 0; i < runtimes.size(); i++) {
                if (i != row) {
                    runtimes.get(i).setSelected(false);
                } else {
                    runtimes.get(i).setSelected(true);
                }
            }
            targetsTable.getTableModel().fireTableDataChanged();
            if (SwingUtilities.isRightMouseButton(e)) {
                controlMenu.show(targetsTable, e.getX(), e.getY());
            }
        } else if (e.getSource() == mainsList) {
            mainsList.setSelectedIndex(mainsList.locationToIndex(e.getPoint()));
            if (SwingUtilities.isRightMouseButton(e)) {
                mainsMenu.show(mainsList, e.getX(), e.getY());
            }
        }
    }

    public void mouseEntered(MouseEvent e) {
    }

    public void mouseExited(MouseEvent e) {
    }

    public void mousePressed(MouseEvent e) {
    }

    public void mouseReleased(MouseEvent e) {
    }

    public void parseProperties() {
        List<String> keys = getProperties().getKeys();
        List<String> vals = getProperties().getValues();
        for(int i = 0; i < keys.size();i++) {
            if (keys.get(i).contains(TecData.TEC_SEP)) {
                List<String> parts = StringTool.explode(keys.get(i),TecData.TEC_SEP);
                targetsTable.addRow(new Runtime(parts.get(0),parts.get(1),parts.get(2)));
            }
        }
    }
    
    @Override
    public void processEnded(Runtime runtime) {
        runtime.stop();
    }

    public void removePID(String id, Long pid) {
        List<String> pids = properties.getList(id);
        doLogAction(Launcher.class.getSimpleName() + ".removePID: " + pid.toString());
        for (int i = pids.size() - 1; i >= 0; i--) {
            if (pids.get(i).equals(pid.toString())) {
                pids.remove(i);
                properties.set(id, pids); // auto writes, unless delayWrite
            }
        }
    }

    public void removeWatcher(Runtime runtime) {
        for (int i = 0; i < watchers.size(); i++) {
            if (watchers.get(i).getRuntime().equals(runtime)) {
                watchers.get(i).stopRunning();
                watchers.remove(i);
                return;
            }
        }
    }
    
    public void setupGUI() {
        mainsScroller = new JScrollPane(mainsList, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        setLayout(new BorderLayout());
        JPanel targetButtons = new JPanel(new java.awt.GridLayout());
        targetButtons.add(selectJar);
        targetButtons.add(selectPath);
        add(targetButtons, BorderLayout.NORTH);
        selectJar.addActionListener(this);
        selectPath.addActionListener(this);

        JPanel left = new JPanel(new BorderLayout(), false); // we could call this mainsHolder
        left.add(mainsLabel, BorderLayout.NORTH);
        mainsScroller = new JScrollPane(mainsList, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        left.add(mainsScroller, BorderLayout.CENTER);

        JPanel right = new JPanel(new BorderLayout(), false);
        JPanel targetsHolder = new JPanel(new BorderLayout(), false);
        targetsHolder.add(targetsLabel, BorderLayout.NORTH);
        targetsScroller = new JScrollPane(targetsTable, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        targetsHolder.add(targetsScroller, BorderLayout.CENTER);
        right.add(targetsHolder, BorderLayout.CENTER);

        split1 = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, left, right);
        add(split1, BorderLayout.CENTER);
        int width = getSize().width / 3;
        split1.setDividerLocation(width);

        JPanel bottom = new JPanel(new BorderLayout());
        JPanel argsPanel = new JPanel(new BorderLayout());
        argsPanel.add(new JLabel("Arguments: "), BorderLayout.WEST);
        argsPanel.add(args, BorderLayout.CENTER);
        bottom.add(argsPanel, BorderLayout.NORTH);
        add(bottom, BorderLayout.SOUTH);
        validate();
        args.addActionListener(this);

        targetsTable.getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        mainsList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        mainsList.addMouseListener(this);
        targetsTable.addMouseListener(this);

        mainsMenu.add(addTarget);
        addTarget.addActionListener(this);

        controlMenu.add(start);
        controlMenu.add(startWithArgs);
        controlMenu.add(stop);
        controlMenu.add(restart);
        controlMenu.add(startMany);
        controlMenu.add(stopInstances);
        controlMenu.add(viewConsole);
        controlMenu.addSeparator();
        controlMenu.add(remove);
        start.addActionListener(this);
        startWithArgs.addActionListener(this);
        stop.addActionListener(this);
        restart.addActionListener(this);
        startMany.addActionListener(this);
        stopInstances.addActionListener(this);
        viewConsole.addActionListener(this);
        remove.addActionListener(this);
    }

    public void start() {
        ProcessesTableModel model = targetsTable.getTableModel();
        Runtime runtime = model.getSelectedRuntime();
        runtime.start();
        runningApps.add(runtime);
        addWatcher(runtime);
        Process process = runtime.getProcess();
        model.setRuntime(model.indexOf(runtime), runtime);
        model.fireTableDataChanged();
        addPID(runtime.getKey(TecData.TEC_SEP), runtime.getPID());
        runtime.setConsoleVisible(true);
        if (runtime.getConsole().getProperties().wasCreated()) {
            runtime.getConsole().setLocation_BottomRight();
        }
    }
    
    public void stop() {
        ProcessesTableModel model = targetsTable.getTableModel();
        Runtime runtime = model.getSelectedRuntime();
        stop(runtime);
    }
    
    public void stop(Runtime runtime) {
        ProcessesTableModel model = targetsTable.getTableModel();
        runtime.stop();
        runtime.getConsole().dispose();
        runningApps.remove(runtime);
        removePID(runtime.getKey(TecData.TEC_SEP), runtime.getPID());
        removeWatcher(runtime);
        model.setRuntime(model.indexOf(runtime), runtime);
        model.fireTableDataChanged();
    }
    
}
