package ca.tecreations.apps.javacompiler;

import ca.tecreations.File;
import ca.tecreations.Platform;
import ca.tecreations.ProjectPath;
import ca.tecreations.Properties;
import ca.tecreations.StringTool;
import ca.tecreations.SystemTool;
import ca.tecreations.lang.java.GetPackage;
import ca.tecreations.misc.Time;

import java.io.*;
import java.nio.file.*;
import static java.nio.file.LinkOption.*;
import static java.nio.file.StandardWatchEventKinds.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * Example to watch a directory (or tree) for changes to files.
 *
 */
public class JavaCompilerThread extends Thread {
    private WatchService service = null;
    private Map<WatchKey, Path> keys = null;
    boolean debug = false;
    String path;
    JavaCompiler gui;
    boolean running = true;
    SystemTool tool = new SystemTool();

    List<String> paths = new ArrayList<>();
    List<WatchKey> listWatchKeys = new ArrayList<>();

    /**
     * Creates a WatchService and registers the given directory
     */
    public JavaCompilerThread() {
        try {
            service = FileSystems.getDefault().newWatchService();
            //keys = new HashMap<WatchKey, Path>();
            keys = new HashMap<>();
        } catch (IOException ioe) {
            System.out.println("IOE: " + ioe);
        }
    }

    public JavaCompilerThread(JavaCompiler gui) {
        this();
        this.gui = gui;
    }

    @SuppressWarnings("unchecked")
    static <T> WatchEvent<T> cast(WatchEvent<?> event) {
        return (WatchEvent<T>) event;
    }

    /*
    public static void main(String[] args) {
        SystemCompilerThreaded thread = new SystemCompilerThreaded();
        thread.start("c:\\users\\tim\\documents\\tecreations-core\\");
        SystemCompilerThreaded thread2 = new SystemCompilerThreaded();
        thread2.start("c:\\users\\tim\\documents\\eclipse-aether\\");
    }       
     */
    public void cancel(String path) {
        int index = paths.indexOf(path);
        listWatchKeys.get(index).cancel();
        //System.out.println("Cancelled: " + path);
    }

    public void cancelAndRemove(String path) {
        int index = paths.indexOf(path);
        if (index != -1) {
            listWatchKeys.get(index).cancel();
            paths.remove(index);
            listWatchKeys.remove(index);
            paths.remove(index);
            //System.out.println("Cancelled: " + path);
        }
    }

    public void cancelAll() {
        //System.out.println("Cancelling All...");
        for (int i = paths.size() - 1; i >= 0; i--) {
            cancel(paths.get(i));
        }
        //System.out.println("Cancelled All");
    }

    public void cancelAndRemoveAll() {
        //System.out.println("Cancelling All...");
        for (int i = paths.size() - 1; i >= 0; i--) {
            cancelAndRemove(paths.get(i));
        }
        //System.out.println("Cancelled All");
    }

    public void deleteAnyAdditionalClassesStartingWithName(String absPath) {
        File file = new File(absPath);
        String path = absPath.substring(0, absPath.lastIndexOf(File.separator) + 1);
        String pathLower = path.toLowerCase();
        File[] list = new File(path).listFiles();
        String selector;
        String msg = "";
        if (list != null) {
            for (int i = 0; i < list.length; i++) {
                selector = pathLower + file.getFilenameOnly().toLowerCase() + "$";
                if (list[i].getAbsolutePath().toLowerCase().startsWith(selector)) {
                    msg = "Deleted: " + list[i] + " : " + list[i].delete(true);
                    gui.addElement(msg);
                    System.out.println(msg);
                }
                
            }
        } 
    }
    
    public void doDebugPrint(List<String> data) {
        for(int i = 0; i < data.size();i++) {
            System.out.println(data.get(i) + ",");
        }
    }

    public String getTime() { return new Time().getAppEventTime(); }
    
    /**
     * Register the given directory with the WatchService
     */
    private void register(Path dir) {
        WatchKey key = null;
        try {
            key = dir.register(service, ENTRY_CREATE, ENTRY_DELETE, ENTRY_MODIFY);
            paths.add(dir.toFile().getAbsolutePath());
        } catch (IOException ioe) {
            System.err.println("Skipping: " + dir.toString());
        }
        if (key != null) {
            Path prev = keys.get(key);
            listWatchKeys.add(key);
            if (prev == null) {
                if (debug) {
                    System.out.format("JavaSystemCompiler: register: %s\n", dir);
                }
            } else {
                if (!dir.equals(prev)) {
                    if (debug) {
                        System.out.format("update: %s -> %s\n", prev, dir);
                    }
                }
            }
            keys.put(key, dir);
        }
    }

    /**
     * Register the given directory, and all its sub-directories, with the
     * WatchService.
     */
    private void registerAll(final Path start) {
        java.io.File[] roots = start.toFile().listFiles();
        if (roots != null) {
            register(start);
            for (int i = 0; i < roots.length; i++) {
                if (roots[i].isDirectory()) {
                    if (roots[i].getAbsolutePath().toLowerCase().endsWith(File.separator + "jars")) {
                        // skip the jars, we don't want to compile dependency code, but we can
                    } else {
                        // exclude the one's giving errors
                        if (!roots[i].getAbsolutePath().toLowerCase().endsWith("config.msi")
                                && !roots[i].getAbsolutePath().equals("/System/Library/DirectoryServices/DefaultLocalDB/Default")) {
                            registerAll(Paths.get(roots[i].getAbsolutePath()));
                        }
                    }
                }
            }
        }
    }

    /**
     * Process all events for keys queued to the watcher
     */
    public void run() {
        String stringOutput = "";
        WatchKey key;
        boolean step = false;
        while (running) {
            Properties properties = gui.getProperties();

            // wait for key to be signalled
            try {
                key = service.take();
            } catch (InterruptedException x) {
                return;
            }  
            Path dir = keys.get(key);
                     
            if (dir == null) {
                System.err.println("WatchKey not recognized!!: " + key.toString());
                continue;
            } 
            List<WatchEvent<?>> events = key.pollEvents();
            for (int i = 0; i < events.size(); i++) {
                WatchEvent<Path> ev = cast(events.get(i));
                Path name = ev.context();
                Path child = dir.resolve(name);
                String path = child.toString();
                String extension = path.substring(path.lastIndexOf(".") + 1).toLowerCase();
                boolean isJava = extension.equals("java");
                WatchEvent.Kind kind = events.get(i).kind();

                // store all this data in a List<> to pass to doDebugPrint
                List<String> debugData = new ArrayList<>();
                debugData.add("ev: " + events.get(i).toString());
                debugData.add("name: " + name.toString());
                debugData.add("child: " + child.toString());
                debugData.add("path: " + path);
                debugData.add("ext: " + extension);
                debugData.add("isJava: " + isJava);
                String kindString = null;
                if (kind == OVERFLOW) {
                    kindString = "overflow";
                } else if (kind == ENTRY_CREATE) {
                    kindString = "create";
                } else if (kind == ENTRY_MODIFY) {
                    kindString = "modify";
                } else if (kind == ENTRY_DELETE) {
                    kindString = "delete";
                }
                debugData.add("kind: " + kindString);

                // handle overflow... for this we just print it out
                if (kind == OVERFLOW) {
                    System.out.println("JavaCompilerThread: KEY: OVERFLOW");
                    continue;
                }
 
                // obtain the projectDir from the item
                String prjsHome = ProjectPath.getProjectsHome();
                String tmp = "";
//                System.out.println("tmp: " + tmp);
                String projectDir = "DOES_NOT_CONTAIN_A_PROJECT_DIR : " + path;
                try {
                    tmp = path.substring(prjsHome.length());
                    if (tmp.indexOf(File.separator) != -1) {
                        projectDir = tmp.substring(0,tmp.indexOf(File.separator));
                    }
                } catch (ArrayIndexOutOfBoundsException aioobe) {
                    System.err.println("Does not contain a project dir: " + path);
                }
                if (kind.equals(ENTRY_CREATE)) {
                    System.out.println("ENTRY_CREATE: child: " + child);
                    if (Files.isDirectory(child, NOFOLLOW_LINKS)) {
                        // register the directory 
                        doDebugPrint(debugData);
                        if (new File(child).exists()) {
                            register(new File(child).toPath());
                        }
                    } else { 
                        // isFile()
                        if (isJava) { 
                            // compile created file
                            String pkg = new GetPackage().getPackage(path);
                            System.out.println("Package: " + pkg);
                            String fqcnPath = "";
                            String pkgPath = pkg.replace(".", File.separator);
                            if (pkgPath.length() > 0) {
                                pkgPath += File.separator;
                            }
                            fqcnPath = pkgPath + new File(path).getName();
                            String classPath = path.substring(0, path.length() - fqcnPath.length());
                            classPath = prjsHome + projectDir + File.separator;
        
                            tool.compile(classPath, new File(path), true);
                            gui.setErrorOutput(tool.getTokens());
                            String s = getTime() + ": create : " + projectDir + " : " + File.getClassNameFrom(path);
                            if (gui != null) {
                                gui.addElement(s);
                            }
                            System.out.println(s);
                        } 
                    }
                } else if (kind == ENTRY_DELETE) {
                    if (new File(child).isDirectory()) {
                        // register the directory 
                        System.out.println("cancelAndRemove(" + path + ")");
                        cancelAndRemove(path);
                        System.out.println("Not calling: File(path).delete()");
                        //new File(path).delete();
                    } else { 
                        //System.out.println("Delete"); 
                        if (isJava) {
                            //gui.post("compile: " + SystemTool.getClassName(path) + " : "+ SystemTool.compile(Tecreations.getProjectPath(),path));
                            // align the output for easier reading
                            String s = getTime() + ": " + Platform.getClientPlatform() + ": delete:  " + child.toString();
                            if (gui != null) {
                                gui.addElement(s);
                            } else {
                                System.out.println(s);
                            } 
                            String dotJava = child.toString();
                            String dotClass = dotJava.replace(".java", ".class");
                            if (new File(dotClass).delete(true)) {
                                String msg = "Deleted: " + dotClass;
                                gui.addElement(msg);
                                System.out.println(msg);
                            }
                            deleteAnyAdditionalClassesStartingWithName(dotClass);
                        } 
                    }
                } else if (kind == ENTRY_MODIFY) {
                    if (new File(path).isDirectory()) {
                        // don't remove the old path, there will be no entries
                        // from there unless the user re-creates it
                        // and ENTRY_CREATE performs duplicate skipping
                        register(new File(path).toPath());
                    } else { 
                        if (debug) {
                            System.out.println("Modify...");
                            System.out.println("IsWin : " + Platform.isWin());
                            System.out.println("IsJava: " + isJava);
                        }
                        if (isJava) {
                            String pkg = new GetPackage().getPackage(path);
                            
                            String fqcnPath = "";
                            String pkgPath = pkg.replace(".", File.separator);
                            if (pkgPath.length() > 0) {
                                pkgPath += File.separator;
                            }
                            fqcnPath = pkgPath + new File(path).getName();
                            
                            String classPath = "";
                            String sep = (prjsHome.contains("/") ? "/" : "\\");
                            classPath = path.substring(0, path.indexOf(sep,prjsHome.length() + 1) + 1);
                            //System.out.println("ClassPath: " + classPath);
                            if (Platform.isWin()) {
                                step = !step; 
                                // first execution, Windows Specific Case
                                if (step) {
                                    
                                    tool.compile(classPath, new File(path), true);
                                    Process p = tool.getProcess();
                                    // set the error output
                                    if (gui != null) {
                                        gui.setErrorOutput(tool.getTokens());
                                    }
                                  
                                    // set the status
                                    String shortPkgName = "";
                                    String fqcn = StringTool.replaceAll(pkgPath + new File(path).getFilenameOnly(),File.separator,".");
                                    //shortPkgName = ShortPackageAndName.getFor(fqcn,32);
//                                    stringOutput = getTime() + " : compile: " + projectDir + " : " + shortPkgName + " : " + File.getClassNameFrom(path) + " : " + exitValue;
                                    stringOutput = getTime() + " : compile: " + tool.getExitValue() + " : " + projectDir + " : " + fqcn;
                                    if (gui != null) {
                                        gui.addElement(stringOutput);
                                    }  
                                    System.out.println(stringOutput); 
                                }   
                            }
                        }
                    }
                }
                // reset key and remove from set if directory no longer accessible
                boolean valid = key.reset();
                if (!valid) {
                    keys.remove(key);
                    // all directories are inaccessible  
                    if (keys.isEmpty()) {
                        break;
                    }
                }
            }
        }
    }  

    public void setCompiler(JavaCompiler gui) {
        this.gui = gui;
    }

    public void start(String path) {
        this.path = path;
        System.out.println("Compiling: " + path);
        registerAll(Paths.get(path));
        super.start();
    }

    public void stopRunning() {
        System.out.println("Stopping: " + path);
        cancelAll();
        running = false;
    }
}
