package ca.tecreations.apps.javacompiler;

import ca.tecreations.File;
import ca.tecreations.ImageTool;
import ca.tecreations.LockFile;
import ca.tecreations.Platform;
import ca.tecreations.ProjectPath;
import ca.tecreations.Properties; 
import ca.tecreations.SystemToken;
import ca.tecreations.SystemTool;
import ca.tecreations.TextFile;
import ca.tecreations.components.TFrame;
 
import java.awt.AWTException;
import java.awt.BorderLayout;
import java.awt.CheckboxMenuItem;
import java.awt.FlowLayout;
import java.awt.MenuItem;
import java.awt.PopupMenu;
import java.awt.Rectangle;
import java.awt.TrayIcon;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.imageio.ImageIO;
import javax.swing.*;

/**
 *
 * @author Tim
 */
public class JavaCompiler implements ActionListener, ItemListener {
    public static final String MAKE_DELTAS = "make.deltas";
    public static final java.awt.SystemTray tray = java.awt.SystemTray.getSystemTray();
    public static BufferedImage img = null;
    static {
        String dir = ProjectPath.getProjectPath() + ProjectPath.getSubPath("ca.tecreations.icons");
        String name = "coffeebean_16_16.png";
        
        img = ImageTool.getImage(dir+name);
        if (img != null) {
            img = ImageTool.getResized(img, 16, 16);
            ImageTool.saveImage(img,ProjectPath.getProjectPath() + ProjectPath.getSubPath("ca.tecreations.apps.systemtray") + "coffeebeen_16_16.png");
        }
    }

    public static final TrayIcon trayIcon = new TrayIcon(img);
    final PopupMenu popup = new PopupMenu();
    MenuItem paths = new MenuItem("Paths");
    MenuItem output = new MenuItem("Output");
    MenuItem status = new MenuItem("Status");
    CheckboxMenuItem startOpen = new CheckboxMenuItem("Start Open");
    MenuItem exit = new MenuItem("Exit");
 
    private static boolean standalone = false;
    private static final Object lock = new Object();

    public static JavaCompiler instance;

    public static final String LOCK_FILE_PATH = ProjectPath.getTecreationsPath() + "JavaCompiler.lock";
    public static LockFile lockFile = null;

    public static final String COMPILER_PATHS = "compiler.paths";
    public static final String START_OPEN = "start.open";
    public static final String RUNNING = "running";
    public static final String SHUTDOWN = "shutdown";

    static Properties properties = new Properties(ProjectPath.getTecPropsPath() + "properties" + File.separator + "JavaCompiler.properties");
    TFrame pathsWindow;
    TFrame outputWindow;
    TFrame statusWindow;

    List<JavaCompilerThread> compilers = new ArrayList<>();
    JButton add = new JButton("Add");
    JButton remove = new JButton("Remove");
    DefaultListModel<String> pathsModel = new DefaultListModel<String>();
    JList<String> pathsList = new JList<String>(pathsModel);
    JTextArea outputText = new JTextArea(40, 80);
    DefaultListModel<String> statusModel = new DefaultListModel<String>();
    JList<String> statusList = new JList<String>(statusModel);
    JPanel buttons = new JPanel(false);
    JButton clear = new JButton("Clear");

    public static boolean debug = false;
    public static boolean trace = false;
    Timer shutdownTimer;
    
    public JavaCompiler() {
        properties.set(SHUTDOWN,false);
        properties.set(RUNNING,true);
        pathsWindow = new TFrame(properties, "Paths");
        pathsWindow.setName("Paths");
        pathsWindow.setTitle("JavaCompiler: Paths");
        setupPathsGUI();
        if (standalone) pathsWindow.setExitOnClose(true);
        outputWindow = new TFrame(properties, "Output");
        outputWindow.setName("Output");
        outputWindow.setTitle("JavaCompiler: Output");
        setupOutputGUI();
        if (standalone) outputWindow.setExitOnClose(true);
        statusWindow = new TFrame(properties, "Status");
        statusWindow.setName("Status");
        statusWindow.setTitle("JavaCompiler: Status");
        setupStatusGUI();
        
        // add PROJECTS_HOME to compile paths
        String ppProps = ProjectPath.getTecPropsPath() + "ProjectPath.properties";
        Properties ppProperties = new Properties(ppProps);
        JavaCompilerThread projectsCompiler = new JavaCompilerThread(this);
        projectsCompiler.start(ppProperties.get("PROJECTS_HOME"));
        compilers.add(projectsCompiler);
        
        // get the user-added paths to comnpile
        List<String> targets = properties.getList(COMPILER_PATHS);
        if (targets.equals(null)) {
            properties.set(COMPILER_PATHS,"");
        } 
        for (int i = 0; i < targets.size(); i++) {
            pathsModel.addElement(targets.get(i));
            JavaCompilerThread compiler = new JavaCompilerThread(this);
            compiler.start(targets.get(i));
            compilers.add(compiler);
        } 
  
        // check if should "Start Open"
        Boolean startOpen = properties.getBoolean(START_OPEN);
        if (startOpen == null) {
            startOpen = false;
            properties.set(START_OPEN, startOpen);
        }
        Rectangle screenSize = Platform.getDesktopSize();
        // always open status
        statusWindow.setVisible(true);
        if (startOpen) {
            pathsWindow.setVisible(true);
            outputWindow.setVisible(true);
            statusWindow.toFront(); 
        } else {
            // we always open status, so if not StartOpen, open to back
            statusWindow.toBack();
        }
        this.startOpen.setState(startOpen);
        
        // set the initial state for paths and output windows
        if (properties.wasCreated()) {
            pathsWindow.setSize(640, 480); 
            outputWindow.setSize(640, 480);
            int x = screenSize.width - 640;
            int outputY = (screenSize.height - 480) / 2;
            pathsWindow.setLocation(x, 0);
            outputWindow.setLocation(x, outputY);
        } 
        // always set the status window. We let users change it but we always start at bottomRight X 640x480
        // by making the bottom and right of the status window visible, we can see if it is working and if not
        // what the last update was. This means our windows need to only extend so far on the destop, we need
        // about a 32x32 free space at the bottom right of the desktop (above or left of any bars) for this to work.
        statusWindow.setSize(640,480);
        int statusY = screenSize.height - 480 - buttons.getSize().height;
        statusWindow.setLocation(screenSize.width - 640,statusY); 
   
        setupSystemTray(); // make our operations available to the user
        
        shutdownTimer = new Timer(1000,this);
        shutdownTimer.start();
    } 
 
    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == paths) {
            pathsWindow.setVisible(true);
            pathsWindow.toFront();
        } else if (e.getSource() == output) {
            outputWindow.setVisible(true);
            outputWindow.toFront();
        } else if (e.getSource() == status) {
            statusWindow.setVisible(true);
            statusWindow.toFront();
        } else if (e.getSource() == exit) {
            exitProgram();
            System.exit(0);
        } else if (e.getSource() == add) {
            String path = Platform.requestDirectory(null, new File(ProjectPath.getDocumentsPath()), "Select a project dir...");
            if (path != null) {
                addCompilePath(path);
            }
        } else if (e.getSource() == remove) {
            int index = pathsList.getSelectedIndex();
            if (index != -1) {
                compilers.get(index).stopRunning();
                compilers.remove(index);
                pathsModel.removeElementAt(index);
                properties.set(COMPILER_PATHS, pathsModel.elements());
            }
        } else if (e.getSource() == clear) {
            statusModel.removeAllElements();
        } else if (e.getSource() == shutdownTimer) {
            properties.read(false);
            if (properties.getBoolean(SHUTDOWN)) exitSystemAfterShutdown();
        }
    } 

    public void addCompilePath(String path) {
        JavaCompilerThread compiler = new JavaCompilerThread(this);
        compiler.start(path);
        compilers.add(compiler);
        pathsModel.addElement(path);
        properties.set(COMPILER_PATHS, pathsModel.elements());

    } 

    public void addElement(String s) {
        statusModel.addElement(s);
        statusList.ensureIndexIsVisible(statusModel.getSize() - 1);
        statusList.setSelectedIndex(statusModel.getSize() - 1);
    }

    public static void createAndShowGUI() {
        if (trace) System.out.println("createAndShowGUI()");
        SwingUtilities.invokeLater(() -> {
            instance = new JavaCompiler();
            instance.setStatusVisible(true); // always open status
        });
    }
    
    public void exitProgram() {
        for (int i = compilers.size() - 1; i >= 0; i--) {
            compilers.get(i).stopRunning();
            compilers.remove(i);
        }
        while (new File(LOCK_FILE_PATH).exists()) {
            synchronized (lock) {
                lockFile.unlock();
            } 
            new File(LOCK_FILE_PATH).delete(true);
            if (new File(LOCK_FILE_PATH).exists()) {
                System.out.println("Re-trying deletion in (3s)...");
                Platform.sleep(3000);
            }
        }
        pathsWindow.dispose();
        outputWindow.dispose();
        statusWindow.dispose();
        properties.set(RUNNING,false);
        instance = null;
        if (!java.awt.SystemTray.isSupported()) {
            System.out.println("SystemTray is not supported");
            return;
        }
        tray.remove(trayIcon);
        System.gc();
    }

    public void exitSystemAfterShutdown() {
        properties.set(SHUTDOWN,false);
        exitProgram();
        Integer pid = properties.getIntOrZero("pid");
        if (pid != null) {
            if (pid > 0) {
                String cmd = "taskkill /F /PID " + pid;
                if (Platform.isNix()) {
                    cmd = "sudo kill -9 " + pid;
                }
                new SystemTool().runAndGet(cmd);
            } else {
                System.exit(0);
            }
        }
    } 

    public static File getLockFile() {
        return new File(JavaCompiler.LOCK_FILE_PATH);
    }
    
    public TFrame getOutput() { return outputWindow; }
    
    public JTextArea getOutputText() {
        return outputText;
    }

    public boolean getOutputVisible() { 
        return outputWindow.isVisible();
    }
    
    public TFrame getPaths() { return pathsWindow; }
    
    public boolean getPathsVisible() {
        return pathsWindow.isVisible();
    }
    
    public int getPID() { 
        properties.read(false);
        return properties.getInt("pid"); 
    }
     
    public static String getRuntimePath() { return ProjectPath.getRuntimePath(JavaCompiler.class.getProtectionDomain()); }
    
    public static Properties getProperties() {
        return properties;
    }

    public TFrame getStatus() { return statusWindow; }
    
    public boolean getStatusVisible() { 
        return statusWindow.isVisible();
    }
    
    public static boolean gotLockFile() {
        if (!new File(ProjectPath.getTecreationsPath()).exists()) {
            new File(ProjectPath.getTecreationsPath()).mkdirs();
        }
        String path = LOCK_FILE_PATH;
        if (!new File(path).exists()) {
            TextFile file = new TextFile(path);
            file.write();
            file.close();
        }
        lockFile = new LockFile(LOCK_FILE_PATH);
        lockFile.lock();
        return lockFile.isLocked();
    }

    public void itemStateChanged(ItemEvent e) {
        properties.set(START_OPEN, startOpen.getState());
    }

    public static void launch() {
        if (trace) System.out.println(JavaCompiler.class.getSimpleName() + ".launch()");
        if (lockFile == null && gotLockFile()) {
            createAndShowGUI();
        } else {
            System.err.println("Lockfile in use.");
        }
    } 

    public static void main(String[] args) {
        launch();
    }
   
    public void setErrorOutput(List<SystemToken> tokens) {
        outputText.setText("");
        for (int i = 0; i < tokens.size(); i++) {
            outputText.append(tokens.get(i).getText() + "\n");
        } 
    } 

    public void setOutputVisible(boolean state) {
        outputWindow.setVisible(state);
    }
    
    public void setPathsVisible(boolean state) {
        pathsWindow.setVisible(state);
    }

    public void setStatusVisible(boolean state) {
        statusWindow.setVisible(state);
    } 
     
//    public void setSystemErr(PrintStream err) {
//        System.setErr(err);
//    }

//    public void setSystemOut(PrintStream out) {
//        System.setOut(out);
//    }

    public void setupOutputGUI() {
        outputWindow.setLayout(new BorderLayout());
        outputText.setEditable(false);
        
        JScrollPane scrollPane = new JScrollPane(outputText, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        outputWindow.add(scrollPane, BorderLayout.CENTER);
        outputWindow.validate();
        outputWindow.setDefaultCloseOperation(JFrame.HIDE_ON_CLOSE);
    }

    public void setupPathsGUI() {
        pathsWindow.setLayout(new BorderLayout());

        JPanel buttons = new JPanel(new FlowLayout(FlowLayout.LEFT));
        buttons.add(add);
        buttons.add(remove);
        add.addActionListener(this);
        remove.addActionListener(this);
        pathsWindow.add(buttons, BorderLayout.NORTH);
 
        JScrollPane scrollPane = new JScrollPane(pathsList, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        pathsWindow.add(scrollPane, BorderLayout.CENTER);
        pathsWindow.validate();
        pathsWindow.setDefaultCloseOperation(JFrame.HIDE_ON_CLOSE);
    } 
  
    public void setupStatusGUI() {
        statusWindow.setLayout(new BorderLayout());

        JScrollPane scrollPane = new JScrollPane(statusList, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        statusWindow.add(scrollPane, BorderLayout.CENTER);
        buttons.add(clear);
        clear.addActionListener(this);
        statusWindow.add(buttons, BorderLayout.SOUTH);
        statusWindow.validate();
        statusWindow.setDefaultCloseOperation(JFrame.HIDE_ON_CLOSE);
    }
    
    public void setupSystemTray() {
        if (!java.awt.SystemTray.isSupported()) {
            System.out.println("SystemTray is not supported");
            return;
        }
        try {
            tray.add(trayIcon);
        } catch (AWTException e) {
            System.out.println("TrayIcon could not be added.");
        }
        popup.add(paths);
        popup.add(output);
        popup.add(status); 
        popup.addSeparator();
        popup.add(startOpen);
        popup.addSeparator();
        popup.add(exit);
        paths.addActionListener(this);
        output.addActionListener(this);
        status.addActionListener(this);
        startOpen.addItemListener(this);
        exit.addActionListener(this);
        trayIcon.setPopupMenu(popup);
    }
}
