package ca.tecreations.apps.editors;

import ca.tecreations.Color;
import ca.tecreations.Point;
import ca.tecreations.TextToken;
import ca.tecreations.components.Movable;
import ca.tecreations.components.TextStatus;
import ca.tecreations.interfaces.HasCursor;
import ca.tecreations.text.Cursor;
import ca.tecreations.text.LineOfTextTokenPainter;
import ca.tecreations.text.TextPoints;
import ca.tecreations.text.TextTokenPainter;

import java.awt.Graphics;
import java.awt.event.*;
import java.util.ArrayList;
import java.util.List;

import javax.swing.*;
import java.awt.event.KeyEvent;

/**
 *
 * @author tim
 */
public class TextEditPanel extends Movable implements
        HasCursor, KeyListener,
        MouseListener, MouseMotionListener, MouseWheelListener {
    public static boolean debug = true;
    TextEdit textEdit;
    int vsbHeight; 
    int hsbWidth;  
    LineNumbers lineNumbers;
    LineLocator lineLocator;
    TextStatus status = null;
    public static final int BETWEEN = 2;
    JScrollPane scroller;

    static TextPoints points = ca.tecreations.TecData.CODE_POINTS;
    static {
        points.setMonospaced(true);
    }
    
    int maxWidth = 0;
    
    //List<List<TextTokenPainter>> lines = new ArrayList<>();
    //List<TextTokenPainter> line = new ArrayList<>();
    //TextTokenPainter current = null;
    
    Color lineBg = Colors.currentLineBG;
    Color textColor = Colors.midGray;
    Color panelBG = Colors.panelBG;
    
    List<LineOfTextTokenPainter> lines = new ArrayList<>();
    
    int lineIndex = 0;
    int colNum = 0;
    
    Action doEnter;
    Action doSpace;

    private boolean isBackspace = false;
    private boolean isDelete = false;
    
    int dubWidth;
    
    Cursor cursorTimer = new Cursor(this,500);;
    Color cursorColor = Color.BLACK;
    boolean isInsert = true;
    int cursorX;
    int cursorY;
    
    int highlighted = 0;
    
    int selStartLine = 0;
    int selStartCol = 0;
       
    int lastLine = 0;
    int lastCol = 0;
    
    public TextEditPanel(TextEdit textEdit) {
        super(Movable.NONE);
        this.setDoubleBuffered(true);
        
        this.textEdit = textEdit;
        this.lineNumbers = textEdit.getLineNumbers();
        this.lineLocator = textEdit.getLineLocator();

        status = textEdit.getStatus();
        setupGUI();
        addKeyListener(this);
        addMouseListener(this);
        addMouseMotionListener(this);
        addMouseWheelListener(this);

        setBackground(Colors.panelBG);
        setForeground(Colors.midGray);
        setKeyMappings();
        dubWidth = points.getWidth('W');
    }

    public void addLine(LineOfTextTokenPainter painter) {
        painter.setBackground(getBackground());
        painter.setForeground(getForeground());
        if (debug) painter.setDemarcationColor(Color.BRIGHT_BLUE);
        lines.add(painter);
        textEdit.tec_redoLayout();
        repaint();
    }
    
    public void conformPanelSize() {
        maxWidth = 0;
        if (points.isMonospaced()) {
            int dubWidth = points.getWidth("W");
            for(int i = 0; i < lines.size();i++) {
                maxWidth = Math.max(maxWidth,lines.get(i).length() * (BETWEEN+dubWidth));
            }   
        } else {
            int tokensWidth = 0;
            for(int i = 0; i < lines.size();i++) {
                LineOfTextTokenPainter painters = lines.get(i);
                tokensWidth = painters.getTextWidth();
                maxWidth = Math.max(maxWidth,tokensWidth);
            }
        }
        //System.out.println("addLine: lnw: " + lnw + " tokens: " + tokensWidth + " max: " + maxWidth);
        int width = Math.max(maxWidth,scroller.getHorizontalScrollBar().getSize().width);
        int height = Math.max(lines.size() * points.getFontSize(),scroller.getVerticalScrollBar().getSize().height);
        setSize(width,height);
    }
    
    public void cursorHide() {
        Graphics g = getGraphics();
        if (g != null) {
            LineOfTextTokenPainter line = null;
            if (lines.size() > 0) line = lines.get(lineIndex);
            int lineHeight = getLineHeight();
            int y = lineIndex * lineHeight;
            Color reverse = Color.getReverse(lineBg);

            if (isInsert) {
                g.setColor(lineBg);
                g.fillRect(cursorX,cursorY,BETWEEN,lineHeight);
                //if (line != null && colNum < line.length()) {
                //    line.paintChar(g,y,colNum,reverse);
                //}
            } else {
                int half = (int)((double)BETWEEN / 2.0);
                if (points.isMonospaced()) {
                    g.setColor(lineBg);
                    g.fillRect(cursorX + half, cursorY,points.getWidth('W') + BETWEEN,lineHeight);
                    if (line != null && colNum < line.length()) {
                        line.paintChar(g,y,colNum,reverse);
                    }
                } else {
                    g.setColor(lineBg);
                    g.fillRect(cursorX + half, cursorY,lines.get(lineIndex).charWidthAt(colNum) + BETWEEN,lineHeight);
                    if (line != null && colNum < line.length()) {
                        line.paintChar(g,y,colNum,reverse);
                    }
                }
            }
        }
    }

    public void cursorMove(int lineIndex, int colNum) {
        this.lineIndex = lineIndex;
        this.colNum = colNum;
        this.lastLine = lineIndex;
        this.lastCol = colNum;
        LineOfTextTokenPainter line = lines.get(lineIndex);
        status.setLine(lineIndex + 1);
        status.setCol(this.colNum);
        cursorY = lineIndex * getLineHeight();
        boolean isEndOfLine = colNum >= line.getText().length();
        if (isEndOfLine) cursorX = line.getTextWidth();
        else cursorX = line.getColumnX(colNum);
        repaint();
    }

    public void cursorShow() {
        Graphics g = getGraphics();
        if (g != null) {
            LineOfTextTokenPainter line = null;
            if (lines.size() > 0) line = lines.get(lineIndex);
            int lineHeight = getLineHeight();
            int y = lineIndex * lineHeight;
            if (colNum > line.length()) colNum = line.length();
            
            Color reverse = Color.getReverse(lineBg);
            if (isInsert) {
                g.setColor(cursorColor);
                g.fillRect(cursorX,cursorY,BETWEEN,lineHeight);
                //if (line != null && colNum < line.length()) {
                //    line.paintChar(g,y,colNum,reverse);
                //}
            } else {
                int half = (int)((double)BETWEEN / 2.0);
                if (points.isMonospaced()) {
                    g.setColor(lineBg);
                    g.drawLine(cursorX,cursorY,1,lineHeight);
                    g.setColor(cursorColor);
                    g.fillRect(cursorX + half, cursorY,points.getWidth('W') + BETWEEN,lineHeight);
                    if (line != null && colNum < line.length()) {
                        line.paintChar(g,y,colNum,reverse);
                    }
                } else {
                    g.setColor(lineBg);
                    g.drawLine(cursorX,cursorY,1,lineHeight);
                    g.setColor(cursorColor);
                    g.fillRect(cursorX + half, cursorY,lines.get(lineIndex).charWidthAt(colNum) + BETWEEN,lineHeight);
                    if (line != null && colNum < line.length()) {
                        line.paintChar(g,y,colNum,reverse);
                    }
                }
            }
        }
    }
    
    public void demarcate() {
        Color c;
        if (points.isMonospaced()) {
            c = new Color(210,172,204);
        } else {
            c = Color.red; 
        }
        for(int i = 0; i < lines.size();i++) {
            lines.get(i).setDemarcationColor(c);
        }
    }

    public void deselect() {
        for(int i = 0;i < lines.size();i++) {
            lines.get(i).deselect();
        }
        repaint();
    }
    
    public void deselect1FromMax() {
        
    }
    
    public void deselect1FromMin() {
        
    }
    
    public void doTest() {
        lines.get(0).selectFrom(12);
    }
    
    public void fontSizeDown() {
        int newSize = points.getSize() - 1;
        if (newSize > 0) {
            points = points.getSized(newSize);
            lineNumbers.fontSizeDown();
            maxWidth = 0;
            LineOfTextTokenPainter line;
            for(int i = 0; i < lines.size();i++) {
                line = lines.get(i);
                line.setPoints(points);
            }
        }
    }
    
    public void fontSizeUp() {
        int newSize = points.getSize() + 1;
        points = points.getSized(newSize);
        lineNumbers.fontSizeUp();
        maxWidth = 0;
        LineOfTextTokenPainter line;
        for(int i = 0; i < lines.size();i++) {
            line = lines.get(i);
            line.setPoints(points);
        }
    }
    
    public int getCursorX() { return cursorX; }
    
    public JScrollBar getHorizontalScrollBar() { return textEdit.hsb; }
    
    public LineOfTextTokenPainter getLastLine() {
        return lines.get(lines.size() - 1);
    }
    
    public LineOfTextTokenPainter getLine(int index) {
        return lines.get(index);
    }
    
    public int getLineHeight() {
        return points.getFontSize();
    }
    
    public int getLineIndex(int y) {
        int lineHeight = getLineHeight();
        if (y < lineHeight) return 0;
        else {
            int lineNum = (int)((double)y / (double)lineHeight);
            if (lineNum > lines.size()) return lines.size() - 1;
            else return lineNum;
        }
    }
    
    public List<LineOfTextTokenPainter> getLines() {
        return lines;
    }
    
    public int getLinesSize() {
        return lines.size();
    }
    
    public int getMaxColumn(int lineIndex) {
        return lines.get(lineIndex).length();
    }
    
    public int getMaxLines() {
        return getSize().height / points.getFontSize();
    }

    public TextPoints getPoints() { return points; }

    public JScrollPane getScroller() { return scroller; }
    
    public TextEdit getTextEdit() { return textEdit; }
    
    public JScrollBar getVerticalScrollBar() { return textEdit.vsb; }

    public int getVisibleLines() { return (int)((double)vsbHeight / (double)points.getFontSize()); }
    
    public int getWidthForColumn(int colNum) {
        LineOfTextTokenPainter line = lines.get(lineIndex);
        return line.charWidthAt(colNum);
    }
   
    public int getYOffset() {
        int lineHeight = getLineHeight();
        int absY = Math.abs(getLocation().y);
        int offset = absY - (absY - lineHeight);
        if (offset == lineHeight) return 0;
        else return offset;
    }
    
/*    public void goTo(int line, int col) {
        if (line <= lines.size()) {
            int y = (lineIndex - 1) * points.getFontSize();
            int scrollHeight = scroller.getVerticalScrollBar().getSize().height;
            int maxLine = lines.size() - (scrollHeight / points.getFontSize());
            if (lineIndex < maxLine) {
                setLocation(getLocation().x,-y);
            } else {
                setLocation(getLocation().x,-(getSize().height - scrollHeight));
            }
            cursorMove(line - 1,col);
            repaint();
        }
    }
*/    
    public void goToEnd() {
        System.out.println("goToEnd");
        deselect();
        colNum = lines.get(lines.size() - 1).length();
        cursorMove(lines.size() - 1,colNum);
        setLocationMaxY();
    }
    
    public void goToEndOfLine() {
        System.out.println("goToEndOfLine");
        cursorMove(lineIndex,lines.get(lineIndex).length());
    }
    
    public void goToStart() {
        System.out.println("goToStart");
        deselect();
        cursorMove(0,colNum);
        setLocation(getLocation().x,0);
    }
    
    public void goToStartOfLine() {
        System.out.println("goToStartOfLine");
        cursorMove(lineIndex,0);
    }

    public int getCursorY() {
        return (lineIndex + 1) * points.getFontSize();
    }
    
    public void highlight(int index) {
        lines.get(index).setBackground(lineBg);
    }
    
    public void insertLine(int index, LineOfTextTokenPainter painter) {
        painter.setBackground(getBackground());
        painter.setForeground(getForeground());
        if (debug) painter.setDemarcationColor(Color.BRIGHT_BLUE);
        lines.add(index,painter);
        textEdit.tec_redoLayout();
        repaint();
    }
    
    public void insertMeasurementLine() {
        String eighty = "12345678911234567892123456789312345678941234567895123456789612345678971234567898";
        List<TextTokenPainter> painters = new ArrayList<>();
        painters.add(new TextTokenPainter(points,new TextToken(eighty),getForeground()));
        LineOfTextTokenPainter line = new LineOfTextTokenPainter(painters);
        insertLine(0,line);
    }
    
    public boolean isCursorHigher() {
        int y = Math.abs(getLocation().y);
        return getCursorY() - points.getFontSize() < y;
    }
    
    public boolean isCursorLower() {
        int y = Math.abs(getLocation().y) + vsbHeight;
        return getCursorY() >= y;
    }
    
    public boolean isLower() {
        return (lineIndex >= selStartLine  | 
                lineIndex == selStartLine && colNum > selStartCol);
    }
    
    public boolean isUpper() {
        return (lineIndex < selStartLine & (lineIndex == selStartLine && colNum < selStartCol));
    }
    
    public void jumpLeft() {
        System.out.println("Jump Left");
        deselect();
    }
    
    public void jumpRight() {
        System.out.println("Jump Right");
        deselect();
    }
    
    public void keyPressed(KeyEvent ke) {
        if (ke.getKeyCode() == KeyEvent.VK_BACK_SPACE) { 
            isBackspace = true;
        }
        boolean isAlt = ke.isAltDown();
        boolean isCtrl = ke.isControlDown();
        boolean isShift = ke.isShiftDown();
        if (ke.getKeyCode() == KeyEvent.VK_LEFT) {
            ke.consume();
            if (!isCtrl && isShift) shiftLeft();
            else if (isCtrl && !isShift) jumpLeft();
            else if (isCtrl && isShift) shiftJumpLeft();
            else moveLeft();
        } else if (ke.getKeyCode() == KeyEvent.VK_RIGHT) {
            ke.consume();
            if (!isCtrl && isShift) shiftRight();
            else if (isCtrl && !isShift) jumpRight();
            else if (isCtrl && isShift) shiftJumpRight();
            else moveRight();
        } else if (ke.getKeyCode() == KeyEvent.VK_UP) {
            ke.consume();
            if (!isCtrl && isShift) shiftLineUp();
            else if (isCtrl && !isShift) scrollLineUp();
            else if (isCtrl && isShift)shiftScrollLineUp();
            else lineUp();
        } else if (ke.getKeyCode() == KeyEvent.VK_DOWN) {
            ke.consume();
            if (!isCtrl && isShift) shiftLineDown();
            else if (isCtrl && !isShift) scrollLineDown();
            else if (isCtrl && isShift) shiftScrollLineDown();
            else lineDown();
        } else if (ke.getKeyCode() == KeyEvent.VK_PAGE_UP) {
            ke.consume();
            if (!isCtrl && !isShift) pageUp();
            else if (!isCtrl && isShift) shiftPageUp();
            else if (isCtrl && isShift) shiftToStart();
            else if (isCtrl && !isShift) scrollPageUp();
        } else if (ke.getKeyCode() == KeyEvent.VK_PAGE_DOWN) {
            ke.consume();
            if (!isCtrl && !isShift) pageDown();
            else if (!isCtrl && isShift) shiftPageDown();
            else if (isCtrl && isShift) shiftToEnd();
            else if (isCtrl && !isShift) scrollPageDown();
        } else if (ke.getKeyCode() == KeyEvent.VK_INSERT) {
            isInsert = !isInsert;
        } else if (ke.getKeyCode() == KeyEvent.VK_HOME) {
            if (isCtrl && !isShift) scrollToStart();
            else if (isCtrl && isShift) shiftToStart();
            else if (!isCtrl && isShift) shiftToStartOfLine();
            else goToStartOfLine();
        } else if (ke.getKeyCode() == KeyEvent.VK_END) {
            if (isCtrl && !isShift) scrollToEnd();
            else if (!isCtrl && isShift) shiftToEndOfLine();
            else goToEndOfLine();
        } 
        textEdit.repaint();
    }

    public void keyReleased(KeyEvent ke) {
         if (ke.getKeyCode() == KeyEvent.VK_BACK_SPACE) { 
             isBackspace = false;
         }
    }
    
    public void keyTyped(KeyEvent ke) {
        if (!isBackspace) {
            char key = ke.getKeyChar();
          //  int start = getStartColIndex();
          //  int index = getTokenIndex(start);
          //  if (isInsert) {
          //      current.insert(index, key);
          //  } else {
          //      current.replace(index,key);
          //  }
            cursorHide();
            colNum++;
            status.setCol(colNum);
            cursorMove(lineIndex,colNum);
            cursorShow();
            repaint();
        }
    }
    
    public void lineDown() {
        System.out.println("lineDown");
        unhighlight(lineIndex);
        deselect();
        if (lineIndex + 1 < lines.size()) lineIndex++;
        colNum = Math.min(colNum,lines.get(lineIndex).length());
        cursorMove(lineIndex,colNum);
        selStartLine = lineIndex;
        selStartCol = colNum;
        if (isCursorLower()) scrollLineDown();
    }
    
    public void lineUp() {
        System.out.println("lineUp");
        deselect();
        unhighlight(lineIndex);
        if (lineIndex - 1 >= 0) lineIndex--;
        colNum = Math.min(colNum,lines.get(lineIndex).length());
        cursorMove(lineIndex,colNum);
        selStartLine = lineIndex;
        selStartCol = colNum;
        if (isCursorHigher()) scrollLineUp();
    }
  
    public static void main(String[] args) {
        TextEdit.launch();
    }
     
    public void mouseClicked(MouseEvent e) {
        
    } 

    public void mouseDragged(MouseEvent e) {
        int oldLineIndex = lineIndex;
        unhighlight(oldLineIndex);
        cursorHide();
        if (e.getY() < points.getFontSize()) {
            lineIndex = 0;
        } else {
            lineIndex = getLineIndex(e.getY());
        }
        LineOfTextTokenPainter line = lines.get(lineIndex);
        int targetColNum = line.getColumn(e.getX());
        if (targetColNum >= 0) {
            colNum = targetColNum;
            cursorMove(lineIndex,colNum);
            selectSelection();
            cursorShow();
            System.err.println("md.Selection: " + selStartLine + "," + selStartCol + " to: " + lineIndex + "," + colNum);
        }
    }

    public void mouseEntered(MouseEvent e) {
        setCursor(java.awt.Cursor.getPredefinedCursor(java.awt.Cursor.TEXT_CURSOR));
    }

    public void mouseExited(MouseEvent e) {
        setCursor(java.awt.Cursor.getPredefinedCursor(java.awt.Cursor.DEFAULT_CURSOR));
    }
    
    public void mouseMoved(MouseEvent e) {
    }
    
    public void mousePressed(MouseEvent e) {
        if (e.isControlDown()) {
            doTest();
            return;
        }
        int oldLineIndex = lineIndex;
        lineIndex = getLineIndex(e.getY());
        if (lineIndex < lines.size()) {
            cursorHide();
            unhighlight(oldLineIndex);
            LineOfTextTokenPainter line = lines.get(lineIndex);
            colNum = line.getColumn(e.getX());
            cursorMove(lineIndex,colNum); // these are both 0 based values
            
            if (!e.isShiftDown()) {
                deselect();
                selStartLine = lineIndex;
                selStartCol = colNum;
            } else {
                lastLine = lineIndex;
                lastCol = colNum;
                selectSelection();
            }
            cursorShow();
        } 
        System.out.println("mp.Selection: " + selStartLine + "," + selStartCol + " to: " + lineIndex + "," + colNum);
    }
    
    public void mouseReleased(MouseEvent e) {
    }

    public void mouseWheelMoved(MouseWheelEvent e) {
        JScrollBar hsb = textEdit.hsb;
        JScrollBar vsb = textEdit.vsb;

        if (e.isControlDown() && !e.isAltDown() && !e.isShiftDown()) {
            if (e.getWheelRotation() >= 1) {
                fontSizeDown();
            } else {
                fontSizeUp();
            }
            conformPanelSize();
            textEdit.tec_redoLayout();
            textEdit.repaint();
            //System.out.println("CTRL-WHEEL: " + e.getWheelRotation());
        } else if (e.isAltDown()) {
            if (e.getWheelRotation() >= 1) {
                hsb.setValue(hsb.getValue() + hsb.getUnitIncrement());
            } else {
                hsb.setValue(hsb.getValue() - hsb.getUnitIncrement());
            }
            repaint();
        } else if (!e.isControlDown() && !e.isAltDown() && !e.isShiftDown()) {
            if (e.getWheelRotation() >= 1) {
                vsb.setValue(vsb.getValue() + vsb.getUnitIncrement());
                int absPanelY = Math.abs(getLocation().y);
                if (absPanelY + textEdit.vsb.getSize().height > getSize().height - points.getFontSize()) {
                    vsb.setValue(vsb.getMaximum());
                }
            } else {
                if (vsb.getValue() - vsb.getUnitIncrement() >= 0) {
                    vsb.setValue(vsb.getValue() - vsb.getUnitIncrement());
                } else {
                    vsb.setValue(0);
                }
            }
            textEdit.repaint();
        }
    }

    public void moveLeft() {
        deselect();
        System.out.println("MoveLeft");
        if (colNum > 0) {
            cursorMove(lineIndex,--colNum);
        } else {
            lineIndex--;
            colNum = lines.get(lineIndex).length();
            cursorMove(lineIndex,colNum);
        }
        selStartLine = lineIndex;
        selStartCol = colNum;
    }
    
    public void moveRight() {
        deselect();
        System.out.println("MoveRight");
        if (colNum < lines.get(lineIndex).length()) {
            cursorMove(lineIndex,++colNum);
        } else {
            if (lineIndex + 1 < lines.size()) {
                lineIndex++;
                colNum = 0;
                cursorMove(lineIndex, colNum);
            }
        }
        selStartLine = lineIndex;
        selStartCol = colNum;
    }
    

    public void pageDown() {
        System.out.println("PageDown");
        unhighlight(lineIndex);
        if (lineIndex + getVisibleLines() < lines.size() - 1) {
            lineIndex += getVisibleLines();
        } else {
            lineIndex = lines.size() - 1;
        }
        int y = Math.abs(getLocation().y) + vsbHeight;
        if (y > getSize().height - vsbHeight) y = getSize().height - vsbHeight;
        setLocation(getLocation().x,-y);
        cursorMove(lineIndex,colNum);
        int minY = lineIndex * points.getFontSize();
        System.out.println("MinY: " + minY + " loc.y: " + getLocation().y);
        if (minY < Math.abs(getLocation().y)) {
            setLocation(getLocation().x,-minY);
        }
    }
    
    public void pageUp() {
        System.out.println("PageUp");
        unhighlight(lineIndex);
        if (lineIndex - getVisibleLines() > 0) {
            lineIndex -= getVisibleLines();
        } else {
            lineIndex = 0;
        }
        int y = Math.abs(getLocation().y) - vsbHeight;
        if (y < 0) y = 0;
        setLocation(getLocation().x,-y);
        cursorMove(lineIndex,colNum);
        int maxY = (lineIndex + 2) * points.getFontSize();
        if (maxY > (Math.abs(getLocation().y) + vsbHeight)) {
            System.err.println("adjust");
        System.out.println("MaxY: " + maxY + " loc.y: " + getLocation().y);
            setLocation(getLocation().x,-(maxY - vsbHeight));
        }
    }

    @Override
    public void paint(Graphics g) {
        Color oldColor = new Color(g.getColor());
        //super.paint(g);
        
        g.setColor(getBackground());
        int fontSize = points.getFontSize();
        int absX = Math.abs(getLocation().x);
        int absY = Math.abs(getLocation().y);
        int minX = absX;
        int maxX = minX + textEdit.hsb.getSize().width;
        int minY = absY - fontSize;
        int maxY = absY + textEdit.vsb.getSize().height + fontSize;

        int y = getYOffset();
        // paint the text area background
        g.fillRect(minX, minY, maxX - minX,maxY - minY);
        
        // draw NetBeans-style 80 column line, only if monospaced
        int dubWidth = points.getWidth("W");
        int eighty = 80 * (BETWEEN + dubWidth);
        LineOfTextTokenPainter painter = null;
        for (int i = 0; i < lines.size(); i++) {
            if (y >= minY && y <= maxY) {
                painter = lines.get(i);
                if (i == lineIndex) {
                    g.setColor(lineBg);
                    g.fillRect(0, y, getSize().width,fontSize);
                    painter.setBackground(lineBg);
                    painter.paintAt(g,BETWEEN,y);
                } else {
                    g.setColor(getBackground());
                    //painter.setBackground(getBackground());
                    //painter.setForeground(getForeground());
                    painter.paintAt(g,BETWEEN,y);
                }
                if (points.isMonospaced()) {
                    g.setColor(new Color(210,173,204));
                    g.drawLine(eighty,y,eighty,y+points.getFontSize());
                }
                if (painter.getDemarcationColor() != null) {
                    g.setColor(painter.getDemarcationColor());
                    int demarcX = painter.getPaintingWidth() - BETWEEN - 1; // -1 for pen
                    g.drawLine(demarcX,y,demarcX,y+points.getFontSize());
                }
            }
            y += fontSize;
        }
        g.setColor(oldColor);
    }

    public void scrollLineDown() {
        System.out.println("scrollLineDown");
        int y = Math.abs(getLocation().y);
        if (y < getSize().height - vsbHeight) {
            y = -(Math.abs(getLocation().y) + points.getFontSize());
        } else {
            y = -(getSize().height - vsbHeight);
        }
        setLocation(getLocation().x, y);
    }
    
    public void scrollLineUp() {
        System.out.println("scrollLineUp");
        int y = Math.abs(getLocation().y);
        if (y - points.getFontSize() >= 0) {
            y = -(Math.abs(getLocation().y) - points.getFontSize());
        } else {
            y = 0;
        }
        setLocation(getLocation().x, y);
    }
    
    public void scrollPageDown() {
        System.out.println("scrollPageDown");
        int newY = Math.abs(getLocation().y) + textEdit.vsb.getSize().height;
        if (newY >= getSize().height) {
            newY = -(getSize().height - textEdit.vsb.getSize().height);
        } else {
            newY = -(Math.abs(getLocation().y) + textEdit.vsb.getSize().height);
        }
        setLocation(getLocation().x,newY);
    }
    
    public void scrollPageUp() {
        System.out.println("scrollPageUp");
        int newY = Math.abs(getLocation().y) + textEdit.vsb.getSize().height;
        if (newY <= 0) {
            newY = 0;
        } else {
            newY = -(Math.abs(getLocation().y) - textEdit.vsb.getSize().height);
        }
        setLocation(getLocation().x,newY);
        
    }

    public void scrollToBottom() {
        System.out.println("scrollToBottom");
        setLocation(getLocation().x,-(getSize().height - vsbHeight));
    }
    
    public void scrollToEnd() {
        System.out.println("scrollToEnd");
        setLocationMaxY();
    }

    public void scrollToMiddle() {
        System.out.println("scrollToMiddle");
        int y = lineIndex * points.getFontSize();
        setLocation(getLocation().x,-(y - vsbHeight / 2));
    }
    
    public void scrollToStart() { 
        System.out.println("scrollToStart");
        setLocation(0,0);
    }
    
    public void scrollToTop() {
        setLocation(getLocation().x,0);
    }
    
    public void selectLine(int lineIndex) {
        this.lineIndex = lineIndex;
        LineOfTextTokenPainter line = lines.get(lineIndex);
        colNum = Math.min(line.length(),colNum);
        cursorMove(lineIndex,colNum);
    }
    
    public void selectLineWithY(int targetY) {
        int lineIndex = 0;
        int lineY = 0;
        int fontSize = points.getFontSize();
        while (lineY <= targetY) {
            lineY += fontSize;
            lineIndex++;
        }
        int maxY = getSize().height - textEdit.vsb.getSize().height;
        if (targetY < maxY) {
            lineY -= fontSize; // the WHILE test is for the start of the next line
            setLocation(0,-(lineY));
            selectLine(lineIndex);
        } else {
            setLocation(0,-maxY);
            selectLine(lines.size() - 1);
        }
    }
    
    public void setKeyMappings() {
        getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER,0),"enter");
        getActionMap().put("enter",  new AbstractAction() {
            public void actionPerformed(ActionEvent e) {
                // insert new line painter and move cursor
                if (lineIndex == 0) {
                    lines.add(lineIndex, new LineOfTextTokenPainter(points,
                                                new Color(getBackground()),
                                                new Color(getForeground()))
                                       );
                } else
                cursorMove(lineIndex + 1,0);
                conformPanelSize();
            }
        });
        
        getInputMap().put(KeyStroke.getKeyStroke(" "),"space");
        getActionMap().put("space",  new AbstractAction() {
            public void actionPerformed(ActionEvent e) {
                cursorMove(lineIndex,colNum++);
            }
        });
        
        getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_BACK_SPACE,0),"backspace");
        getActionMap().put("backspace",  new AbstractAction() {
            public void actionPerformed(ActionEvent e) {
                lines.get(lineIndex).backspace();
                repaint();
            }
        });
        
    }
    
    public void selectSelection() {
        int selLine1, selCol1, selLine2, selCol2;
        selLine1 = Math.min(lineIndex,selStartLine);
        if (selLine1 == lineIndex) {
            selCol1 = colNum;
            selLine2 = selStartLine;
            selCol2 = selStartCol;
        } else {
            selCol1 = selStartCol;
            selLine2 = lineIndex;
            selCol2 = colNum;
        }
        if (selLine1 == selLine2) {
            int low = Math.min(selCol1,selCol2);
            int high = Math.max(selCol1, selCol2) - 1;
            for(int i = 0; i < lines.size();i++) lines.get(i).deselect();
            lines.get(selLine1).selectInclusive(low,high);
            System.out.println("line: " + low + " through " + high);
        } else {
            for(int i = 0; i < selLine1;i++) lines.get(i).deselect();
            LineOfTextTokenPainter line = lines.get(selLine1);
            line.selectFrom(selCol1);
            for(int i = (selLine1 + 1);i < selLine2;i++) {
                lines.get(i).select();
            }
            line = lines.get(selLine2);
            line.selectTo(selCol2);
            for(int i = (selLine2 + 1); i < lines.size();i++) {
                lines.get(i).deselect();
            }
        }
        repaint();
        
    }
    
    public void setLastLine(LineOfTextTokenPainter line) {
        lines.set(lines.size() - 1, line);
        for(int i = 0; i < lines.size();i++) maxWidth += lines.get(i).getTextWidth();
        setSize(maxWidth, lines.size() * points.getFontSize());
    }

    public void setLines(List<LineOfTextTokenPainter> lines) {
        this.lines = lines;
        conformPanelSize();
    }
    
    public void setLocationMaxY() {
        int x = getLocation().x;
        int y = -(getSize().height - textEdit.vsb.getSize().height);
        setLocation(new Point(x,y));
    }

    public void setPainters(List<LineOfTextTokenPainter> lines) {
        this.lines = lines;
        repaint();
    }
    
    public void setScroller(JScrollPane scroller) {
        this.scroller = scroller;
    }
    
    public void setupGUI() {
        setBackground(Color.DARK_GREY);
        setFocusable(true);
    }
    
    public void shiftJumpLeft() {
        System.out.println("Shift Jump Left");
    }
    
    public void shiftJumpRight() {
        System.out.println("Shift Jump Right");
    }
    
    public void shiftLeft() {
        System.out.println("Shift Left");
        deselect();
        if (colNum > 0) {
            colNum--;
        } else {
            if (lineIndex > 0) {
                unhighlight(lineIndex);
                lineIndex--;
                colNum = lines.get(lineIndex).length();
           }
        }
        cursorMove(lineIndex,colNum);
        selectSelection();
    }

    public void shiftLineDown() {
        System.out.println("Shift Line Down");
        if (lineIndex < lines.size()) {
            unhighlight(lineIndex);
            deselect();
            lineIndex++;
            colNum = Math.min(colNum,lines.get(lineIndex).length());
            cursorMove(lineIndex,colNum);
            //ifLowerThanBottomThenScrollLineDown();
        }
        selectSelection();
    }
    
    public void shiftLineUp() {
        System.out.println("Shift Line Up");
        if (lineIndex > 0) {
            unhighlight(lineIndex);
            deselect();
            lineIndex--;
            colNum = Math.min(colNum,lines.get(lineIndex).length());
            cursorMove(lineIndex,colNum);
//            ifHigherThanTopThenScrollLineUp();
        }
        selectSelection();
    }
    
    public void shiftPageDown() {
        System.out.println("shiftPageDown");
        unhighlight(lineIndex);
        int visible = (int)((double)vsbHeight / (double)points.getFontSize());
        deselect();
        if (lineIndex + visible < lines.size()) {
            lineIndex += visible;
            colNum = Math.min(colNum,lines.get(lineIndex).length());
            setLocation(getLocation().x, -Math.abs(getLocation().y) + (visible * points.getFontSize()));
        } else {
            lineIndex = lines.size() - 1;
            colNum = lines.get(lineIndex).length();
            setLocation(getLocation().x,-(getSize().height - vsbHeight));
        }
        validatePanel();
        cursorMove(lineIndex,colNum);
        selectSelection();
    }
    
    public void shiftPageUp() {
        System.out.println("shiftPageUp");
        unhighlight(lineIndex);
        int visible = (int)((double)vsbHeight / (double)points.getFontSize());
        deselect();
        if (lineIndex - visible > 0) {
            lineIndex -= visible;
            colNum = Math.min(colNum,lines.get(lineIndex).length());
            setLocation(getLocation().x, -Math.abs(getLocation().y) - (visible * points.getFontSize()));
        } else {
            lineIndex = 0;
            colNum = 0;
            setLocation(getLocation().x,0);
        }
        validatePanel();
        cursorMove(lineIndex,colNum);
        selectSelection();
    }
    
    public void shiftRight() {
        System.out.println("Shift Right");
        deselect();
        if (colNum < lines.get(lineIndex).length()) {
            colNum++;
        } else {
            if (lineIndex < lines.size()) {
                unhighlight(lineIndex);
                lineIndex++;
                colNum = 0;
           }
        }
        cursorMove(lineIndex,colNum);
        selectSelection();
    }

    
    public void shiftScrollLineDown() { 
        System.out.println("shiftScrollLineDown");
        setLocation(getLocation().x,getLocation().y + points.getFontSize());
        validatePanel();
        if (isUpper()) {
            shiftLineDown();
        } else {
            shiftLineUp();
        }
        textEdit.repaint();
    }
    
    public void shiftScrollLineUp() { 
        System.out.println("shiftScrollLineUp");
        setLocation(getLocation().x,getLocation().y - points.getFontSize());
        validatePanel();
        if (isUpper()) {
            shiftLineUp();
        } else {
            shiftLineDown();
        }
        textEdit.repaint();
    }
    
    public void shiftToEnd() {
        System.out.println("shiftToEnd");
        unhighlight(lineIndex);
        lineIndex = lines.size() - 1;
        colNum = lines.get(lineIndex).length();
        cursorMove(lineIndex,colNum);
        selectSelection();
    }
    
    public void shiftToEndOfLine() {
        System.out.println("shiftToEndOfLine");
        deselect();
        colNum = lines.get(lineIndex).length();
        cursorMove(lineIndex,colNum);
        selectSelection();
    }
    
    public void shiftToStart() {
        System.out.println("shiftToStart");
        deselect();
        unhighlight(lineIndex);
        lineIndex = 0;
        colNum = 0;
        cursorMove(lineIndex,colNum);
        selectSelection();
    }

    public void shiftToStartOfLine() {
        System.out.println("shiftToStartOfLine");
        deselect();
        colNum = 0;
        cursorMove(lineIndex,colNum);
        selectSelection();
    }
    
    public void unhighlight(int index) {
        lines.get(index).setBackground(getBackground());
    }
    
    public void validatePanel() {
        System.out.println("validatePanel");
        int x = getLocation().x;
        int y = getLocation().y;
        if (getLocation().x > 0) x = 0;
        else if (getLocation().x + hsbWidth > getSize().width) x = getSize().width - hsbWidth;
        if (getLocation().y > 0) y = 0;
        else if (getLocation().y + vsbHeight > getSize().height) y = getSize().height - vsbHeight;
        setLocation(x,y);
    }
}
