package ca.tecreations.apps.draw;

import ca.tecreations.Color;
import ca.tecreations.ImageTool;
import ca.tecreations.Pixel;
import ca.tecreations.Point;
import ca.tecreations.TecData;
import ca.tecreations.TextToken;
import ca.tecreations.components.Movable;
import ca.tecreations.text.GUITextTokenPainter;
import ca.tecreations.text.TextPoints;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.List;

import javax.swing.*;

/**
 *
 * @author Tim
 */
public class DrawPanel extends Movable implements ActionListener, ItemListener,
        MouseListener, MouseMotionListener, MouseWheelListener {

    public static DrawPanel current = null;
    public static int numPanels = 0;
    public int idNum;
    Driver driver;

    List<List<Pixel>> strokes = new ArrayList<>();
    List<Pixel> stroke = null;

    Color outlineX = null;
    boolean guides = true;

    Color bg = Color.DEFAULT_SWING_BG;
    boolean identify = false;

    JPopupMenu popup;
    JCheckBoxMenuItem setIdentify = new JCheckBoxMenuItem("Identify");
    JMenuItem repaint = new JMenuItem("Repaint");

    List<List<DrawPanel>> panels;

    int lastX;
    int lastY;

    public int gridX = 0;
    public int gridY = 0;

    public DrawPanel(Driver driver, int w, int h) {
        super(Movable.NONE);
        this.driver = driver;
        setSize(w, h);

        idNum = numPanels;
        numPanels++;

        addMouseListener(this);
        addMouseMotionListener(this);
        addMouseWheelListener(this);
//        System.out.println("DrawPanel: " + idNum + " : " + this.toString());

        setIdentify.setSelected(identify);
        setIdentify.addItemListener(this);
        repaint.addActionListener(this);
    }

    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == repaint) {
            repaint();
        }
    }

    public void addToMyStroke(int x, int y) {
        Graphics g = getGraphics();
        g.setColor(Colors.PEN_COLOR);
        g.drawLine(x, y, x, y);
        stroke.add(new Pixel(Colors.PEN_COLOR, x, y));
    }

    public void doPainting(Graphics g) {
        //System.out.println("DrawPanel.doPainting()");
        //System.out.println("DrawPanel.doPainting(" + outlineX + ")");
        if (outlineX != null) {
            g.setColor(outlineX);
            g.drawRect(0, 0, getSize().width - 1, getSize().height - 1); // right? because the outline
            g.drawLine(0, 0, getSize().width, getSize().height); // top right to bottom left
            g.drawLine(0, getSize().height, getSize().width, 0); // bottom left to top right
        }
        if (guides) {
            // draw the guide lines, in black
            g.setColor(Color.black);
            int percent25 = (int) ((double) getSize().height / (double) 4);
            int half = (int) ((double) getSize().height / (double) 2);
            int percent75 = (int) ((double) getSize().height / (double) 4 * (double) 3);
            // upper -- solid
            g.drawLine(0, percent25, getSize().width, percent25);

            // middle -- dashed
            boolean draw = true;
            for (int x = 0; x < getSize().width; x += 10) {
                if (draw) {
                    g.drawLine(x, half, x + 10, half);
                }
                draw = !draw;
            }
            // lower -- solid
            g.drawLine(0, percent75, getSize().width, percent75);
        }

        Pixel pixel;
        List<Pixel> stroke;
        for (int i = 0; i < strokes.size(); i++) {
            stroke = strokes.get(i);
            for (int j = 0; j < stroke.size(); j++) {
                pixel = stroke.get(j);
                g.setColor(pixel.color);
                g.drawLine(pixel.x, pixel.y, pixel.x, pixel.y);
            }
        }

        if (identify) {
            TextPoints points = TecData.LG_CODE_POINTS;
            points = points.getSized(getSize().height);
            GUITextTokenPainter idPainter = new GUITextTokenPainter(points, new TextToken("" + idNum));
            int x = (int) (((double) getSize().width - (double) idPainter.getTextWidth()) / 2.0);
            int y = (int) (((double) getSize().height - (double) idPainter.getFontSize()) / 2.0);
            idPainter.paintAt(g, x, y);
            //System.out.println("Identifying(" + idNum + "): width: " + getSize().width + " height: " + getSize().height + " x: " + x + " y: " + y);
        }
    }

    public void endStroke() {
        stroke = null;
    }

    public Color getBackground() {
        return bg;
    }

    public DrawPanel getBottomRight() {
        List<List<DrawPanel>> panels = driver.getActual().getCurrentArea().getPanels();
        if (panels.size() == 0) {
            System.err.println("getBottomRight: panels.size == 0");
            return this;
        }
        List<DrawPanel> lastStrip = panels.get(panels.size() - 1);
        DrawPanel botRight = lastStrip.get(lastStrip.size() - 1);
        return botRight;
    }

    public Point getCoordinate() {
        return new Point(gridX, gridY);
    }

    public DrawPanel getEmpty() {
        DrawPanel panel = new DrawPanel(driver, getSize().width, getSize().height);
        panel.setIdentify(getIdentify());
        panel.setBackground(getBackground());
        panel.setOutlineX(getOutlineX());
        panel.setGuides(getGuides());
        return panel;
    }

    public boolean getGuides() {
        return guides;
    }

    public boolean getIdentify() {
        return identify;
    }

    public BufferedImage getImage() {
        BufferedImage img = ImageTool.getNewBufferedImage(getSize().width, getSize().height);
        Graphics g = img.getGraphics();
        paint(g);
        return img;
    }

    public String getMoveDirection(int dx, int dy) {
        if (dx < 0) {
            if (dy < 0) {
                return "NW";
            } else if (dy == 0) {
                return "W";
            } else { // dy > 0
                return "SW";
            }
        } else if (dx == 0) {
            if (dy < 0) {
                return "N";
            } else if (dy == 0) {
                return "STILL";
            } else { // dy > 0 {
                return "S";
            }
        } else { // dx > 0
            if (dy < 0) {
                return "NE";
            } else if (dy == 0) {
                return "E";
            } else { //if (dy > 0) {
                return "SE";
            }
        }
    }

    public Color getOutlineX() {
        return outlineX;
    }

    public int getPanelNum() {
        return idNum;
    }

    public BufferedImage getScaled(int scale) {
        BufferedImage img = getImage();
        Dimension scaledSize = getScaledSize(scale);
        return ImageTool.getResized2(img, scaledSize.width, scaledSize.height);
    }

    public Dimension getScaledSize(int scale) {
        int scaledWidth = (int) ((double) getSize().width / (double) scale);
        int scaledHeight = (int) ((double) getSize().height / (double) scale);
        return new Dimension(scaledWidth, scaledHeight);
    }

    public List<Pixel> getStroke(int index) {
        if (index < strokes.size()) {
            return strokes.get(index);
        }
        return null;
    }

    public List<List<Pixel>> getStrokes() {
        return strokes;
    }

    public DrawPanel getTopLeft() {
        List<List<DrawPanel>> panels = driver.getActual().getCurrentArea().getPanels();
        if (panels.size() == 0) {
            System.err.println("getTopLeft: panels.size == 0");
            return this;
        }
        DrawPanel tl = panels.get(0).get(0);
        return panels.get(0).get(0);
    }

    public void itemStateChanged(ItemEvent e) {
        if (e.getSource() == setIdentify) {
            setIdentify(setIdentify.isSelected());
        }
    }

    public static void main(String[] args) {
        boolean working = true;
        if (working) {
            Driver.launch();
        } else {
            JFrame testApp = new JFrame();
            DrawPanel panel = new DrawPanel(null, 300, 300);
            testApp.add(panel, BorderLayout.CENTER);
            testApp.pack();
            testApp.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            testApp.setVisible(true);
        }
    }

    @Override
    public void mouseClicked(MouseEvent e) {
        if (SwingUtilities.isRightMouseButton(e)) {
            if (e.isControlDown()) {
                setBackground(Colors.BACK_COLOR);
                driver.getActual().getCurrentArea().getPanel(gridX,gridY).setBackground(Colors.BACK_COLOR);
            } else {
                popupCreate();
                popupAddIdentify();
                popupAddRepaint();
                popupShow(e);
            }
        }
    }

    @Override
    public void mouseDragged(MouseEvent e) {
        current = (DrawPanel) e.getSource();
        int deltaX;
        int deltaY;
        java.awt.Point p = e.getLocationOnScreen();
        deltaX = -(lastX - p.x);
        deltaY = -(lastY - p.y);

        // use these to locate from after move is computed, unless CREATING
        int startX = getTopLeft().getLocation().x + deltaX;
        int startY = getTopLeft().getLocation().y + deltaY;

        if (e.isControlDown() && e.isShiftDown()) {
            // use these to determine whether to create or not
            String moveDirection = getMoveDirection(deltaX, deltaY);
            DrawPanel topLeft = getTopLeft();
            DrawPanel botRight = getBottomRight();
            java.awt.Point tl = topLeft.getLocation();
            java.awt.Point br = botRight.getLocation();
            int halfWidth = getSize().width / 2;
            int halfHeight = getSize().height / 2;
            int holderWidth = driver.getScribble().getHolder().getSize().width;
            int holderHeight = driver.getScribble().getHolder().getSize().height;
            String msg = "";
            String createDirection = "";

            if (moveDirection.equals("SE")) {
                if (tl.x > halfWidth && tl.y > halfHeight) {
                    createDirection = "NW";
                }
            } else if (moveDirection.equals("S")) {
                if (tl.y >= halfHeight) {
                    createDirection = "N";
                }
            } else if (moveDirection.equals("SW")) {
                if (br.x + halfWidth <= holderWidth && tl.y > halfHeight) {
                    createDirection = "NE";
                }
            } else if (moveDirection.equals("W")) {
                if (br.x + halfWidth <= holderWidth) {
                    createDirection = "E";
                }
            } else if (moveDirection.equals("NW")) {
                if (br.x + halfWidth < holderWidth - halfWidth
                        && br.y + halfHeight < holderHeight - halfHeight) {
                    createDirection = "SE";
                }
            } else if (moveDirection.equals("N")) {
                if (br.y + halfHeight < holderHeight - halfHeight) {
                    createDirection = "S";
                }
            } else if (moveDirection.equals("NE")) {
                if (tl.x - halfWidth > 0
                        && br.y + halfHeight <= holderHeight - halfHeight) {
                    createDirection = "SW";
                }
            } else if (moveDirection.equals("E")) {
                if (tl.x - halfWidth > 0) {
                    createDirection = "W";
                }
            }
            if (!createDirection.equals("")) {
                Point loc = driver.getActual().create(createDirection);
                System.out.println("Created: " + createDirection + " Size: " + driver.getActual().getCurrentArea().getPanelCount() + " Loc: " + loc.x + "," + loc.y);
                driver.getScribble().relayoutFrom(loc.x, loc.y);
            }
            driver.getActual().resetArea();
        } else if (e.isControlDown()) {
            driver.getScribble().relayoutFrom(startX, startY);
        } else {
            DrawPanel src = (DrawPanel) e.getSource();
            int x = e.getX();
            int y = e.getY();
            int w = getSize().width;
            int h = getSize().height;
            if (x >= 0 && x < w && y >= 0 && y < h) {
                addToMyStroke(x, y);
            } else {
                AreaRep rep = driver.getScribble().getAreaRep();
                int coordX;
                if (x < 0) {
                    if (x > -w) coordX = -1;
                    else coordX = (x / w) - 1;
                } else coordX = x / w;
                int coordY;
                if (y < 0) {
                    if (y > -h) coordY = -1;
                    else coordY = (y / h) - 1;
                } else coordY = y / h;
                int newCoordX = gridX + coordX;
                int newCoordY = gridY + coordY;
                //System.out.println("P: " + x + "," + y + " Coord: " + coordX + "," + coordY + " NewCoord: " + newCoordX + "," + newCoordY);
                int px = x % w;
                int py = y % h;
                if (px < 0) px = w + px;
                if (py < 0) py = h + py;
                //System.out.println("PXY: " + px + "," + py);
                if (newCoordX >= 0 && newCoordX < rep.getMaxX() &&
                    newCoordY >= 0 && newCoordY < rep.getMaxY()) 
                {
                    
                    rep.addToStroke(newCoordX,newCoordY,px,py);
                }
            }
        }
        lastX = p.x;
        lastY = p.y;
    }

    @Override
    public void mouseEntered(MouseEvent e) {
        current = (DrawPanel) e.getSource();
    }

    @Override
    public void mouseExited(MouseEvent e) {
    }

    @Override
    public void mouseMoved(MouseEvent e) {
        java.awt.Point p = e.getLocationOnScreen();
        lastX = p.x;
        lastY = p.y;
    }

    @Override
    public void mousePressed(MouseEvent e) {
        current = (DrawPanel) e.getSource();
        super.mousePressed(e);
        driver.getScribble().getAreaRep().startStrokes();
        addToMyStroke(e.getX(), e.getY()); // must be inside panel to get event
        if (SwingUtilities.isLeftMouseButton(e)) {
            if (driver.getColors().isRandom(Colors.PEN_COLOR)) {
                driver.getColors().paletteDoMove();
            }
        } else if (SwingUtilities.isRightMouseButton(e)) {
            if (driver.getColors().isRandom(Colors.BACK_COLOR)) {
                driver.getColors().paletteDoMove();
            }
        }
    }

    @Override
    public void mouseReleased(MouseEvent e) {
        super.mouseReleased(e);
        driver.getScribble().getAreaRep().endStrokes();
    }

    @Override
    public void mouseWheelMoved(MouseWheelEvent e) {
    }

    public void paint(Graphics g) {
        super.paint(g);

        // override background painting
        //g.setColor(bg);
        //g.fillRect(0,0,getSize().width,getSize().height);
        //System.out.println("DrawPanel.paint()");
        doPainting(g);
    }

    /**
     * I know everybody says to override paintComponent, but when I do the
     * tracing I always find that it is paint() that gets called, so if you can
     * make that work better, by all means, go nuts.
     */
//    public void paintComponent(Graphics g) {
//        super.paintComponent(g);
    // override background painting
    //g.setColor(bg);
    //g.fillRect(0,0,getSize().width,getSize().height);
//        System.out.println("DrawPanel.paintComponent()");
//        doPainting(g);
//    }
    public void popupCreate() {
        popup = new JPopupMenu();
    }

    public void popupAddIdentify() {
        popup.add(setIdentify);
    }

    public void popupAddRepaint() {
        popup.add(repaint);
    }

    public void popupAddSeparator() {
        popup.addSeparator();
    }

    public void popupShow(MouseEvent e) {
        popup.show(this, e.getX(), e.getY());
    }

    public void reset() { // reset the contents only
        strokes = new ArrayList<>();
    }

    public void setIdentify(boolean state) {
        identify = state;

        // turn off event processing, update the menu state, and resume event processing
        setIdentify.removeItemListener(this);
        setIdentify.setSelected(state);
        setIdentify.addItemListener(this);

        // refresh the Canvas/Panel
        repaint();
    }

    public void setBackground(Color c) {
        bg = c;
        super.setBackground(c);
        repaint();
    }

    public void setCoordinate(int x, int y) {
        gridX = x;
        gridY = y;
    }

    public void setGuides(boolean state) {
        guides = state;
        repaint();
    }

    public void setPanelNum(int i) {
        idNum = i;
        repaint();
    }

    public void setOutlineX(Color c) {
        outlineX = c;
        repaint();
    }

    public void startStroke(List<Pixel> newEmptyStroke) {
        stroke = newEmptyStroke;
        strokes.add(stroke);
    }

    public String toString() {
        return "loc: " + getLocation()
                + " size: " + getSize()
                + " bg: " + bg
                + " outlineX: " + outlineX
                + " identify: " + identify
                + " idNum: " + idNum;
    }
}
