package ca.tecreations;

import ca.tecreations.misc.Time;
import ca.tecreations.lang.java.GetClassPathFor;
import ca.tecreations.net.TecStreamPrinterClient;
import ca.tecreations.net.TLS_TSPS;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;

/**
 *
 * @author tim
 */
public class SystemTool {
    public static String TARGET_JAVA_VERSION = "17";
    
    private final Object lock = new Object();
    private boolean debug = false;
    public static boolean verbose = false;
    Runtime runtime = Runtime.getRuntime();
    Process process;

    int exitValue;

    List<SystemToken> tokens = new ArrayList<>();
 
    public TLS_TSPS tsps = null;
    
    public SystemTool() {
    }
   
    public List<SystemToken> buildPackage(String classPath, String pkg) {
        List<SystemToken> result = new ArrayList<>();
        classPath = StringTool.getUnwrapped(classPath);
        pkg = StringTool.getUnwrapped(pkg);
        if (pkg.length() > 0 && !pkg.endsWith(".")) pkg += ".";
        String subPath = pkg;
        subPath = StringTool.replaceAll(subPath,".",File.separator);
        File[] entries = new File(classPath + subPath).listFiles();
        List<File> dirs = new ArrayList<>();
        List<File> files = new ArrayList<>();
        for(int i = 0;i < entries.length;i++) {
            if (entries[i].isFile() && entries[i].getExtension().equals("java")) {
                files.add(entries[i]);
            } else if (entries[i].isDirectory()) {
                dirs.add(entries[i]);
            }
        }
        dirs = Sort.sortListOfFile(dirs);
        files = Sort.sortListOfFile(files);
        
        for(int i = 0;i < files.size();i++) {
            compile(classPath,new File(classPath + subPath + files.get(i).getName()),true);
            for(int j = 0; j < tokens.size();j++) {
                result.add(tokens.get(j));
            }
        }
        for(int i = 0; i < dirs.size();i++) {
            List<SystemToken> tokens = buildPackage(classPath,pkg + dirs.get(i).getName());
            for(int j = 0; j < tokens.size();j++) {
                result.add(tokens.get(j));
            }
        }
        return result;
    }
       
    public List<SystemToken> buildSubPath(String classPath, String subPath) {
        List<SystemToken> result = new ArrayList<>();
        classPath = StringTool.getUnwrapped(classPath);
        subPath = StringTool.getUnwrapped(subPath);
        File[] entries = new File(classPath + subPath).listFiles();
        List<File> dirs = new ArrayList<>();
        List<File> files = new ArrayList<>();
        for(int i = 0;i < entries.length;i++) {
            if (entries[i].isFile() && entries[i].getExtension().equals("java")) {
                files.add(entries[i]);
            } else if (entries[i].isDirectory()) {
                dirs.add(entries[i]);
            }
        }
        dirs = Sort.sortListOfFile(dirs);
        files = Sort.sortListOfFile(files);
        
        for(int i = 0;i < files.size();i++) {
            compile(classPath,new File(classPath + subPath + files.get(i).getName()),true);
            for(int j = 0; j < tokens.size();j++) {
                result.add(tokens.get(j));
            }
        }
        for(int i = 0; i < dirs.size();i++) {
            List<SystemToken> tokens = buildPackage(classPath,subPath + dirs.get(i).getName() + File.separator);
            for(int j = 0; j < tokens.size();j++) {
                result.add(tokens.get(j));
            }
        }
        return result;
    }
    
    public void compile(String classPath, File file, boolean debug) {
        int exitValue = -999;
        tokens = new ArrayList<>();
        List<String> parts = getCompileCommand(classPath, file);
        ProcessBuilder builder = new ProcessBuilder(parts);
        classPath = StringTool.getUnwrapped(classPath);
        if (!classPath.endsWith(File.separator)) classPath += File.separator;
        String pkgAndFile = file.getUnwrapped().substring(classPath.length());
        String pkgPathAndName = pkgAndFile.substring(0,pkgAndFile.lastIndexOf("."));
        String sep = pkgPathAndName.contains("/") ? "/" : "\\";
        String pkgAndName = StringTool.replaceAll(pkgPathAndName,sep,".");
        try { 
            process = builder.start();
        } catch (IOException ioe) {
            System.err.println("Unable to start process: " + TypeToType.toSpacedString(parts));
        } 
        // block until completed
        try { 
            process.waitFor(); 
        } catch (InterruptedException ie) {
            System.err.println("Compiling: Interrupted: " + ie);
        }
        while (process.isAlive()) try {
            wait(125, 0);
        } catch (InterruptedException ie) {
            System.err.println("Interrupted: " + ie);
        }
        String appEventTime = new Time().getAppEventTime();
        exitValue = process.exitValue();
        String debugStr = "SystemTool.compile: " + exitValue + " : " + new Time().getAppEventTime() + " : " + pkgAndName;
        if (verbose) debugStr += " : " + TypeToType.toSpacedString(parts);
        if (debug) { 
            if (tsps != null) tsps.send(debugStr);
            System.out.println(debugStr);
        } 
        tokens.add(new SystemToken(debugStr + " : " + process.exitValue(),TecData.SYS_OUT));
        try {
            InputStreamReader oisr = new InputStreamReader(process.getInputStream());
            BufferedReader stdOut = new BufferedReader(oisr);
            InputStreamReader eisr = new InputStreamReader(process.getErrorStream());
            BufferedReader stdError = new BufferedReader(eisr);
            String s = null;
            
            // wait for the process to finish, and then, get the output
            while ((s = stdOut.readLine()) != null) {
                if (debug) System.out.println(s);
                if (tsps != null) tsps.send(s);
                tokens.add(new SystemToken(s, TecData.SYS_OUT));
            }
            stdOut.close();
            oisr.close();
            while ((s = stdError.readLine()) != null) {
                if (debug) System.err.println(s);
                if (tsps != null) tsps.send(s);
                tokens.add(new SystemToken(s, TecData.SYS_ERR));
            }
            stdError.close();
            eisr.close();
        } catch (IOException ioe) {
            System.out.println("Unable to read data from process.");
        } 
    } 

    public String getCommand(List<String> parts) {
        String command = "";
        for (int i = 0; i < parts.size() - 1; i++) {
            command += parts.get(i) + " ";
        }
        if (parts.size() > 0) {
            command += parts.get(parts.size() - 1);
        }
        return command;
    }

    public String getCommand(String[] parts) {
        String command = "";
        for (int i = 0; i < parts.length; i++) {
            command += parts[i] + " ";
        }
        if (parts.length > 0) {
            command += parts[parts.length - 1];
        }
        return command;
    }

    // https://docs.oracle.com/en/java/javase/17/docs/specs/man/javac.html#:~:text=Description,Java%20source%20files%20and%20classes.
    public List<String> getCompileCommand(String classPath, File file) {
        classPath = StringTool.getUnwrapped(classPath);
        String sep = (classPath.contains("/") ? "/" : "\\");
        if (!classPath.endsWith(sep) && !classPath.equals(".")) {
            classPath += sep;
        }
        List<String> parts = new ArrayList<>();
        parts.add("javac");
        parts.add("-cp");
        parts.add(new GetClassPathFor(StringTool.getDoubleQuoted(classPath)).getResult());
        //parts.add("--release");
        //parts.add(TARGET_JAVA_VERSION);;
        parts.add("--source");
        parts.add(TARGET_JAVA_VERSION);
        parts.add("--target");
        parts.add(TARGET_JAVA_VERSION); 
        parts.add("--system");
        parts.add(System.getProperty("java.home"));
         
        // use these, if you want
        parts.add("-Xlint:deprecation");
        parts.add("-Xlint:unchecked");
        parts.add("-parameters");
        parts.add(file.getUnwrapped());
        return parts;  
    } 

    public boolean getDebug() {
        return debug;
    }

    public int getExitValue() {
        return exitValue;
    }

    public Process getProcess() {
        return process;
    }

    public List<String> getRunCommand(String classPath, String className, String[] args) {
        List<String> parts = new ArrayList<>();
        parts.add("java");
        parts.add("-cp");
        parts.add(new GetClassPathFor(ProjectPath.instance.getProjectPath()).getResult());
        parts.add(className);
        if (args != null) {
            for(int i = 0;i < args.length;i++) {
                parts.add(args[i]);
            }
        }
        return parts;
    }
    
    public List<String> getRunCommand(Jar jar,String className, String[] args) {
        List<String> parts = new ArrayList<>();
        parts.add("java");
        parts.add("-cp");
        parts.add(new GetClassPathFor(jar.getUnwrapped()).getResult());
        parts.add(className);
        if (args != null) {
            for(int i = 0; i < args.length;i++) {
                parts.add(args[i]);
            }
        }
        return parts;
    }

    public static String getShortPackageAndNameForJavaFilename(String src) {
        src = StringTool.replaceAll(src,File.separatorChar,".");
        //System.out.println("Src: " + src);
        String shortName = "";
        String remainder = "";
        if (src.contains(".")) {
            shortName += src.charAt(0) + ".";
            remainder = src.substring(src.indexOf(".") + 1);
        } else return src; // in default package
        //System.out.println("Remainder: " + remainder);
        while (remainder.indexOf(".") > 0) {
            shortName += remainder.charAt(0) + ".";
            remainder = remainder.substring(remainder.indexOf(".") + 1);
        }
        if (!remainder.contains(".")) shortName += remainder;
        return shortName;
    }
    
    public List<SystemToken> getTokens() {
        return tokens;
    }

    public boolean isPOSIX() { return !isWin(); }
    
    public boolean isWin() {
        return File.separator.equals("\\");
    }
    
    public boolean isWindows() { return isWin(); }
    
    public boolean isWrapped(String s) {
        s = s.trim();
        return s.startsWith("\"") && s.endsWith("\"");
    }
    
    public static void main(String[] args) {
        String classPath = "F:\\projects\\tec8\\";
        File file = new File("F:\\projects\\tec8\\RunFindImports.java");
        System.out.println("GetCompileCommand: " + new SystemTool().getCompileCommand(classPath,file));
        
        //System.out.println("ShortPackage('A_SpawnSystemCompiler'): " + getShortPackageAndNameForJavaFilename("A_SpawnSystemCompiler"));
        //System.out.println("ShortPackage('ca" + File.separator + "tecreations" + File.separator + "BuildProject'): " + getShortPackageAndNameForJavaFilename("ca\\tecreations\\BuildProject"));
        //System.out.println("ShortPackage('ca" + File.separator + "Pair): " + getShortPackageAndNameForJavaFilename("ca\\Pair"));
        //System.exit(0);
        
//        new TecStreamPrinterServer();
//        TecStreamPrinterClient client = new SystemTool().spawnWithNetworkOutput(SystemTool.class.getSimpleName(),"java -cp " + new GetClassPathFor(ProjectPath.getSourceJarPath()).getResult() + " ca.tecreations.FindInFiles JarReader",true);
        //Process process = new SystemTool().spawnAndOutput(System.out,System.err,"java -cp " + new GetClassPathFor(ProjectPath.getSourceJarPath()).getResult() + " ca.tecreations.FindInFiles JarReader",true);
//        Process process = client.getProcess();
//        while (process.isAlive()) {
//            Platform.sleep(500); 
//        }
//        System.exit(0);

        
    }
    
    /**
     * Runs a process and returns the java.lang.Process unmodified.
     * @param command
     * @return 
     */
    
    public Process runAndGet(String command) {
        return runAndGet(command,false);
    }
    
    public Process runAndGet(List<String> parts, boolean debug) {
        if (debug) {
            System.out.println("SystemTool.runAndGet(1): " + TypeToType.toString(parts));
        }  
        ProcessBuilder builder = new ProcessBuilder(parts);
        try { 
            process = builder.start();
            while (!process.isAlive()) Platform.sleep(125);
        } catch (IOException ioe) {
            System.err.println("runAndGet: Unable to start process: " + TypeToType.toString(parts));
        }
        return process;
    }
    
    public Process runAndGet(String command,boolean debug) {
        if (debug) {
            System.out.println("SystemTool.runAndGet(2): " + command);
        }
        if (command.startsWith("`") && command.endsWith("`")) {
            command = command.substring(0,command.lastIndexOf("`"));
        }
        
        ProcessBuilder builder = new ProcessBuilder(StringTool.explode(command,' '));
        try {
            process = builder.start();
            while (!process.isAlive()) Platform.sleep(125);
        } catch (IOException ioe) {
            System.err.println("runAndGet: Unable to start process: " + command);
        }
        return process;
    }

    public Process runAndGet(String command, String[] args, boolean debug) {
        if (args.length > 0) {
            command += " " + args[0];
        }
        for(int i = 1;i < args.length;i++) {
            command += " " + args[i];
        }
        if (debug) System.out.println("SystemTool.runAndGet(3): " + command);
        return runAndGet(command);
    } 
    
    public void runAndOutput(String command, boolean debug) {
        command = StringTool.getUnwrapped(command);
        tokens = new ArrayList<>();
        if (debug) {
            System.out.println("SystemTool.runAndOutput: " + command);
        }
        ProcessBuilder builder = new ProcessBuilder(StringTool.explode(command,' '));
        try {
            process = builder.start();
            //while (!process.isAlive()) Platform.sleep(125); // causes the program to stall on Ubuntu Linux 16.04
        } catch (IOException ioe) {
            System.err.println("Unable to start process: " + command);
        }
        try {
            process.waitFor();
        } catch (InterruptedException ie) {
            System.err.println("Interrupted: " + ie);
        }
        String s = null;
        try {
            InputStreamReader isr = new InputStreamReader(process.getErrorStream());
            BufferedReader stdErr = new BufferedReader(isr);
            while ((s = stdErr.readLine()) != null) {
                System.err.println(s);
            }
        } catch (IOException ioe) {
            System.out.println("Unable to read ERR output from process.");
        }
        try {
            InputStreamReader isr2 = new InputStreamReader(process.getInputStream());
            BufferedReader stdOut = new BufferedReader(isr2);
            while ((s = stdOut.readLine()) != null) {
                System.out.println(s);
            }
            stdOut.close();
            isr2.close();
        } catch (IOException ioe) {
            System.out.println("Unable to read OUT output from process.");
        }
    }
     
    public int runAndWait(List<String> parts, boolean debug) {
        if (debug) {
            System.out.println("SystemTool.runAndWait(1): " + TypeToType.toString(parts));
        }  
        ProcessBuilder builder = new ProcessBuilder(parts);
        try { 
            process = builder.start();
            while (!process.isAlive()) Platform.sleep(125);
        } catch (IOException ioe) {
            System.err.println("runAndWait: Unable to start process: " + TypeToType.toString(parts));
        }
        try {
            process.waitFor();
        } catch (InterruptedException ie) {
            System.err.println("runAndWait: interrupted: " + parts);
        }
        return process.exitValue();
    }

    public int runAndWait(String command, String[] args, boolean debug) {
        if (args.length > 0) {
            command += " " + args[0];
        }
        for(int i = 1;i < args.length;i++) {
            command += " " + args[i];
        }
        if (debug) System.out.println("SystemTool.runAndWait(3): " + command);
        return runAndWait(command);
    } 
    
    public int runAndWait(String command) {
        return runAndWait(command,false);
    }
    
    public int runAndWait(String command,boolean debug) {
        if (debug) {
            System.out.println("SystemTool.runAndWait(2): " + command);
        }
        if (command.startsWith("`") && command.endsWith("`")) {
            command = command.substring(0,command.lastIndexOf("`"));
        }
        
        ProcessBuilder builder = new ProcessBuilder(StringTool.explode(command,' '));
        try {
            process = builder.start();
            while (!process.isAlive()) Platform.sleep(125);
        } catch (IOException ioe) {
            System.err.println("runAndWait: unable to start process: " + command);
        }
        try {
            process.waitFor();
        } catch (InterruptedException ie) {
            System.err.println("runAndWait: interrupted: " + command);
        }
        return process.exitValue();
    }

    
    public List<SystemToken> runForOutput(String command, boolean debug) {
        command = StringTool.getUnwrapped(command);
        tokens = new ArrayList<>();
        if (debug) {
            System.out.println("SystemTool.runForOutput: " + command);
        }
        ProcessBuilder builder = new ProcessBuilder(StringTool.explode(command,' '));
        try {
            process = builder.start();
            //while (!process.isAlive()) Platform.sleep(125); // causes the program to stall on Ubuntu Linux 16.04
        } catch (IOException ioe) {
            System.err.println("Unable to start process: " + command);
        }
        try {
            process.waitFor();
        } catch (InterruptedException ie) {
            System.err.println("Interrupted: " + ie);
        }
        String s = null;
        try {
            InputStreamReader isr = new InputStreamReader(process.getErrorStream());
            BufferedReader stdErr = new BufferedReader(isr);
            while ((s = stdErr.readLine()) != null) {
                tokens.add(new SystemToken(s,TecData.SYS_ERR));
            }
            stdErr.close();
            isr.close();
        } catch (IOException ioe) {
            System.out.println("Unable to read ERR output from process.");
        }
        try {
            InputStreamReader isr2 = new InputStreamReader(process.getInputStream());
            BufferedReader stdOut = new BufferedReader(isr2);
            while ((s = stdOut.readLine()) != null) {
                tokens.add(new SystemToken(s,TecData.SYS_OUT));
            }
            stdOut.close();
            isr2.close();
        } catch (IOException ioe) {
            System.out.println("Unable to read OUT output from process.");
        }
        return tokens;
    }
     
    public List<String> runForOutputListString(String command, boolean debug) {
        // trim to start
        command = command.trim();
        
        // we'll make it the practice to possibly wrap commands in backticks for
        // things like TLSClient/TLSServer where we might have a single arg composing
        // several pieces, ie: `ps -ax | grep tec` is one command on Linux made up of
        // several pieces. On Windows, we might have something like:
        // `notepad.exe "c:\temp file.txt"`
        if (command.startsWith("`") && command.endsWith("`")) {
            command = command.substring(1,command.lastIndexOf("`"));
        }
        
        List<String> lines = new ArrayList<>();
        if (debug) {
            System.out.println("SystemTool.runForOutputListString: " + command);
        }
        ProcessBuilder builder = new ProcessBuilder(StringTool.explode(command,' '));
        try {
            process = builder.start();
            //while (!process.isAlive()) Platform.sleep(125); // causes the program to stall on Ubuntu Linux 16.04
        } catch (IOException ioe) {
            System.err.println("Unable to start process: " + command);
        }
        try {
            process.waitFor();
        } catch (InterruptedException ie) {
            System.err.println("Interrupted: " + ie);
        }
        String s = null;
        try {
            InputStreamReader isr = new InputStreamReader(process.getErrorStream());
            BufferedReader stdErr = new BufferedReader(isr);
            while ((s = stdErr.readLine()) != null) {
                lines.add(s);
            }
            stdErr.close();
            isr.close();
        } catch (IOException ioe) {
            System.out.println("Unable to read ERR output from process.");
        }
        try {
            InputStreamReader isr2 = new InputStreamReader(process.getInputStream());
            BufferedReader stdOut = new BufferedReader(isr2);
            while ((s = stdOut.readLine()) != null) {
                lines.add(s);
            }
            stdOut.close();
            isr2.close();
        } catch (IOException ioe) {
            System.out.println("Unable to read OUT output from process.");
        }
        return lines;
    }
     
    public Process runJava(String classPath, String className, String args) {
        List<String> parts = StringTool.explode(args,' ');
        String[] argsParts = new String[parts.size()];
        for(int i = 0; i < parts.size();i++) argsParts[i] = parts.get(i);
        return runJava(classPath,className,argsParts);
    }
 
    public Process runJava(String classpath, String className, List<String> args) {
        String[] argsParts = new String[args.size()];
        for(int i = 0; i < args.size();i++) argsParts[i] = args.get(i);
        return runJava(classpath,className,argsParts);
    }
 
    public Process runJava(String classpath, String className, String[] args) {
        List<String> parts = getRunCommand(classpath, className,args);
        ProcessBuilder builder = new ProcessBuilder(parts);
        System.out.println("SystemTool.runJava: " + new Time().getAppEventTime() + " : " + TypeToType.toSpacedString(parts));
        try {
            process = new ProcessBuilder(parts).start();
            while (!process.isAlive()) Platform.sleep(125);
        } catch (IOException ioe) {
        }
        return process;
    }
 
    public void runAndOutputAndWait(PrintStream out,PrintStream err, String command, boolean debug) {
        ProcessBuilder builder = new ProcessBuilder(StringTool.explode(command,' '));
        if (debug) System.out.println("SystemTool.runAndOutput: " + new Time().getAppEventTime() + " : " + command);
        try {
            process = builder.start();
        } catch (IOException ioe) {
            System.out.println("Unable to start process.");
        }
        new StreamPrinter(process,out,err);
        try {
            process.waitFor();
        } catch (InterruptedException ie) {
            System.out.println("While waiting: " + ie);
        }
    }
    
    public Process runAndInheritIO(String command,boolean debug) {
        tokens = new ArrayList<>();
        ProcessBuilder builder = new ProcessBuilder(StringTool.explode(command,' '));
        if (debug) System.out.println("SystemTool.runAndInheritIO: " + new Time().getAppEventTime() + " : " + command);
        try {
            builder.inheritIO();
            process = builder.start();
            while (!process.isAlive()) Platform.sleep(125);
           
        } catch (IOException ioe) {
            System.err.println("Unable to start process: " + command);
        }
        return process;
    }
    
    public void setTSPS(TLS_TSPS tsps) {
        this.tsps = tsps;
    }
    
    public Process spawn(String command,boolean debug) {
        command = StringTool.getUnwrapped(command);
        if (debug) System.out.println("SystemTool.spawn: '" + command + "'");
        try { 
            process = runtime.exec(command);
        } catch (IOException ioe) {
            System.out.println("Unable to execute: " + command);
        }
        return process;
    }
 
    public Process spawnAndOutput(PrintStream out,PrintStream err,String command,boolean debug) {
        command = StringTool.getUnwrapped(command);
        if (debug) System.out.println("SystemTool.spawnAndOutput: '" + command + "'");
        try { 
            process = runtime.exec(command);
            while (!process.isAlive()) Platform.sleep(125);
            new StreamPrinter(process,out,err);
        } catch (IOException ioe) {
            System.out.println("Unable to execute: " + command);
        }
        return process;
    }

    public Process spawnJava(String classPath, String className, String args) {
        List<String> parts = StringTool.explode(args,' ');
        String[] argsParts = new String[parts.size()];
        for(int i = 0; i < parts.size();i++) argsParts[i] = parts.get(i);
        return spawnJava(classPath,className,argsParts);
    }
 
    public Process spawnJava(String classpath, String className, List<String> args) {
        String[] argsParts = new String[args.size()];
        for(int i = 0; i < args.size();i++) argsParts[i] = args.get(i);
        return runJava(classpath,className,argsParts);
    }
 
    public Process spawnJava(String classPath, String className, String[] args)  {
        List<String> parts = getRunCommand(classPath, className,args);
        System.out.println("SystemTool.spawnJava: " + new Time().getAppEventTime() + " : " + TypeToType.toSpacedString(parts));
        ProcessBuilder pb = new ProcessBuilder(parts);
        pb.inheritIO();
        try {
            process = pb.start();
        } catch (IOException ioe) {
            System.err.println("Process starting: " + ioe);
        }
        return process;
    }

    public TecStreamPrinterClient spawnWithNetworkOutput(String sourceClassName, String command,boolean debug) {
        command = StringTool.getUnwrapped(command);
        if (debug) System.out.println("SystemTool.spawnWithNetworkOutput: '" + command + "'");
        try { 
            process = runtime.exec(command);
            while (!process.isAlive()) Platform.sleep(125);
        } catch (IOException ioe) {
            System.out.println("Unable to execute: " + command);
        } 
        return new TecStreamPrinterClient(process,sourceClassName);
    }
    
    public List<SystemToken> spawnWithOutput(String cmd, boolean debug) {
        tokens = new ArrayList<>();
        tokens.add(new SystemToken("test",TecData.SYS_ERR));
        cmd = StringTool.getUnwrapped(cmd);
        if (debug) System.out.println("SystemTool.spawnWithOutput: '" + cmd + "'");
        try { 
            process = runtime.exec(cmd);
        } catch (IOException ioe) {
            System.out.println("Unable to execute: " + cmd);
        }
        try {
            process.waitFor();
        } catch (InterruptedException ie) {
            System.out.println("Interrupted");
        }
        return tokens;
    }
}
