package ca.tecreations;

import ca.tecreations.components.event.*;
import ca.tecreations.misc.BitSet;
import ca.tecreations.misc.Time;
import ca.tecreations.net.ExceptionHandler;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.DosFileAttributes;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.GroupPrincipal;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.nio.file.attribute.UserPrincipal;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 *
 * @author Tim
 *
 */
public class File {
    public static final int DOS = 0;
    public static final int POSIX = 1;
    protected int type;

    String absPath;

    public static boolean debug = true;
    public static boolean verbose = false;
    
    List<String> basicStats;
    List<String> basicAttributes;

    public static final String separator = java.io.File.separator;
    public static final char separatorChar = java.io.File.separatorChar;
    public static final String pathSeparator = java.io.File.pathSeparator;
    public static final char pathSeparatorChar = java.io.File.pathSeparatorChar;
    

    public File(java.io.File file) {
        this(file.getAbsolutePath());
    }

    public File(java.nio.file.Path path) {
        java.io.File javaFile = path.toFile();
        if (javaFile.isDirectory()) {
            absPath = javaFile.getAbsolutePath() + javaFile.separator;
        } else {
            absPath = javaFile.getAbsolutePath();
        }
        this.absPath = StringTool.getDoubleQuoted(absPath);
    }
    
    public File(String path) {
        String unwrapped = StringTool.getUnwrapped(path);
        java.io.File javaFile = new java.io.File(unwrapped);
        if (javaFile.isDirectory()) {
            if (unwrapped.contains(":")) {
                if (!unwrapped.endsWith("\\")) unwrapped += "\\";
            } else {
                if (!unwrapped.endsWith("/")) unwrapped += "/";
            }
        }
        this.absPath = StringTool.getDoubleQuoted(unwrapped);
    }
    
    public File(String path, boolean oldTec) {
        if (oldTec) {
            String absPath = StringTool.getUnwrapped(path);
            if (absPath.endsWith(":")) absPath += "\\";
            if (!absPath.contains("\\") && !absPath.contains("/")
                && !absPath.equals(".") && !absPath.equals("..")
                && !absPath.equals("")) {
                // just a filename, so get the current directory, append the name
                // and assign
                this.absPath = new File(".").getDeepestDirectory().getUnwrapped() + absPath;
            } else {
                if (absPath.equals(".") || absPath.equals("")) {
                    absPath = new java.io.File(".").getAbsolutePath();
                    absPath = absPath.substring(0, absPath.lastIndexOf("."));
                }
                if (!absPath.contains("/") && !absPath.contains("\\")) {
                    // so just a filename, get the CWD and prepend
                    this.absPath = new java.io.File(".").getAbsolutePath() + absPath;
                } 
                if (exists()) {
                    if (isDirectory()) {
                        if (absPath.contains("/") && !absPath.endsWith("/")) {
                            absPath += "/";
                        } else if (absPath.contains("\\") && !absPath.endsWith("\\")) {
                            absPath += "\\";
                        }   
                    }
                }
                this.absPath = StringTool.getDoubleQuoted(absPath);
            }
        } else {
            java.io.File javaFile = new java.io.File(absPath);
            if (javaFile.isDirectory()) {
                this.absPath = StringTool.getDoubleQuoted(javaFile.getAbsolutePath() + javaFile.separator);
            } else {
                this.absPath = StringTool.getDoubleQuoted(javaFile.getAbsolutePath());
            }
        }
    }
    //
    //==========================================================================
    //
    
    public boolean canExecute() {
        return getJavas().canExecute();
    }

    public boolean canRead() {
        return getJavas().canRead();
    }

    public boolean canWrite() {
        return getJavas().canWrite();
    }

    public static void copyA(File srcFile, File dstDir, boolean detailed) {
        if (detailed) doLogAction("File.copy: srcFile: " + srcFile.getAbsolutePath() + " dstDir: " + dstDir.getAbsolutePath());
        dstDir.mkdirs();
        File deepest = dstDir.getDeepestDirectory();
        deepest.mkdirs();
        FileInputStream fis = null;
        DataInputStream dis = null;
        FileOutputStream fos = null;
        DataOutputStream dos = null;
        try {
            fis = new FileInputStream(srcFile.getJavas());
            dis = new DataInputStream(fis);
        } catch (IOException ioe) {
            System.err.println("File.copy: opening input(" + srcFile.getAbsolutePath() + " ): " + ioe);
        }
        String outPath = deepest.getUnwrapped() + srcFile.getName();
        try {
            fos = new FileOutputStream(outPath);
            dos = new DataOutputStream(fos);
        } catch (IOException ioe) {
            System.err.println("File.copy: opening output(" + outPath + " ): " + ioe);
        }
        if (dis == null | dos == null) {
            System.err.println("File.copy: NULL stream: dis: " + dis + " dos: " + dos);
        } else {
            doCopy(dis, dos);
        }
        try {
            if (dos != null) dos.flush();
            if (dos != null) dos.close();
            if (fos != null) fos.close();
            if (dis != null) dis.close();
            if (fis != null) fis.close();
        } catch (IOException ioe) {
            System.err.println("File.copy: While closing: " + ioe);
        }
    }

    public static void copyB(InputStream is, long length, String dst, ProgressListener pl) {
        long oneMeg = 1024 * 1024;
        int oldMegs = 0;
        if (pl != null) pl.updateItem(0);
        FileOutputStream fos = null;
        byte[] buffer = new byte[60 * 1024];
        long itemTotal = 0;
        int bytesRead = 0;

        new File(dst).getDeepestDirectory().mkdirs();
        try {
            fos = new FileOutputStream(StringTool.getUnwrapped(dst));
        } catch (FileNotFoundException fnfe) {
            ExceptionHandler.handleIO("File.copyB", "file not found", fnfe);
        }
        if (is != null && fos != null) {
            try {
                boolean done = false;
                while (!done) {
                    bytesRead = is.read(buffer, 0, buffer.length);
                    if (bytesRead == -1) {
                        done = true;
                        if (verbose) {
                            doLogAction("Gile.copyB: wrote: " + bytesRead + " Total: " + itemTotal);
                        }
                    } else {
                        fos.write(buffer, 0, bytesRead);
                        fos.flush();
                        itemTotal += bytesRead;
                        int megs = (int) (itemTotal / oneMeg);
                        if (megs > oldMegs) {
                            oldMegs = megs;
                            if (verbose) {
                                doLogAction("getFile: wrote: " + bytesRead + " Total: " + itemTotal);
                            }
                        }
                        //seconds = (int) ((double) (now - jobStart) / (double) 1000);
                        int itemPercent = (int) ((double) itemTotal / (double) length * (double) 100);
                        if (pl != null) pl.updateItem( itemPercent);
                    }
                }
                if (pl != null) pl.updateItem(100);
            } catch (IOException ioe) {
                ExceptionHandler.handleIO("File.copyB", "copying", ioe);
            } finally {
                try {
                    fos.close();
                    // SSLClient uses Files.copy, so unsure if socket is closed in that
                } catch (IOException ioe) {
                    ExceptionHandler.handleIO("File.copyB", "closing", ioe);
                }
            }
        } else {
            doLogAction("File.copyB: failure: is: " + is + " fos: " + fos);
        }
    }
    
    /** This method copies a dir into a destination dir.
     * @param srcDir The source directory, wrapped.
     * @param dstDir The destination directory, wrapped
     * 
     */
    
    public static TextFile log;
    public static void copyDir(String srcDir, String dstDir) {
        if (log != null) log.add("File.copyDir: src: " + srcDir + " dst: " + dstDir);
        String root = new File(srcDir).getDeepestDirectory().getParent().getUnwrapped();
        copyDir(root,"",srcDir,dstDir,true); 
        
    }
    
    /** So for this, we want root and subPath unwrapped and both srcDir and dstDir are already
     * I'm copying the contents of srcDir into the contents of dstDir.
     * @param subPath the offset from root
     * @param srcDir the orignal source directory including name and file separator, double quoted
     * @param dstDir  the destination directory including name and file separator, double quoted
     */
    public static void copyDir(String root, String subPath, String srcDir,String dstDir,boolean printDebugOutput) {
        if (!new File(srcDir).exists() && new File(srcDir).isDirectory()) {
            throw new IllegalArgumentException("copyDir: src: " + srcDir + " : must exist and be a directory");
        }

        doLogAction("copyDir: root: " + root + " subPath: " + subPath + " srcDir: " + srcDir + " dstDir: " + dstDir);
        
        File dst = new File(StringTool.getUnwrapped(dstDir) + subPath); 
        dst.mkdirs();
        File[] list = new File(StringTool.getUnwrapped(srcDir) + subPath).listFiles();
        if (list != null) {
            for(int i = 0; i < list.length;i++) {
                if (list[i].isDirectory()) {
                    // subPath will always have a terminating File.separator except
                    // for the first call which will be an empty String -- ""
                    String newSubPath = subPath + list[i].getName() + File.separator;
                    copyDir(root,newSubPath,srcDir,dstDir,printDebugOutput);
                } else {
                    copyA(list[i],dst,printDebugOutput);
                }
            } 
        } 
    }
    
    public static void copyFileToFile(String srcFile, String dstFile) {
        new File(srcFile).copyToFile(dstFile,false);
    }
    
    public static void copyFileToFile(String srcFile, String dstFile, boolean debug) {
        new File(srcFile).copyToFile(dstFile,debug);
    }
    
    public void copyToFile(String newName,boolean output) {
        if (log != null) log.add("File.copyToFile: src: " + getAbsolutePath() + " dst: " + newName);
        if (!isFile()) {
            throw new IllegalArgumentException("File.copyToFile: src must be a file: " + getAbsolutePath());
        }
        File deepest = new File(newName).getDeepestDirectory();
        deepest.mkdirs();
        FileInputStream fis = null;
        DataInputStream dis = null;
        FileOutputStream fos = null;
        DataOutputStream dos = null;
        try {
            fis = new FileInputStream(getJavas());
            dis = new DataInputStream(fis);
        } catch (IOException ioe) {
            doLogAction("File.copyToFile: opening input: " + ioe);
        }
        try {
            if (debug) {
                doLogAction("File.copyToFile: dst: " + deepest.getUnwrapped() + getName());
            }
            fos = new FileOutputStream(StringTool.getUnwrapped(newName));
            dos = new DataOutputStream(fos);
        } catch (IOException ioe) {
            doErrAction("File.copyToFile: opening output: " + ioe);
        }
        if (dis == null | dos == null) {
            doLogAction("File.copyToFile: NULL stream: dis: " + dis + " dos: " + dos);
        } else {
            doCopy(dis, dos);
        }
        try {
            if (dos != null) {
                dos.flush();
            }
            if (dos != null) {
                dos.close();
            }
            if (fos != null) {
                fos.close();
            }
            if (dis != null) {
                dis.close();
            }
            if (fis != null) {
                fis.close();
            }
        } catch (IOException ioe) {
            doErrAction("File.copyToFile: While closing: " + ioe);
        }
    }
    
    public synchronized void copyToDir(String dstPath, boolean output) {
        System.out.println("copyToDir(" + output + ": absPath: " + absPath + " dstPath: " + dstPath);
        File dst = new File(dstPath);
        if (!dst.isDirectory()) {
            throw new IllegalArgumentException("File.copyToDir(a1): dst must be a directory: " + dstPath);
        }
        copyA(this, dst, output);
    }

    public synchronized void copyToDir(String outPath, ProgressListener pl, boolean debug) {
        File dst = new File(outPath);
        if (!isFile()) {
            throw new IllegalArgumentException("File.copyToDir: src must be a file: " + getAbsolutePath());
             
        }
        long total = 0;
        long length = length();
        double portion = (double) length / (double) 100;
        int percent = 0;
        File deepest = dst.getDeepestDirectory();
        deepest.mkdirs();
        FileInputStream fis = null;
        DataInputStream dis = null;
        FileOutputStream fos = null;
        DataOutputStream dos = null;
        try {
            fis = new FileInputStream(getJavas());
            dis = new DataInputStream(fis);
        } catch (IOException ioe) {
            System.err.println("File.copyToDir(b1): opening input: " + ioe);
        }
        try {
            fos = new FileOutputStream(StringTool.getUnwrapped(outPath + getName()));
            dos = new DataOutputStream(fos);
        } catch (Exception e) {
            ExceptionHandler.handleIO("File.copyToDir(b2)","Opening", e);
        }
        if (dis == null | dos == null) {
            System.out.println("File.copyToDir(b3): NULL stream: dis: " + dis + " dos: " + dos);
        } else {
            byte[] buffer = new byte[64000];
            int noOfBytes = 0;
            try {
                while ((noOfBytes = dis.read(buffer)) != -1) {
                    dos.write(buffer, 0, noOfBytes);
                    dos.flush();
                    total += noOfBytes;
                    percent = (int) ((double) total / (double) portion);
                    if (pl != null) pl.updateItem(percent);
                }
            } catch (IOException ioe) {
                System.err.println("File.copyToDir(b4): While copying: " + ioe);
            }
        }
        try {
            if (dos != null) dos.flush();
            if (dos != null) dos.close();
            if (fos != null) fos.close();
            if (dis != null) dis.close();
            if (fis != null) fis.close();
        } catch (IOException ioe) {
            System.err.println("File.copyTo(b5): While closing: " + ioe);
        }
    }

    public synchronized void copyToDir(String outPath, ProgressListener pl) {
        copyToDir(outPath,pl,false);
    }
    
    public void delete() {
        delete(false);
    }

    public synchronized boolean delete(boolean output) {
        boolean deleted = getJavas().delete();
        if (output) {
            System.out.println("File.delete(" + output + "): " + getAbsolutePath() + " : Deleted: " + deleted);
        }
        return deleted;
    }

    public synchronized void deleteAllDirectories() {
        File[] entries = listFiles();
        if (entries != null) {
            for (int i = 0; i < entries.length; i++) {
                if (debug) {
                    System.out.println("deleteAllDirectories: current: " + entries[i].getAbsolutePath());
                }
                if (entries[i].isDirectory()) {
                    prune(entries[i].getUnwrapped(), false);
                    entries[i].delete();
                }
            }
        }
    }

    private synchronized void deleteAllDirectories(java.io.File parentDir) {
        java.io.File[] entries = parentDir.listFiles();
        if (entries != null) {
            for (int i = 0; i < entries.length; i++) {
                if (debug) {
                    System.out.println("deleteAllDirectories: current: " + entries[i].getAbsolutePath());
                }
                if (entries[i].isDirectory()) {
                    deleteAllDirectories(entries[i]);
                    entries[i].delete();
                } else {
                    entries[i].delete();
                }
            }
        }
    }

    public static void doCopy(DataInputStream dis, DataOutputStream dos) {
        byte[] buffer = new byte[64000];
        int noOfBytes = 0;
        try {
            while ((noOfBytes = dis.read(buffer)) != -1) {
                dos.write(buffer, 0, noOfBytes);
                dos.flush();
                if (debug && verbose) {
                    System.out.println("Wrote: " + noOfBytes + " bytes.");
                }
            }
        } catch (IOException ioe) {
            System.err.println("File.doCopy: While copying: " + ioe);
        }
    }

    public static void doErrAction(String msg) {
        if (log != null) log.add(msg);
        System.err.println(msg);
    }

    public static void doLogAction(String msg) {
        
        /* You need to specify your location. */
        
//        TextFile log = ca.tecreations.apps.app.App.log;
//        if (log != null) log.add(msg);
//        System.out.println(msg);
    }

    public synchronized void empty() {
        prune(getAbsolutePath(), false);
        mkdir();
    }

    public boolean exists() {
        return getJavas().exists();
    }

    public String getAbsolutePath() {
        return absPath;
    }

    public String getAppPermissionsString() {
        String s = (canRead() ? "R" : "-");
        s += (canWrite() ? "W" : "-");
        s += (canExecute() ? "X" : "-");
        return s;
    }

    public Path getAsPath() {
        return Paths.get(getUnwrapped());
    }

    public String getBasic(int index) {
        return basicAttributes.get(index);
    }

    public List<String> getBasicAttributes() {
        basicAttributes = new ArrayList<>();
        //if (isFile()) {
        Path nioPath = getAsPath();
        BasicFileAttributes attr;

        try {
            attr = Files.readAttributes(nioPath, BasicFileAttributes.class
            );
            basicAttributes.add("" + attr.creationTime());
            basicAttributes.add("" + attr.lastAccessTime());
            basicAttributes.add("" + attr.lastModifiedTime());
            basicAttributes.add("" + attr.isDirectory());
            basicAttributes.add("" + attr.isOther());
            basicAttributes.add("" + attr.isRegularFile());
            basicAttributes.add("" + attr.isSymbolicLink());
            basicAttributes.add("" + attr.size());
        } catch (IOException ex) {
        }
        //}
        return basicAttributes;
    }

    public boolean getBasicIsDirectory() {
        return basicAttributes.get(basicAttributes.size() - 4).equals("true");
    }

    public long getBasicSize() {
        return Long.parseLong(basicAttributes.get(basicAttributes.size() - 1));
    }

    public List<String> getBasicStats() {
        if (basicStats == null) {
            basicStats = new ArrayList<>();
            basicStats.add(getAbsolutePath());
            basicStats.add((isDirectory()) ? "GET" : "" + length());
            basicStats.add((canRead()) ? "R" : "-");
            basicStats.add((canWrite()) ? "W" : "-");
            basicStats.add((canExecute()) ? "X" : "-");
            basicStats.add(Time.longToYMD("" + getLastModified()));
        }
        return basicStats;
    }

    public byte[] getBytes() {
        if (debug) {
            System.out.println("File.getBytes: Reading file bytes: " + getAbsolutePath());
        }
        int length = (int) (length());
        byte[] bytes = new byte[length];
        FileInputStream fin = null;
        DataInputStream in = null;
        try {
            fin = new FileInputStream(getJavas());
            in = new DataInputStream(fin);
        } catch (FileNotFoundException fnfe) {
            System.err.println("File.getBytes: File not found: " + fnfe);
        }
        try {
            in.readFully(bytes);
        } catch (IOException ioe) {
            System.err.println("File.getBytes: IOException: " + ioe);
        }
        try {
            in.close();
            fin.close();
        } catch (IOException ioe) {
            System.err.println("File.getBytes: Unable to close: " + ioe);
        }
        return bytes;
    }

    public static String getClassNameFrom(String path) {
        String name = new File(path).getName();
        if (name.contains(".")) {
            return name.substring(0, name.lastIndexOf("."));
        }
        return name;
    }

    public File getDeepestDirectory() {
        String unwrapped = getUnwrapped();
        if (unwrapped.endsWith("/") || unwrapped.endsWith("\\")) {
            return this; // this File represents a directory, so it will be deepest
        }
        if (unwrapped.contains("/")) {  
            // a file on a POSIX Compliant System?
            String path = unwrapped.substring(0, unwrapped.lastIndexOf("/") + 1);
            return new File(path);
        } else {
            // windows
            String path = unwrapped.substring(0, unwrapped.lastIndexOf("\\") + 1);
            return new File(path); 
        }
    }

    public String getDeepestDirectoryName() {
        File deepest = getDeepestDirectory();
        String tmp = deepest.getUnwrapped();
        String sep = tmp.contains("/") ? "/" : "\\";
        String prefixPath = tmp.substring(0,tmp.lastIndexOf(sep));
        String deepestDir = prefixPath.substring(prefixPath.lastIndexOf(sep) + 1);
        return deepestDir;
    }

    public static List<File> getDirs(String absPath) {
        File[] entries = new File(absPath).listFiles();
        List<File> dirs = new ArrayList<>();
        if (entries != null) {
            for (File entry : entries) {
                if (entry.isDirectory()) {
                    dirs.add(entry);
                }
            }
        }
        File[] dirsArray = new File[dirs.size()];
        for (int i = 0; i < dirs.size(); i++) {
            dirsArray[i] = dirs.get(i);
        }
        List<String> sorted = Sort.getSortedByName(dirsArray);
        List<File> result = new ArrayList<>();
        for (int i = 0; i < sorted.size(); i++) {
            result.add(new File(sorted.get(i)));
        }
        return result;
    }

    public static long getDirSize(String path) {
        long size = 0L;
        File[] list = new File(path).listFiles();
        if (list != null) {
            for (int i = 0; i < list.length; i++) {
                if (list[i].isDirectory()) {
                    long sizeToAdd = getDirSize(list[i].getAbsolutePath());
                    // here, you could add the OperatingSystems::FileSytem::Label::SIZE_FOR_DIRECTORY_IN_BYTES
                    size += sizeToAdd;
                } else {
                    size += list[i].getLength();
                }
            }
        }
        return size;
    }

    public DosFileAttributes getDOSAttributes() {
        Path file = Paths.get(StringTool.getUnwrapped(getAbsolutePath()));
        DosFileAttributes attr = null;

        try {
            attr = Files.readAttributes(file, DosFileAttributes.class
            );
        } catch (IOException ioe) {
            System.out.println("While attempting to get attributes: " + ioe);
        } catch (UnsupportedOperationException y) {
            System.err.println("DOS file attributes not supported:" + y);
        }
        return attr;
    }

    public String getDOSAttributesString() {
        DosFileAttributes attr = getDOSAttributes();
        String s = (attr.isReadOnly() ? "R" : "-");
        s += (attr.isHidden() ? "H" : "-");
        s += (attr.isSystem() ? "S" : "-");
        s += (attr.isArchive() ? "A" : "-");
        return s;
    }

    public String getExtension() {
        String name = getName();
        //System.err.println("File.getExtension: " + name);
        if (name.indexOf(".") == -1) {
            return "";
        } else if (name.lastIndexOf(".") > 0) {
            String ext = name.substring(name.lastIndexOf(".") + 1);
            //System.err.println("File.getExtension: name: " + name + " ext: " + ext);
            return ext;
        } else {
            return "";
        }
    }

    public static String getExtension(String path) {
        File f = new File(path);
        return f.getExtension();
    }

    public String getExtensionLower() {
        return getExtension().toLowerCase();
    }

    public static String getExtensionLower(String path) {
        File f = new File(path);
        return f.getExtensionLower();
    }

    public static String getExtensionLower(File file) {
        String unwrapped = file.getUnwrapped();
        return unwrapped.substring(unwrapped.lastIndexOf(".") + 1).toLowerCase();
    }

    public InputStream getFileInputStream() {
        FileInputStream in = null;
        try {
            in = new FileInputStream(getUnwrapped());
        } catch (FileNotFoundException fnfe) {
            System.err.println("File not found: " + getAbsolutePath());
        }
        return in;
    }

    public List<String> getFileLines() {
        return readFileLines(getAbsolutePath());
    }
    
    /**
     *
     * @return The files name, excluding the file extension.
     */
    public String getFilenameOnly() { // for convenience, make it intuitive and easy
        return getFileNameOnly();
    }

    /**
     *
     * @return The files name, excluding the file extension.
     */
    public String getFileNameOnly() {
        String name = getName();
        if (name.contains(".")) {
            name = name.substring(0, name.lastIndexOf("."));
        }
        return name;
    }

    public static List<File> getFiles(String absPath) {
        //System.out.println("File.getFiles(" + absPath + ")");
        List<File> files = new ArrayList<>();
        File[] entries = new File(absPath).listFiles();
        File entry;
        if (entries != null) {
            for (int i = 0; i < entries.length; i++) {
                if (entries[i].isFile()) {
                    files.add(entries[i]);
                }
            }
        }
        File[] filesArray = new File[files.size()];
        for (int i = 0; i < files.size(); i++) {
            filesArray[i] = files.get(i);
        }
        List<String> sorted = Sort.getSortedByName(filesArray);
        List<File> result = new ArrayList<>();
        for (int i = 0; i < sorted.size(); i++) {
            result.add(new File(sorted.get(i)));
        }
        return result;
    }

    public static List<File> getFiles(String absPath, String ext) {
        //System.out.println("File.getFiles(" + absPath + ")");
        List<File> files = new ArrayList<>();
        File[] entries = new File(absPath).listFiles();
        File entry;
        if (entries != null) {
            for (int i = 0; i < entries.length; i++) {
                if (entries[i].isFile() && entries[i].getExtension().equals(ext)) {
                    files.add(entries[i]);
                }
            }
        }
        File[] filesArray = new File[files.size()];
        for (int i = 0; i < files.size(); i++) {
            filesArray[i] = files.get(i);
        }
        List<String> sorted = Sort.getSortedByName(filesArray);
        List<File> result = new ArrayList<>();
        for (int i = 0; i < sorted.size(); i++) {
            result.add(new File(sorted.get(i)));
        }
        return result;
    }

    public long getFreeSpace() {
        return getJavas().getFreeSpace();
    }
    
    public String getFQCN(String path) {
        String fqcn = "";
        String s = getUnwrapped().substring(path.length());
        if (s.contains(".")) {
            fqcn = s.substring(0,s.lastIndexOf("."));
        }
        fqcn = StringTool.replaceAll(fqcn, File.separatorChar,".");
        return fqcn;
    }
    
    public String getGroup() {
        return getPOSIXGroup();
    }

    public java.io.File getJavas() {
        return new java.io.File(getUnwrapped());
    }

    public long getLastModified() {
        return getJavas().lastModified();
    }
    
    public FileTime getNIOLastModified() {
        FileTime fileTime = null;
        try {
            fileTime = Files.getLastModifiedTime(Paths.get(getUnwrapped()));
        } catch (IOException ioe) {
            System.err.println("File.getNIOLastModified: Unable: " + getUnwrapped() + " : " + ioe);
        }
        return fileTime;
    }

    public String getTecLastModified() {
        Long dateTime = getLastModified();
//        connection.disconnect();
        ZonedDateTime modified = ZonedDateTime.ofInstant(Instant.ofEpochMilli(dateTime), ZoneId.of("GMT"));   

        DateTimeFormatter dateFormatter = 
                DateTimeFormatter.ofPattern("yyyy-MM-dd");
        DateTimeFormatter timeFormatter = 
                DateTimeFormatter.ofPattern("HH:mm:ss:S");
        DateTimeFormatter zoneFormatter = 
                DateTimeFormatter.ofPattern("Z");

        return "D" + modified.format(dateFormatter) + "T" + modified.format(timeFormatter) + "Z" + modified.format(zoneFormatter);
    }
    
    public long getLength() {
        return length(); // ca.tecreations.File.length() == getJavas().length();
    }

    public String getMinusExtension() {
        String unwrapped = getUnwrapped();
        String upToEndOfName = unwrapped.substring(0,unwrapped.lastIndexOf("."));
        return upToEndOfName;
    }
    
    public String getName() {
        String name;
        String unwrapped = getUnwrapped();
        // if it ends with a FileSeparator, return the deepest directory name
        if (unwrapped.contains("\\")) {
            if (unwrapped.endsWith("\\")) {
                String deepest = unwrapped.substring(0,unwrapped.length() - 1);
                name = deepest.substring(deepest.lastIndexOf("\\") + 1);
            } else {
                name = unwrapped.substring(unwrapped.lastIndexOf("\\") + 1);
            }
        } else if (unwrapped.contains("/")) {
            if (unwrapped.endsWith("/")) {
                if (unwrapped.length() > 1) {
                    String deepest = unwrapped.substring(0,unwrapped.length() - 1);
                    name = deepest.substring(deepest.lastIndexOf("/") + 1);
                } else {
                    name = "";
                }
            } else {
                name = unwrapped.substring(unwrapped.lastIndexOf("/") + 1);
            }
        } else {
            name = "File.getName: ??? : " + getAbsolutePath();
        }
        return name;
    }

    public static File getNewestDirectory(String path) {
        File[] list = new File(path).listFiles();
        if (list != null && list.length > 0) {
            File found = null;
            File target = null;
            for (int i = 0; i < list.length; i++) {
                target = list[i];
                if (target.isDirectory()) {
                    if (found == null) {
                        found = list[i];
                    } else {
                        target = list[i];
                        if (target.lastModified() > found.lastModified()) {
                            found = target;
                        }
                    }
                }
            }
            return found;
        } else {
            return null;
        }
    }

    public static File getNewestFile(String path) {
        File[] list = new File(path).listFiles();
        if (list != null && list.length > 0) {
            File found = null;
            File target = null;
            for (int i = 0; i < list.length; i++) {
                target = list[i];
                if (target.isFile()) {
                    if (found == null) {
                        found = list[i];
                    } else {
                        target = list[i];
                        if (target.lastModified() > found.lastModified()) {
                            found = target;
                        }
                    }
                }
            }
            return found;
        } else {
            return null;
        }
    }

    /**
     * Conform this in such a way that it receives the destination directory and,
     * if necessary, constructs a new directory such that the pattern dir, dir (1), 
     * dir (2), etc is fulfilled.
     * @param path the output path for which we may create a new path name.
     * @return the new path name, if any, otherwise, the non-existent path.
     */
    public static String getNextDirectoryName(String path) {
        File f = new File(path);
        if (f.exists()) {
            File parent = f.getParent();
            String name = f.getFilenameOnly();
            String ext = f.getExtension();
            if (name.contains(" (") && name.contains(")")) {
                int start = name.lastIndexOf(" (") + 2;
                int end = name.lastIndexOf(")");
                String indexString = name.substring(start, end);
                String nameString = name.substring(0, name.lastIndexOf("(") + 1);
                int index = Integer.valueOf(indexString) + 1;
                String newName = parent.getUnwrapped() + nameString + index + ")";
                if (!ext.equals("")) {
                    newName += "." + ext;
                }
                return getNextDirectoryName(newName);
            } else {
                String newName = parent.getUnwrapped() + name + " (1)";
                if (!ext.equals("")) {
                    newName += "." + ext;
                }
                if (new File(newName).exists()) {
                    return getNextDirectoryName(newName);
                } else {
                    return (ext.equals("") ? newName + File.separator : newName);
                }
            }
        }
        String unwrapped = f.getUnwrapped();
        if (unwrapped.contains("\\") && ! unwrapped.endsWith("\\")) unwrapped += "\\";
        else if (unwrapped.contains("/") && !unwrapped.endsWith("/")) unwrapped += "/";
        return StringTool.getDoubleQuoted(unwrapped);
    }

    public static String getNextFileName(String path) {
        File f = new File(path);
        if (f.isDirectory()) {
            return getNextDirectoryName(path);
        }
        File deepest = f.getDeepestDirectory();
        String name = f.getFilenameOnly();
        String ext = f.getExtension();
        if (f.exists()) {
            if (name.contains(" (") && name.contains(")")) {
                int start = name.lastIndexOf(" (") + 2;
                int end = name.lastIndexOf(")");
                String indexString = name.substring(start, end);
                //System.out.println("IndexString: '" + indexString + "'");
                String nameString = name.substring(0, name.lastIndexOf("(") + 1);
                //System.out.println("NameString : '" + nameString + "'");
                int index = Integer.valueOf(indexString) + 1;
                String newName = deepest.getUnwrapped() + nameString + index + ")";
                if (!ext.equals("")) {
                    newName += "." + ext;
                }
                return getNextFileName(newName);
            } else {
                String newName = deepest.getUnwrapped() + name + " (1)";
                if (!ext.equals("")) {
                    newName += "." + ext;
                }
                return getNextFileName(newName);
            }
        }
        return f.getAbsolutePath();
    }

    public static File getOldestDirectory(String path) {
        FileTime ft1 = null;
        FileTime ft2 = null;
        File[] list = new File(path).listFiles();
        if (list != null && list.length > 0) {
            File found = null;
            File target = null;
            //System.out.println("File: " + found);
            for (int i = 0; i < list.length; i++) {
                //System.out.println("File: " + list[i]);
                target = list[i];
                if (target.isDirectory()) {
                    if (found == null) {
                        found = list[i];
                    } else {
                        try {
                            ft1 = Files.getLastModifiedTime(Paths.get(found.getUnwrapped()), LinkOption.NOFOLLOW_LINKS);
                            ft1.to(TimeUnit.DAYS);
                            ft2 = Files.getLastModifiedTime(Paths.get(target.getUnwrapped()), LinkOption.NOFOLLOW_LINKS);
                            //System.out.println(ft1.toString());
                            //System.out.println(ft2.toString());
                        } catch (IOException ioe) {
                        }
                        if (ft1 != null && ft2 != null) {
                            long l = ft2.to(TimeUnit.DAYS);
                            //System.out.println(l + ": " + (ft2.compareTo(ft1)));
                            if (ft2.compareTo(ft1) == -1) {
                                found = target;
                            }
                        }
                    }
                }
            }
            return found;
        } else {
            return null;
        }
    }

    public static File getOldestFile(String path) {
        File[] list = new File(path).listFiles();
        List<File> files = new ArrayList<>();
        if (list != null && list.length > 0) {
            for (int i = 0; i < list.length; i++) {
                if (list[i].isFile()) {
                    files.add(list[i]);
                }
            }
        }
        File oldest = null;
        if (files.size() == 0) ; else if (files.size() == 1) {
            return files.get(0);
        } else {
            oldest = files.get(0);
            for (int i = 1; i < files.size(); i++) {
                if (files.get(i).lastModified() < oldest.lastModified()) {
                    oldest = files.get(i);
                }

            }
        }
        return oldest;
    }

    public String getOwner() {
        return getPOSIXGroup();
    }

    public String getPackagePart() {
        List<String> lines = getFileLines();
        String line;
        for(int i = 0; i < lines.size();i++) {
            line = lines.get(i).trim();
            if (line.startsWith("package")) {
                return line.substring("package".length(),line.indexOf(";")).trim();
            }
        }
        return "";
    }
    
    public File getParent() {
        String unwrapped = getUnwrapped();
        boolean isNix = unwrapped.contains("/");
        String deepest = getDeepestDirectory().getUnwrapped();
        String parent = "";
        if (deepest.endsWith("/") || deepest.endsWith("\\")) {
            parent = deepest.substring(0,deepest.length() - 1);
            if (isNix) {
                parent = parent.substring(0,parent.lastIndexOf("/") + 1);
            } else {
                parent = parent.substring(0,parent.lastIndexOf("\\") + 1);
            }
        }
        if (parent.equals("")) {
            if (File.separator.equals("\\")) {
                parent = new java.io.File(".").getAbsolutePath().substring(0,3);
            } else {
                parent = "/";
            }
        }
        return new File(parent);
    }

    public String getParentDirectoryName() {
        return getParent().getName();
    }

    public String getPathWithoutExtensionUnquoted() {
        String unwrapped = getUnwrapped();
        String withoutExtension = unwrapped.substring(0, unwrapped.indexOf("."));
        return withoutExtension;
    }

    public static String getPermissionString(int val) {
        String permission = "";
        BitSet bitSet = new BitSet(val);
        permission += (bitSet.isSet(0) ? "R" : "-");
        permission += (bitSet.isSet(1) ? "W" : "-");
        permission += (bitSet.isSet(2) ? "X" : "-");
        return permission;
    }

    public PosixFileAttributes getPOSIXAttributes() {
        Path file = getAsPath();
        PosixFileAttributes attr = null;

        try {
            attr = Files.readAttributes(file, PosixFileAttributes.class
            );
        } catch (IOException ioe) {
            System.out.println("Couldn't retrieve POSIX attributes for: " + getAbsolutePath());
        }
        return attr;
    }

    public String getPOSIXAttributesCSV() {
        PosixFileAttributes attr = getPOSIXAttributes();
        return attr.owner().getName() + "," + attr.group().getName() + "," + PosixFilePermissions.toString(attr.permissions());
    }

    public String getPOSIXOwner() {
        return getPOSIXAttributes().owner().getName();
    }

    public String getPOSIXGroup() {
        return getPOSIXAttributes().group().getName();
    }

    public String getPOSIXPermissions() {
        return PosixFilePermissions.toString(getPOSIXAttributes().permissions());
    }

    public static String getPOSIXPermissionString(int owner, int group, int other) {
        String permissions = "";
        permissions += getPermissionString(owner);
        permissions += getPermissionString(group);
        permissions += getPermissionString(other);
        return permissions;
    }

    public String getRoot() {
        String unwrapped = getUnwrapped();
        if (unwrapped.startsWith("/")) {
            return "/";
        } else {
            return unwrapped.substring(0, 3);
        }
    }

    public char getRootChar() {
        return getUnwrapped().charAt(0);
    }

    public String getRootOf(String path) {
        String unwrapped = StringTool.getUnwrapped(path);
        if (unwrapped.startsWith("/")) {
            return "/";
        } else {
            return unwrapped.substring(0, 3);
        }
    }

    public long getSize(String caller) {
        if (isFile()) {
            return length();
        } else {
            System.out.println(caller + " : File.getSize() : " + getAbsolutePath());
            return getSize(caller,getAbsolutePath());
        }
    }

    public static long getSize(String caller, String path) {
        File target = new File(path);
        if (debug) {
            System.out.println(caller + " : File.getSize(" + path + ") : " + target.isFile());
        }
        if (target.isFile()) {
            return target.length();
        } else {
            long size = 0L;
            File[] entries = target.listFiles();
            if (entries != null) {
                for (int i = 0; i < entries.length; i++) {
                    if (entries[i].isFile()) {
                        size += entries[i].getLength();
                    } else {
                        size += getDirSize(entries[i].getUnwrapped());
                    }
                }
            }
            return size;
        }
    }

    public static List<File> getSorted(List<File> list) {
        List<String> names = new ArrayList<>();
        for (int i = 0; i < list.size(); i++) {
            names.add(list.get(i).getAbsolutePath());
        }
        names = Sort.sort(names);
        List<File> files = new ArrayList<>();
        for (int i = 0; i < names.size(); i++) {
            files.add(new File(names.get(i)));
        }
        return files;
    }

    public static List<File> getSubDirs(String path) {
        File[] entries = new File(path).listFiles();
        List<File> retValue = new ArrayList<>();
        if (entries != null) {
            for (int i = 0; i < entries.length; i++) {
                if (entries[i].isDirectory()) {
                    retValue.add(entries[i]);
                    List<File> subs = getSubDirs(entries[i].getAbsolutePath());
                    for (int j = 0; j < subs.size(); j++) {
                        retValue.add(subs.get(j));
                    }
                }
            }
        }
        return retValue;
    }

    /**
     * gets a subpath component for the current Platform
     * @param src
     * @return anything trailing after /, or \, or a fqcn to File.separator or
     *         the src components full value, minus any leading '/' or '\'
     *         sample: \path\to\dir\ returns path\to\dir\ on windows and path/to/dir/ on nix
     *         sample: someUnknownString returns someUnknownString ?+= File.separator
     *         sample: ca.tecreations.apps returns ca\tecreations\apps\ on Windows, and ca/tecreations/apps/ on nix
     */
    public static String getSubPath(String src) {
        boolean prefixed = false;
        char prefix;
        String separator;
        String subPart = src;
        if (src.startsWith("/") || src.startsWith("\\")) {
            prefixed = true;
            prefix = src.charAt(0);
            subPart = src.substring(1);
        } 
        boolean doForDot = false;
        boolean doForSlash = false;
        boolean doForBackSlash = false;
        if (subPart.contains(".") && !subPart.contains("\\") && !subPart.contains("/")) {
            doForDot = true;
        } else if (subPart.contains("\\")) {
            doForBackSlash = true;
        } else {
            doForSlash = true;
        }
        String nextPart = subPart;
        if (doForDot) {
            nextPart = StringTool.replaceAll(subPart,".",File.separator);
        } else if (doForBackSlash) {
            nextPart = StringTool.replaceAll(subPart,"\\", File.separator);
        } else if (doForSlash) {
            nextPart = StringTool.replaceAll(subPart,"/",File.separator);
        }
        if (!nextPart.endsWith(File.separator)) {
            nextPart += File.separator; // always return a subPath type String
        }
        return nextPart;
    }
    
    public static String getSubPath(String src, String root) {
        File srcDir = new File(src);
        if (!srcDir.isDirectory()) {
            throw new IllegalArgumentException("src is not a directory");
        }
        File rootDir = new File(root);
        if (!rootDir.isDirectory()) {
            throw new IllegalArgumentException("root is not a directory");
        }
        if (!src.endsWith(separator)) {
            src += separator;
        }
        if (!root.endsWith(separator)) {
            root += separator;
        }
        String subPath = src.substring(root.length());
        return subPath;
    }

    public static String getSubPath(File file, String root) {
        if (!file.isFile()) {
            throw new IllegalArgumentException("Specified ca.tecreations.File is a directory -- must be a file in call to getSubPath(File,String root)");
        }
        File rootDir = new File(root);
        if (!rootDir.isDirectory()) {
            throw new IllegalArgumentException("root is not a directory");
        }
        // unknown path, unwrap if necessary
        root = StringTool.getUnwrapped(root);
        // unknown path, so conform to tec style
        if (!root.endsWith(separator)) {
            root += separator;
        }
        String unwrappedFilePath = file.getUnwrapped();
        String fromRoot = unwrappedFilePath.substring(root.length());
        String subPath = "";
        if (fromRoot.length() > 0) {
            if (fromRoot.contains(separator)) {
                subPath = fromRoot.substring(0, fromRoot.lastIndexOf(separator) + 1); // always take the separator
            } else {
                subPath = "";
            }
            if (subPath.startsWith(separator)) {
                subPath = subPath.substring(1);
            }
        }
        if (!subPath.endsWith(separator) && subPath.length() > 0) {
            subPath += separator;
        }
        return subPath;
    }

    public static List<String> getTextFile(String absPath) {
        TextFile textFile = new TextFile(StringTool.getUnwrapped(absPath)); // for all the 3rd party
        // calls to outside classes, probably you should
        // unwrap in case they don't or the code changes
        return textFile.getLines();
    }

    public static List<String> getTextFile(File file) {
        TextFile textFile = new TextFile(file.getUnwrapped()); // TextFile also unwraps
        return textFile.getLines();
    }

    public long getTotalSpace() {
        return getJavas().getTotalSpace();
    }
    
    public String getUnwrapped() {
        absPath = absPath.trim();
        // this is very bad.... it could be multipl-y wrapped ? ""something""?
        String unwrapped = StringTool.getUnwrapped(absPath);
        while (unwrapped.startsWith("\"") && unwrapped.endsWith("\"")) {
            //System.out.println("File.getUnwrapped: " + unwrapped);
            unwrapped = StringTool.getUnwrapped(unwrapped);
        }
        return unwrapped;
    }

    public Long getUsableSpace() {
        if (separator.equals("/")) {
            return new java.io.File("/").getUsableSpace();
        } else {
            return getJavas().getUsableSpace();
        }
    }

    public boolean hasMatchingClass() {
        File aClassFile = new File(getDeepestDirectory().getUnwrapped() + getFileNameOnly() + ".class");
        return aClassFile.exists();
    }
    
    public boolean hasMain() {
        List<String> lines = getFileLines();
        String line;
        for(int i = 0; i < lines.size();i++) {
            line = lines.get(i).trim();
            if (line.startsWith("public static void main")) {
                return true;
            }
        }
        return false;
    }
    
    public static String getPkgPathPart(String pkg) {
        if (pkg.equals("")) return "";
        else if (pkg.contains(".") && !pkg.endsWith(".")) { // should never happen
            return StringTool.replaceAll(pkg + ".",".",File.separator);
        } else {
            return pkg + File.separator; // just a single package, like 'any'
        }
    }

    public String getProjectPath() {
        String pkg = getPackagePart();
        String pkgPathPart = getPkgPathPart(pkg);
        String projectPath = "";
        if (getAbsolutePath().contains(pkgPathPart)) {
            String unwrapped = getUnwrapped();
            projectPath = unwrapped.substring(0,unwrapped.indexOf(pkgPathPart));
        } else {
            projectPath = getDeepestDirectory().getAbsolutePath();
        }
        return StringTool.getDoubleQuoted(projectPath);
    }

    public boolean isC() {
        return getExtension().equals("c");
    }
    
    public boolean isClass() {
        return getExtension().equals("class");
    }
    
    public boolean isCPP() {
        return getExtension().equals("cpp");
    }
    
    public boolean isDirectory() {
        return getJavas().isDirectory();
    }

    public boolean isFile() {
        return getJavas().isFile();
    }

    public boolean isHTML() {
        return getExtension().equals("html");
    }
    
    public boolean isJava() {
        return getExtension().equals("java");
    }
    
    public boolean isJar() {
        return getExtension().equals("jar");
    }
    
    /**
     * Checks the given string for any invalid characters for dirs/files on the
     * target platform.
     *
     * @param separator the target platform's separator
     * @param item The dir/file name to check for illegal characters
     * @return The invalid character KEYCODE, if found, otherwise -1, not found.
     */
    public static int isValidFileNameChars(String separator, String item) {
        String linuxIllegal = "?~!&*-;:'\"<>/\\|\t\r\n";
        String winIllegal = "/\\;*?\"<>|";
        if (separator.equals("/")) {
            for (int i = 0; i < item.length(); i++) {
                for (int j = 0; j < linuxIllegal.length(); j++) {
                    if (item.charAt(i) == linuxIllegal.charAt(j)) {
                        return linuxIllegal.charAt(j);
                    }
                }
            }
            return -1;
        } else {
            for (int i = 0; i < item.length(); i++) {
                for (int j = 0; j < winIllegal.length(); j++) {
                    if (item.charAt(i) == winIllegal.charAt(j)) {
                        return winIllegal.charAt(i);
                    };
                }
            }
            return -1;
        }
    }

    public boolean isZip() {
        return getExtension().equals("zip");
    }
    
    public long lastModified() {
        return getJavas().lastModified();
    }

    public long length() {
        if (!exists()) {
            return -1;
        }
        if (isDirectory()) return getDirSize(getAbsolutePath());
        return getJavas().length();
    }

    public File[] listFiles() {
        java.io.File[] list = getJavas().listFiles();
        if (list != null) {
            ca.tecreations.File[] output = new File[list.length];
            for (int i = 0; i < list.length; i++) {
                output[i] = new ca.tecreations.File(list[i]); // here, let's be explicit intentionally
            }
            return output;
        }
        return new File[0];
    }

    public static File[] listRoots() {
        java.io.File[] roots;
        if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
            roots = java.io.File.listRoots();
            File[] list = new File[roots.length]; // convert to tecreations
            for (int i = 0; i < roots.length; i++) {
                list[i] = new File(roots[i].getAbsolutePath());
            }
            return list;
        } else {
            return new File[]{new File("/")}; // convert and return
        }
    }

    public static void main(String[] args) {

        File f1 = new File("/projects/tec8/any/");
        System.out.println("AbsPath: " + f1.getAbsolutePath());
        System.out.println("Name   : " + f1.getName());
        System.out.println("NameOnly: " + f1.getFilenameOnly());

        File f2 = new File("Executable.java");
        System.out.println("AbsPath: " + f2.getAbsolutePath());
        System.out.println("Name   : " + f2.getName());
        System.out.println("NameOnly: " + f2.getFilenameOnly());

        File f3 = new File("/projects/tec8/any/Executable.java");
        System.out.println("AbsPath: " + f3.getAbsolutePath());
        System.out.println("Name   : " + f3.getName());
        System.out.println("NameOnly: " + f3.getFilenameOnly());

        File f4 = new File("F:\\projects\\tec8\\any\\Executable.java");
        System.out.println("AbsPath: " + f4.getAbsolutePath());
        System.out.println("Name   : " + f4.getName());
        System.out.println("NameOnly: " + f4.getFilenameOnly());
        System.exit(0);
        
        //System.out.println("Package: " + f.getPackagePart());
        //System.out.println("PkgPath: " + getPkgPathPart(f.getPackagePart()));
        //System.out.println("ProjectPath: " + f.getProjectPath());
        System.exit(0);
        
        
        copyDir("\"F:\\GET_TEST\\dropin\\\"","\"F:\\GET_TEST\\timTest\\\"");
        copyDir("\"F:\\GET_TEST\\dropin\\\"","\"F:\\GET_TEST\\tomTest\\\"");
        System.out.println("Equal: " + (new File("\"F:\\GET_TEST\\dropin\\\"").length() == new File("\"F:\\GET_TEST\\timTest\\\"").length()));
        System.out.println("Equal: " + (new File("\"F:\\GET_TEST\\dropin\\\"").length() == new File("\"F:\\GET_TEST\\tomTest\\\"").length()));
        System.out.println("Equal: " + (new File("\"F:\\GET_TEST\\timTest\\\"").length() == new File("\"F:\\GET_TEST\\tomTest\\\"").length()));
        System.exit(0);
        
        
        
        System.out.println("test\\".charAt("test\\".length() - 1) == '\\');
        System.out.println("c:\\test".substring(0,3));
        
        File nix = new File("/home/tim/timtest/put.into");
        System.out.println("deepest: " + nix.getDeepestDirectory().getAbsolutePath());
        System.out.println("d.parent: " + nix.getDeepestDirectory().getParent().getAbsolutePath());
        System.exit(0);
        
        File test = new File("f:\\test\\test.txt");
        System.out.println(test.getName());
        System.exit(0);
        
        File file = new File("F:\\testfile");
        String next = getNextFileName(file.getAbsolutePath());
        System.out.println("Got: " + next);
        if (!new File(next).exists()) {
            TextFile tf = new TextFile(next);
            tf.add(next);
        }

        File dir = new File("F:\\test\\");
        String nextDir = getNextDirectoryName(dir.getAbsolutePath());
        System.out.println("Got: " + nextDir);
        if (!new File(nextDir).exists()) {
            new File(nextDir).mkdirs();
        }

        File file2 = new File("F:\\testfile.txt");
        String next2 = getNextFileName(file2.getAbsolutePath());
        System.out.println("Got: " + next2);
        if (!new File(next2).exists()) {
            TextFile tf = new TextFile(next2);
            tf.add(next2);
        }

        File dir2 = new File("F:\\test.dir\\");
        String nextDir2 = getNextDirectoryName(dir2.getAbsolutePath());
        System.out.println("Got: " + nextDir2);
        if (!new File(nextDir2).exists()) {
            new File(nextDir2).mkdirs();
        }
    }

    public static boolean makeParentDirectory(String path) {
        String unwrapped = StringTool.getUnwrapped(path);
        return new java.io.File(unwrapped).mkdirs();
    }

    public boolean mkdir() {
        getJavas().mkdir();
        return exists();
    }

    public static boolean mkdir(String path) {
        return new File(path).mkdir();
    }
    
    public boolean mkdirs() {
        getJavas().mkdirs();
        return exists();
    }

    public boolean mkdirs(boolean debug) {
        getJavas().mkdirs();
        boolean exists = exists();
        if (debug) System.out.println("Exists: " + getAbsolutePath() + " : " + exists);
        return exists;
    }

    public static boolean mkdirs(String path) { 
        return new File(path).mkdirs();
    }
    
    public synchronized static void moveDir(String src, String dst) {
        replicate(src, dst, false);
        prune(src, false);
    }

    public void printBasicAttributes() {
        List<String> atts = getBasicAttributes();
        for (int i = 0; i < atts.size(); i++) {
            System.out.print(atts.get(i));
            if (i < atts.size() - 1) {
                System.out.print(",");
            }
        }
    }

    public void printDOSAttributes() {
        System.out.print(getDOSAttributesString());
    }

    public void printPOSIXAttributes() {
        System.out.print(getPOSIXOwner() + ",");
        System.out.print(getPOSIXGroup() + ",");
        System.out.println(getPOSIXPermissions());
    }

    public static void prune(String path, boolean output) {
        File target = new File(path);
        File[] list = target.listFiles();
        if (list != null) {
            for (int i = 0; i < list.length; i++) {
                if (list[i].isDirectory()) {
                    prune(list[i].getAbsolutePath(), output);
                }
            }
            for (int i = 0; i < list.length; i++) {
                if (list[i].isFile()) {
                    list[i].delete(output);
                }
            }
        }
        target.delete();
    }

    public List<String> readLines() {
        return readFileLines(getAbsolutePath());
    }

    public static List<String> readFileLines(String absPath) {
        TextFile file = new TextFile(StringTool.getUnwrapped(absPath));
        return file.getLines();
    }

    public static void replicate(String srcPath, String dstPath, boolean detailed) {
        File src = new File(srcPath);
        File dst = new File(dstPath);
        if (!src.isDirectory()) {
            throw new IllegalArgumentException("File.replicate: src must be a directory");
        }
        if (!dst.isDirectory()) {
            throw new IllegalArgumentException("File.replicate: dst must be a directory");
        }
        dst.mkdirs();
        File[] entries = src.listFiles();
        String srcUnwrapped = src.getUnwrapped();
        String dstUnwrapped = dst.getUnwrapped();
        if (entries != null && entries.length > 0) {
            for (int i = 0; i < entries.length; i++) {
                File entry = entries[i];
                if (entry.isDirectory()) {
                    // copy the directory to the appropriate sub path in destination
                    String dirName = entry.getUnwrapped().substring(srcUnwrapped.length());
                    replicateDirectory(srcUnwrapped, dstUnwrapped, dirName, detailed);
                } else {
                    // copy file to dstPath
                    String name = entry.getUnwrapped().substring(srcUnwrapped.length());
                    copyA(entry, new File(dstUnwrapped + name), detailed);
                }
            }
        }
    }

    public static void replicateDirectory(String srcPath, String dstPath, String offset, boolean detailed) {
        File src = new File(srcPath);
        File dst = new File(dstPath);
        if (!src.isDirectory()) {
            throw new IllegalArgumentException("File.replicateDirectory: src must be a directory");
        }
        if (!dst.isDirectory()) {
            throw new IllegalArgumentException("File.replicateDirectory: dst must be a directory");
        }

        //System.out.println("Platform.replicateDirectory(" + srcPath + "," + dstPath + "," + offset + ")");
        // because this method can be called from anywhere and we don't know the state of the variables, 
        // we want to unwrap and do a sanity check for consistency with the tecreations package
        if (!offset.endsWith(separator)) {
            offset += separator;
        }

        if (detailed) {
            String msg = "Replicating directory: " + src.getAbsolutePath() + " to: " + dst.getAbsolutePath();
            System.out.println(msg);
        }

        new File(dst.getUnwrapped() + offset).mkdirs();
        String srcUnwrapped = src.getUnwrapped();
        File[] list = new File(srcUnwrapped + offset).listFiles();
        if (list != null && list.length > 0) {
            for (int i = 0; i < list.length; i++) {
                if (list[i].isDirectory()) {
                    String newOffset = list[i].getAbsolutePath().substring(srcUnwrapped.length());
                    //System.out.println("NewOffset: " + newOffset);
                    replicateDirectory(srcUnwrapped, dst.getUnwrapped(), newOffset, detailed);
                } else {
                    // copy file to dstPath+offset
                    String name = list[i].getAbsolutePath().substring((srcUnwrapped + offset).length() + 1); // be consistent, take the separator
                    //System.out.println("For copyFile: Name: " + name);
                    copyA(list[i], new File(dst.getUnwrapped() + offset + name), detailed);
                }
            }
        }
    }

    public static void removeFirstLine(File file) {
        List<String> lines = getTextFile(file);
        PrintWriter out = null;
        try {
            out = new PrintWriter(new FileWriter(file.getUnwrapped()), true);
        } catch (IOException ioe) {
            System.out.println("Unable to open file for writing: " + file.getAbsolutePath());
        }
        for (int i = 1; i < lines.size(); i++) {
            out.println(lines.get(i));
        }
        out.close();
    }

    public static void removeFirstLine(String path, String extension, boolean recurse) {
        File[] list = new File(path).listFiles();
        for (int i = 0; i < list.length; i++) {
            if (list[i].isDirectory() && recurse) {
                removeFirstLine(list[i].getAbsolutePath(), extension, recurse);
            } else {
                if (list[i].isFile()) {
                    if (getExtension(list[i].getAbsolutePath()).equals(extension)) {
                        List<String> lines = getTextFile(list[i]);
                        List<String> out = new ArrayList<>();
                        for (int j = 1; j < lines.size(); j++) {
                            out.add(lines.get(j));
                        }
                        writeTextFile(list[i], out);
                    }
                }
            }
        }
    }

    public void renameTo(String dst) {
        new java.io.File(getUnwrapped())
            .renameTo(new java.io.File(StringTool.getUnwrapped(dst)));
    }
    
    public void renameTo(File dst) {
        new java.io.File(StringTool.getUnwrapped(absPath)).renameTo(dst.getJavas());
    }
    
    public void setDOSArchive(boolean flag) {
        Path file = getAsPath();
        try {
            Files.setAttribute(file, "dos:archive", flag);
        } catch (IOException ioe) {
            System.err.println("Unable to set archive attribute for: " + getAbsolutePath());
        }
    }

    public void setDOSAttributes(String attributes) {
        setDOSReadOnly(attributes.contains("R"));
        setDOSHidden(attributes.contains("H"));
        setDOSSystem(attributes.contains("S"));
        setDOSArchive(attributes.contains("A"));
    }

    public void setDOSHidden(boolean flag) {
        Path file = getAsPath();
        try {
            Files.setAttribute(file, "dos:hidden", flag);
        } catch (IOException ioe) {
            System.err.println("Unable to set hidden attribute for: " + getAbsolutePath());
        }
    }

    public void setDOSReadOnly(boolean flag) {
        Path file = getAsPath();
        try {
            Files.setAttribute(file, "dos:readonly", flag);
        } catch (IOException ioe) {
            System.err.println("Unable to set readonly attribute for: " + getAbsolutePath());
        }
    }

    public void setDOSSystem(boolean flag) {
        Path file = getAsPath();
        try {
            Files.setAttribute(file, "dos:system", flag);
        } catch (IOException ioe) {
            System.err.println("Unable to set system attribute for: " + getAbsolutePath());
        }
    }

    public void setExecutable(boolean execute) {
        getJavas().setExecutable(execute);
    }

    public void setName(String name) {
        File parent = getDeepestDirectory();
        absPath = parent.getUnwrapped() + name;
    }
    
    
    public void setPOSIXGroup(String group) {
        Path filePath = getAsPath();
        GroupPrincipal groupId = null;
        try {
            groupId = filePath.getFileSystem().getUserPrincipalLookupService().lookupPrincipalByGroupName(group);
        } catch (IOException ioe) {
            System.err.println("Unable to lookup up principal (group): " + group + " : " + ioe);

        }
        try {
            Files.getFileAttributeView(filePath, PosixFileAttributeView.class).setGroup(groupId);
        } catch (IOException ioe) {
            System.err.println("Unable to set group for: " + getAbsolutePath() + " : " + ioe);
        }
    }

    public void setPOSIXOwner(String owner) {
        Path filePath = getAsPath();
        UserPrincipal ownerId = null;
        try {
            ownerId = filePath.getFileSystem().getUserPrincipalLookupService().lookupPrincipalByName(owner);
        } catch (IOException ioe) {
            System.err.println("Unable to look up principal (owner): " + owner + " : " + ioe);
        }
        try {
            Files.setOwner(filePath, ownerId);
        } catch (IOException ioe) {
            System.err.println("Unable to set owner for: " + getAbsolutePath() + " : " + ioe);
        }
    }

    public void setPOSIXPermissions(int owner, int group, int other) {
        String ownerStr = getPermissionString(owner);
        String groupStr = getPermissionString(group);
        String otherStr = getPermissionString(other);
        Path file = getAsPath();
        Set<PosixFilePermission> perms = PosixFilePermissions.fromString(ownerStr + groupStr + otherStr);
        //FileAttribute<Set<PosixFilePermission>> attr = PosixFilePermissions.asFileAttribute(perms);
        try {
            Files.setPosixFilePermissions(file, perms);
        } catch (IOException ioe) {
            System.err.println("Unable to set POSIX file permissions: " + ioe);
        }
    }

    public void setPOSIXPermissions(String permissions) {
        Path file = getAsPath();
        Set<PosixFilePermission> perms = PosixFilePermissions.fromString(permissions);
        //FileAttribute<Set<PosixFilePermission>> attr = PosixFilePermissions.asFileAttribute(perms);
        try {
            Files.setPosixFilePermissions(file, perms);
        } catch (IOException ioe) {
            System.err.println("Unable to set POSIX file permissions: " + ioe);
        }
    }

    public void setReadable(boolean read) {
        getJavas().setReadable(read);
    }

    public void setWritable(boolean write) {
        getJavas().setWritable(write);
    }

    public Path toPath() {
        return getJavas().toPath();
    }

    public String toString() {
        return "File.toString(): " + getAbsolutePath() + " IsDir: " + isDirectory() + " : Length: " + getSize("toString");
    }

    public URI toURI() { return getJavas().toURI(); }
    
    public URL toURL() throws MalformedURLException { return getJavas().toURI().toURL(); }
    
    public static void writeTextFile(File file, List<String> lines) {
        FileWriter fw = null;
        PrintWriter out = null;
        try {
            fw = new FileWriter(file.getUnwrapped());
            out = new PrintWriter(fw, true);
        } catch (IOException ioe) {
            System.out.println("Unable to open file for writing: " + file.getAbsolutePath());
        }
        for (int i = 0; i < lines.size(); i++) {
            out.println(lines.get(i));
        }
        try {
            out.close(); // won't throw an exception per the JavaDocs, but we want to group our
            fw.close();  // closing of files/sockets/pipes/etc for easier confirmation

        } catch (IOException ioe) {
            System.err.println("File.writeTextFile: Closing fw: " + ioe);
        }
    }
}
