package ca.tecreations.text;

import ca.tecreations.TColor;
import ca.tecreations.TecData;
import ca.tecreations.Font;
import ca.tecreations.ImageTool;
import ca.tecreations.Point;
import ca.tecreations.SystemToken;
import ca.tecreations.TextToken;
import ca.tecreations.interfaces.*;
import ca.tecreations.graphics.DrawObject;
import ca.tecreations.text.ansi.ANSILookupResolver;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.util.List;

import javax.swing.JPanel;
import javax.swing.Timer;

/**
 *
 * @author Tim
 */
public class TextTokenPainter extends DrawObject implements TextPainter {
    private int x = 0;
    private int y = 0;
    
    public int BETWEEN = 2;

    public void setBetween(int pixels) {
        BETWEEN = pixels;
    }

    int width;
    int height;
    
    private TextPoints points = TecData.CODE_POINTS;
    TextToken token;

    Color background = TColor.DEFAULT_SWING_BG;
    Color foreground = TColor.black;

    public boolean bold = false;
    public boolean dim = false;
    public boolean italic = false;
    boolean underline = false;
    boolean drawMidLine = false;
    boolean doubleUnderline = false;
    boolean blink = false;
    private Timer blinkTimer;
    private int blinkSpeed = 500; // cycle once per second
//    TColor oldBackground = background;
//    TColor oldForeground = foreground;
    TColor blinkBackground = TColor.white;
    TColor blinkForeground = TColor.black;
    private boolean drawBlink = false;
    boolean reverse = false;
    boolean hidden = false;
    boolean strikethrough = false;
    boolean highlight = false;
    TColor highlightBackground = TColor.yellow;
    TColor highlightForeground = TColor.black;

    boolean drawBaseline = false;
    
    int startColumn = 0;
    int cursorIndex = 0;
    
    boolean[] selected;
    TColor SELECTED_COLOR = new TColor(48,74,146);
    
    boolean debug = true;
    boolean verbose = false;
    
    private TextTokenPainter() {}
    
    public TextTokenPainter copy() {
        // sure you could... to/From/EXTERNAL_FORM...
        TextTokenPainter ttp = new TextTokenPainter();
        ttp.x = x;
        ttp.y = y;

        ttp.BETWEEN = BETWEEN;

        ttp.width = width;
        ttp.height = height;
    
        ttp.points = points;
        ttp.token = token;

        ttp.background = background;
        ttp.foreground = foreground;

        ttp.bold = bold;
        ttp.dim = dim;
        ttp.italic = italic;
        ttp.underline = underline;
        ttp.drawMidLine = drawMidLine;
        ttp.doubleUnderline = doubleUnderline;
        ttp.blink = blink;
        ttp.blinkTimer = null; // operator must start a blinkTimer thread, if need be
        ttp.blinkSpeed = blinkSpeed;
//    TColor oldBackground = background;
//    TColor oldForeground = foreground;
        ttp.blinkBackground = blinkBackground;
        ttp.blinkForeground = blinkForeground;
        ttp.drawBlink = drawBlink;
        ttp.reverse = reverse;
        ttp.hidden = hidden;
        ttp.strikethrough = strikethrough;
        ttp.highlight = highlight;
        ttp.highlightBackground = highlightBackground;
        ttp.highlightForeground = highlightForeground;

        ttp.drawBaseline = drawBaseline;
    
        ttp.startColumn = startColumn;
        ttp.cursorIndex = cursorIndex;
    
        ttp.selected = selected;
        ttp.SELECTED_COLOR = SELECTED_COLOR;
    
        ttp.debug = debug;
        ttp.verbose = verbose;
        
        ttp.points = points;
        ttp.token = token;
        return ttp; // that will do.
    }
    
    public TextTokenPainter(TextPoints points, TextToken token,java.awt.Color fg) {
        this(points,token,new TColor(fg));
    }
    
    public TextTokenPainter(TextPoints points, TextToken token,TColor fg) {
        this.points = points;
        this.token = token;
        this.foreground = fg;
        setForeground(fg);
        int style = points.getFont().getStyle();
        if (style == (Font.BOLD & Font.ITALIC)) {
            bold = true;
            italic = true;
        } else if (style == Font.BOLD) {
            bold = true;
        } else if (style == Font.ITALIC) {
            italic = true;
        }
        getComputedSize();
        //JavaLauncher.tspc.out("SystemTokenPainter.size: " + width + "x" + height);
        selected = new boolean[this.token.length()];
    }

    public void applyCodes(List<Integer> codes) {
        //JavaLauncher.tspc.out("Text: " + getText() + " Codes: " + codes);
        if (codes != null) {
            for (int i = 0; i < codes.size(); i++) {
                int code = codes.get(i);
                if (code == 0) {
                    resetAll();
                } else if (code == 1) {
                    if (codes.size() >= i + 1) {
                        if (codes.get(i + 1) == 34) {
                            i++;
                        } else {
                            setBold(true);
                        }
                    } else {
                        setBold(true);
                    }
                } else if (code == 2) {
                    setDim(true);
                } else if (code == 3) {
                    setItalic(true);
                } else if (code == 4) {
                    setUnderline(true);
  //              } else if (code == 5) {
  //                  setBlink(true);
                } else if (code == 7) {
                    setReverse(true);
                } else if (code == 8) {
                    setHidden(true);
                } else if (code == 9) {
                    setStrikethrough(true);
                } else if (code == 21) {
                    setDoubleUnderline(true);
                } else if (code == 22) {
                    resetBoldAndDim();
                } else if (code == 23) {
                    setItalic(false);
                } else if (code == 24) {
                    setDoubleUnderline(false);
//                } else if (code == 25) {
//                    setBlink(false);
                } else if (code == 27) {
                    setReverse(false);
                } else if (code == 28) {
                    setHidden(false);
                } else if (code == 29) {
                    setStrikethrough(false);
                } else if (code == 30) {
                    setForeground(TColor.black);
                } else if (code == 31) {
                    setForeground(TColor.red);
                } else if (code == 32) {
                    setForeground(TColor.green);
                } else if (code == 33) {
                    setForeground(TColor.yellow);
                } else if (code == 34) {
                    setForeground(TColor.blue);
                } else if (code == 35) {
                    setForeground(TColor.magenta);
                } else if (code == 36) {
                    setForeground(TColor.cyan);
                } else if (code == 37) {
                    setForeground(TColor.white);
                } else if (code == 39) {
                    setForeground(TColor.black);
                } else if (code == 40) {
                    setBackground(TColor.black);
                } else if (code == 41) {
                    setBackground(TColor.red);
                } else if (code == 42) {
                    setBackground(TColor.green);
                } else if (code == 43) {
                    setBackground(TColor.yellow);
                } else if (code == 44) {
                    setBackground(TColor.blue);
                } else if (code == 45) {
                    setBackground(TColor.magenta);
                } else if (code == 46) {
                    setBackground(TColor.cyan);
                } else if (code == 47) {
                    setBackground(TColor.white);
                } else if (code == 49) {
                    background = new TColor(new JPanel().getBackground());
                } else if (code == 90) {
                    setForeground(TColor.BRIGHT_BLACK);
                } else if (code == 91) {
                    setForeground(TColor.BRIGHT_RED);
                } else if (code == 92) {
                    setForeground(TColor.BRIGHT_GREEN);
                } else if (code == 93) {
                    setForeground(TColor.BRIGHT_YELLOW);
                } else if (code == 94) {
                    setForeground(TColor.BRIGHT_BLUE);
                } else if (code == 95) {
                    setForeground(TColor.BRIGHT_MAGENTA);
                } else if (code == 96) {
                    setForeground(TColor.BRIGHT_CYAN);
                } else if (code == 97) {
                    setForeground(TColor.BRIGHT_WHITE);
                } else if (code == 100) {
                    setBackground(TColor.BRIGHT_BLACK);
                } else if (code == 101) {
                    setBackground(TColor.BRIGHT_RED);
                } else if (code == 102) {
                    setBackground(TColor.BRIGHT_GREEN);
                } else if (code == 103) {
                    setBackground(TColor.BRIGHT_YELLOW);
                } else if (code == 104) {
                    setBackground(TColor.BRIGHT_BLUE);
                } else if (code == 105) {
                    setBackground(TColor.BRIGHT_MAGENTA);
                } else if (code == 106) {
                    setBackground(TColor.BRIGHT_CYAN);
                } else if (code == 107) {
                    setBackground(TColor.BRIGHT_WHITE);
                } else if (code == 38) {
                    // next should be 5 or 2
                    int nextCode = codes.get(i + 1);
                    if (nextCode == 5) {
                        int lookup = codes.get(i + 2);
                        setForeground(ANSILookupResolver.getColor(lookup));
                        i += 2;
                    } else if (nextCode == 2) {
                        setForeground(new TColor(codes.get(i + 2), codes.get(i + 3), codes.get(i + 4)));
                    }
                } else if (code == 48) {
                    int nextCode = codes.get(i + 1);
                    if (nextCode == 5) {
                        int lookup = codes.get(i + 2);
                        setBackground(ANSILookupResolver.getColor(lookup));
                        i += 2;
                    } else if (nextCode == 2) {
                        setBackground(new TColor(codes.get(i + 2), codes.get(i + 3), codes.get(i + 4)));
                    }
                }
            }
        }
    }
    
    public void backspace() {
        token.backspace();
    }
    public TextTokenPainter clone() {
        TextTokenPainter clone = new TextTokenPainter(points,new TextToken(),foreground);
        clone.setBackground(background);
        return clone;
    }
    
//    public void cursorHide(Graphics g,int index) {
//        paintCursor(index,background);
//        paintNormal(index);
//    }
    
//    public void cursorShow(Graphics g, int index, TColor cursorTColor) {
//        paintCursor(index, cursorTColor);
//    }
    
    public void cut(int index, int count) {
        token.cut(index,count);
    }
    
    public void deselect() {
        for(int i = 0;i < selected.length;i++) {
            selected[i] = false;
        }
    }
    
    public void draw(Graphics g) {
        paintAt(g,getTX(),getTY());
    }
    
    public void draw(Graphics g, int tx, int ty) {
        paintAt(g,tx,ty);
    }
    
    public void draw(Graphics g, int tx, int ty, Color lineColor) {
        setLineColor(lineColor);
        paintAt(g,tx,ty);
    }
    
    public void draw(Graphics g, int tx, int ty, Color lineColor, Color fillColor) {
        setLineColor(lineColor);
        setFillColor(fillColor);
        paintAt(g,tx,ty);
    }
    
    protected int drawMonospaced(String text, Graphics g, int tx, int ty) {
        int width = 0;
        char ch;
        int capDubWidth = this.points.getWidth((int) 'W');
        //System.out.println("TTP.drawMonospaced: points: " + points.getFontSize() + " Text: " + getText());
        for (int i = 0; i < text.length() - 1; i++) {
            ch = text.charAt(i);
            // get the xOffset to center the char in question
            int xOffset = (capDubWidth - points.getWidth(ch)) / 2;
            if (g != null) {
                Point.drawPoints(g, tx + width + xOffset, ty, points.getPoints(ch));
            } 
            width += capDubWidth + BETWEEN;
        }
        if (text.length() > 0) {
            ch = text.charAt(text.length() - 1);
            int xOffset = (capDubWidth - points.getWidth(ch)) / 2;
            if (g != null) {
                Point.drawPoints(g, tx + width + xOffset, ty, points.getPoints(ch));
            }
            width += capDubWidth;
        }
        return width;

    }

    protected int drawNonMonospaced(String text, Graphics g, int tx, int ty) {
        int width = 0;
        char ch;
        if (text.length() > 0) {
            for (int i = 0; i < text.length() - 1; i++) {
                ch = text.charAt(i);
                if (g != null) {
                    Point.drawPoints(g, tx + width, ty, points.getPoints(ch));
                }
                width += points.getWidth(ch) + BETWEEN;
            }
            ch = text.charAt(text.length() - 1);
            if (g != null) {
                Point.drawPoints(g, tx + width, ty, points.getPoints(ch));
            }
            width += points.getWidth(ch);
        }
        return width;
    }

    public int drawString(String text, Graphics g, int tx, int ty) {
        return drawString(text,g,tx,ty,0);
    }

    public int drawString(String text, Graphics g, int tx, int ty, int between) {
        if (points.isMonospaced()) {
            return drawMonospaced(text, g, tx, ty);
        } else {
            return drawNonMonospaced(text, g, tx, ty);
        }
    }

    public Color getBackground() { 
        return background;
    }
    
    public boolean getBlink() {
        return blink;
    }
    
    public Dimension getComputedSize() {
        width = getTextWidth();
        height = points.getMaxHeight();
        return new Dimension(width, height);
    }

    public int getCursorIndex() { return cursorIndex; }
    
    public int getEndX() { return x + getTextWidth(); }
    
    public int getFontSize() {
        return points.getFontSize();
    }

    public Color getForeground() {
        return foreground;
    }
    
    public int getHeight() {
        return getFontSize();
    }
    
    public boolean getHighlight() {
        return highlight;
    }

    public BufferedImage getImage() {
        BufferedImage image = ImageTool.getNewBufferedImage(width, height);
        Graphics g = image.getGraphics();
        paintAt(g,0,0);
        return image;
    }

    public int getInternalIndex(int x) {
        String text = token.getText();
        int consumed = 0;
        for(int i = 0; i < text.length();i++) {
            int ch = text.charAt(i);
            int min = this.x + consumed - (BETWEEN / 2);
            int max = this.x + consumed + points.getWidth(ch) + (BETWEEN / 2);
            if (min <= x && x <= max) {
                int mid = points.getWidth(ch) / 2;
                int left = min + mid;
                if (x <= left) return i;
                else return i + 1;
            } else consumed += points.getWidth(ch) + BETWEEN;
            System.out.println("Consumed: " + consumed + " min: " + min + " max: " + max + " i: " + i + " x: " + x);
        } 
        return length();
    }
    
    public int getLength() {
        return token.length();
    }
    
    public int getMaxDescent() {
        return points.getMaxDescent();
    }
    
    public int getPaintingWidth() {
        return getTextWidth();
    }
    
    public TextPoints getPoints() {
        return points;
    }
    
    public TextTokenPainter getSpace() {
        TextTokenPainter temp = clone();
        temp.setText(" ");
        return temp;
    }
    
    public String getText() {
        return token.getText();
    }

    public TextToken getTextToken() { return token; }

    public int getTextWidth() {
        int width = 0;
        String text = token.getText();
        int dubWidth = points.getWidth('W');
            
        //System.err.println("Text: '" + text + "'");
        if (points.isMonospaced()) {
            if (text.length() > 0) {
                for(int i = 0; i < text.length() - 1;i++) {
                    width += dubWidth + BETWEEN;
                }
                width += dubWidth;
            }
            return width;
        } else {
            if (text.length() > 0) {
                for(int i = 0;i < text.length() - 1;i++) {
                    width += points.getWidth(text.charAt(i)) + BETWEEN;
                }
                width += points.getWidth(text.charAt(text.length() - 1));
            }
        }
        return width;
    }
    
    public int getWidth(int index) {
        String text = getText();
        if (index < text.length()) {
            char ch = text.charAt(index);
            return points.getWidth(ch);
        } else throw new IllegalArgumentException("TTP.getWidth: index(" + index + ") must be less than length(" + text.length() + ")");
    }
    
    public int getWidthTo(int index) {
        int width = 0;
        String text = token.getText();
        //System.err.println("Text: '" + text + "'");
        if (points.isMonospaced()) {
            int i = 0;
            for(; i < index - 1;i++) {
                width += points.getWidth('W') + BETWEEN;
            }
            if (i < index - 1) width += points.getWidth('W');
            return width;
        }
        int i = 0;
        for (; i < index - 1; i++) {
            width += points.getWidth(text.charAt(i)) + BETWEEN;
        }
        if (i < index - 1) width += points.getWidth(text.charAt(index - 1));
        return width;
    }
    
    public int getWidthAtIndex(int index) {
        char ch = token.getText().charAt(index);
        System.out.println("ch: "+ch);
        return points.getWidth(ch);
    }
    
    public int getWidthForString(String s) {
        String oldText = token.getText();
        token.setText(s);
        int width = getTextWidth();
        token.setText(oldText);
        return width;
    }
    
    public int getX() { return x; }
    
    public boolean hasCol(int start, int col) {
        int max = start + length();
        return (col > start && col < max);
    }
    
    public boolean hasFillPoint(Point p) {
        return false;
    }
    
    public boolean hasPoint(Point p) {
        return false;
    }
    
    public void insert(int index, char c) {
        insert(index,"" + c);   
    }
    
    public void insert(int index, String s) {
        token.insert(index,s);   
    }
    
    public boolean isSystemErr() {
        if (token instanceof SystemToken) {
            if (((SystemToken) token).getStreamType() == TecData.SYS_ERR) {
                return true;
            }
        }
        return false;
    }

    public boolean isSystemOut() {
        if (token instanceof SystemToken) {
            if (((SystemToken) token).getStreamType() == TecData.SYS_OUT) {
                return true;
            }
        }
        return false;
    }
    
    public int length() { return token.length(); }
    
    public void paintAt(Graphics g, int x, int y) {
//        System.out.println("paintAt: bold: " + bold + " italic: " + italic + " Points.font: " + points.toString());
        if (g != null) {
            this.x = x;
            this.y = y;
            Color bg;
            Color fg;
            if (reverse) {
                bg = foreground;
                fg = background;
            } else if (highlight) {
                bg = highlightBackground;
                fg = highlightForeground;
            } else if (blink && drawBlink) {
                bg = blinkBackground;
                fg = blinkForeground;
            } else {
                bg = background;
                fg = foreground;
            }
            int x2 = x;
            String text = token.getText();
            for(int i = 0; i < text.length();i++) {
                int w;
                if (points.isMonospaced()) {
                    w = points.getWidth('W');
                } else {
                    w = points.getWidth(text.charAt(i));
                }
                if (selected.length > 0 && selected[i]) {
                    g.setColor(SELECTED_COLOR);
                } else {
                    g.setColor(bg);
                }
                g.fillRect(x2,y,w + BETWEEN,points.getFontSize());
                x2 += w + BETWEEN;
            }
            
            g.setColor(fg);
            this.x = x;
            if (!hidden) {
                if (debug && verbose) System.out.println("Token: " + token.getText());
                drawString(token.getText(), g, x, y);
            } else {
                if (debug && verbose) System.err.println("Hidden == true");
            }
            // draw to side of 0, not max0
            if (strikethrough && !hidden) g.drawLine(0, height / 2 - 1, width, height / 2 - 1);
            if (underline && !hidden) g.drawLine(0, height + 1, width, height + 1);
            if (drawMidLine && !hidden) g.drawLine(0,height + 2, width, height + 2);
            if (doubleUnderline & !hidden) g.drawLine(0, height + 3, width, height + 3);
            
            int baseline = points.getBaseline();
            if (drawBaseline) {
                g.setColor(TColor.TEC_LIGHT_GREEN);
                g.drawLine(0, baseline, width, baseline);
            }
        }
    }

    public void paintChar(Graphics g, int y, int index, TColor color) {
        int indexOffset = 0;
        boolean isMono = points.isMonospaced();
        int dubWidth = points.getWidth('W');
        for(int i = 0; i < index;i++) {
            if (isMono) {
                indexOffset += BETWEEN + dubWidth;
            } else {
                indexOffset += BETWEEN + points.getWidth(getText().charAt(i));
            }
        }
        g.setColor(color);
        Point.drawPoints(g, x + indexOffset, y,points.getPoints(getText().charAt(index)));
    }
    
    public void paintIndex(Graphics g, int index, TColor lineTColor,TColor fillTColor) {
        int paintX = 0;
        for(int i = 0; i < index;i++) {
            paintX += points.getWidth(token.getText().charAt(i)) + BETWEEN;
        }
        if (fillTColor != null) {
            g.setColor(fillTColor);
            g.fillRect(paintX - 1, y, points.getWidth(token.getText().charAt(index)) + 2,height);
        }
        List<Point> pixels = points.getPoints(getText().charAt(index));
        g.setColor(lineTColor);
        int indexOffset = 0;
        for(int i = 0; i < index;i++) {
            indexOffset += getWidth(getText().charAt(i)) + BETWEEN;
        }
        Point.drawPoints(g, x + indexOffset, y,pixels);
        
    }
    
    public void paintElement(Graphics g) {
        paintAt(g,getTX(),getTY());
    }
    
    //public void paintNormal(int index) {
//        paintAt(g,x,y);
//    }
    
    public void prefix(String s) {
        token.prefix(s);
    }
    
    public void repaint(Graphics g) {
        paintAt(g,x,y);
    }
    
    public void replace(int index, char c) {
        token.replace(index,"" + c);   
    }
    
    public void replace(int index, String s) {
        token.replace(index,s);   
    }
    
    
    public void resetAll() {
        background = new TColor(new JPanel().getBackground());
        foreground = TColor.black;
        bold = false;
        dim = false;
        italic = false;
        underline = false;
        doubleUnderline = false;
        blink = false;
        reverse = false;
        hidden = false;
        strikethrough = false;
    }

    public void resetBoldAndDim() {
        setBold(false);
        setDim(false);
    }

    public void select() {
        for(int i = 0; i < selected.length;i++) {
            selected[i] = true;
        }
    }
    
    public void selectFirst(int cols) {
        for(int i = 0;i < selected.length;i++) {
            if (i < cols) selected[i] = true;
            else selected[i] = false;
        }
    }
    
    public void selectFrom(int col) {
        for(int i = 0;i < selected.length;i++) {
            if (i >= col) selected[i] = true;
            else selected[i] = false;
        }
    }

    public void selectInclusive(int low, int high) {
        for(int i = 0; i < selected.length;i++) {
            if (i >= low && i <= high) {
                selected[i] = true;
            } else {
                selected[i] = false;
            }
        }
    }
    
    public void setBackground(java.awt.Color color) {
        background = new TColor(color);
    }

    public void setBackground(TColor color) {
        background = color;
    }
    
    public void setBold(boolean state) {
        bold = state;
        setTextPointsStyle();
    }

    public void setColumnSelected(int index, boolean val) {
        selected[index] = val;
    }
    
    public void setCursorIndex(int index) {
        cursorIndex = index;
    }
    
    public void setDim(boolean state) {
        dim = state;
    }

    public void setFillColor(Color fillColor) {
        background = fillColor;
    }
    
    public void setDrawMidLine(boolean state) {
        drawMidLine = state;
    }

    public void setDoubleUnderline(boolean state) {
        underline = state;
        doubleUnderline = state;
    }

    public void setFontSize(int size) {
        points = TextPoints.getInstance(points.getFontName(), points.getFontStyle(), size);
        getComputedSize();
    }

    public void setForeground(java.awt.Color color) {
        foreground = new TColor(color);
    }

    public void setHidden(boolean state) {
        hidden = state;
    }

    public TextTokenPainter setHighlight(boolean state) {
        highlight = state;
        return this;
    }

    public TextTokenPainter setHighlightBackground(TColor bg) {
        highlightBackground = bg;
        return this;
    }

    public TextTokenPainter setHighlightForeground(TColor fg) {
        highlightForeground = fg;
        return this;
    }

    public void setItalic(boolean state) {
        italic = state;
        setTextPointsStyle();
    }

    public void setLineColor(Color c) {
        foreground = c;
    }
    
    public void setPoints(TextPoints points) {
        //System.out.println("TTP.setPoints: points: " + points.toString());
        this.points = points;
        //System.out.println("TTP.setPoints: this.points: " + points.toString());
        getComputedSize();
    }
    
    public void setReverse(boolean state) {
        reverse = state;
    }

    public void setStartColumn(int col) {
        startColumn = col;
    }
    
    public void setStrikethrough(boolean state) {
        strikethrough = state;
    }

    public void setText(char ch) {
        token.setText(ch + "");
        getComputedSize();
    }

    public void setText(int i) {
        token.setText("" + i);
        getComputedSize();
    }

    public void setText(String text) {
        token.setText(text);
        getComputedSize();
    }

    public void setTextPoints(TextPoints points) {
        this.points = points;
    }
    
    public void setTextPointsStyle() {
        if (bold && italic) {
            points = TextPoints.getInstance(points.getFontName(), Font.BOLD | Font.ITALIC, points.getFontSize());
        } else if (bold && !italic) {
            points = TextPoints.getInstance(points.getFontName(), Font.BOLD, points.getFontSize());
        } else if (!bold && italic) {
            points = TextPoints.getInstance(points.getFontName(), Font.ITALIC, points.getFontSize());
        } else {
            points = TextPoints.getInstance(points.getFontName(), Font.PLAIN, points.getFontSize());
        }
        width = getTextWidth();
        height = points.getFontSize();
    }

    public void setToken(TextToken token) {
        this.token = token;
    }
    
    public void setUnderline(boolean state) {
        underline = state;
        if (!state) {
            doubleUnderline = false;
        }
    }

    public char[] toCharArray() {
        String text = token.getText();
        char[] array = new char[text.length()];
        for(int i = 0; i < text.length();i++) {
            array[i] = text.charAt(i);
        }
        return array;
    }
    
    public String[] toStringArray() {
        String text = token.getText();
        String[] array = new String[text.length()];
        for(int i = 0; i < text.length();i++) {
            array[i] = "" + text.charAt(i);
        }
        return array;
    }
    
    public void toggleDrawBaseline() {
        drawBaseline = !drawBaseline;
    }

    public void toggleItalic() {
        italic = !italic;
        setItalic(italic);
    }

}
