package ca.tecreations.text;

import ca.tecreations.text.ansi.ANSILookupResolver;
import ca.tecreations.TColor;
import ca.tecreations.TecData;
import ca.tecreations.Font;
import ca.tecreations.ImageTool;
import ca.tecreations.Point;
import ca.tecreations.SystemToken;
import ca.tecreations.TextToken;
import ca.tecreations.components.Movable;
import ca.tecreations.interfaces.Paintable;
import ca.tecreations.interfaces.TextPainter;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.awt.event.*;
import java.util.List;

import javax.swing.JPanel;
import javax.swing.Timer;

/**
 *
 * @author Tim
 */
public class GUITextTokenPainter extends Movable implements ActionListener, 
        MouseListener, TextPainter, Paintable {
    public int BETWEEN = 2;

    public void setBetween(int pixels) {
        BETWEEN = pixels;
    }


    TextPoints points;
    TextToken token;

    Color defaultBackground;
    Color oldBackground = null;
    Color oldForeground = null;

    public boolean bold = false;
    public boolean dim = false;
    public boolean italic = false;
    boolean underline = false;
    boolean doubleUnderline = false;
    boolean blink = false;
    private Timer blinkTimer;
    private int blinkSpeed = 500; // cycle once per second
    Color blinkBackground = Color.white;
    Color blinkForeground = Color.black;
    private boolean drawBlink = false;
    boolean reverse = false;
    boolean hidden = false;
    boolean strikethrough = false;
    boolean highlight = false;
    Color highlightBackground = Color.yellow;
    Color highlightForeground = Color.black;

    boolean drawBaseline = false;
    boolean drawX = false;

    Color xColor = null;
    Color outlineColor = null;

    public GUITextTokenPainter(TextPoints points, TextToken token) {
        super(Movable.NONE);
        //super(1, 1);
        this.points = points;
        this.token = token;
        if (token instanceof SystemToken) {
            if (((SystemToken) token).getStreamType() == TecData.SYS_ERR) {
                setForeground(Color.RED);
                oldForeground = Color.RED;
            }
        }
        int style = points.getFont().getStyle();
        if (style == (Font.BOLD & Font.ITALIC)) {
            bold = true;
            italic = true;
        } else if (style == Font.BOLD) {
            bold = true;
        } else if (style == Font.ITALIC) {
            italic = true;
        }
        defaultBackground = new TColor(getBackground());
        oldBackground = defaultBackground;
        setSize(getComputedSize());
    }

    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == blinkTimer) {
            if (blink) {
                drawBlink = !drawBlink;
            }
            repaint();
        }
    }

    public void applyCodes(List<Integer> codes) {
        //JavaLauncher.tspc.out("Text: " + getText() + " Codes: " + codes);
        if (codes != null) {
            for (int i = 0; i < codes.size(); i++) {
                int code = codes.get(i);
                if (code == 0) {
                    resetAll();
                } else if (code == 1) {
                    if (codes.size() >= i + 1) {
                        if (codes.get(i + 1) == 34) {
                            i++;
                        } else {
                            setBold(true);
                        }
                    } else {
                        setBold(true);
                    }
                } else if (code == 2) {
                    setDim(true);
                } else if (code == 3) {
                    setItalic(true);
                } else if (code == 4) {
                    setUnderline(true);
                } else if (code == 5) {
                    setBlink(true);
                } else if (code == 7) {
                    setReverse(true);
                } else if (code == 8) {
                    setHidden(true);
                } else if (code == 9) {
                    setStrikethrough(true);
                } else if (code == 21) {
                    setDoubleUnderline(true);
                } else if (code == 22) {
                    resetBoldAndDim();
                } else if (code == 23) {
                    setItalic(false);
                } else if (code == 24) {
                    setDoubleUnderline(false);
                } else if (code == 25) {
                    setBlink(false);
                } else if (code == 27) {
                    setReverse(false);
                } else if (code == 28) {
                    setHidden(false);
                } else if (code == 29) {
                    setStrikethrough(false);
                } else if (code == 30) {
                    setForeground(Color.black);
                } else if (code == 31) {
                    setForeground(Color.red);
                } else if (code == 32) {
                    setForeground(Color.green);
                } else if (code == 33) {
                    setForeground(Color.yellow);
                } else if (code == 34) {
                    setForeground(Color.blue);
                } else if (code == 35) {
                    setForeground(Color.magenta);
                } else if (code == 36) {
                    setForeground(Color.cyan);
                } else if (code == 37) {
                    setForeground(Color.white);
                } else if (code == 39) {
                    setForeground(Color.black);
                } else if (code == 40) {
                    setBackground(Color.black);
                } else if (code == 41) {
                    setBackground(Color.red);
                } else if (code == 42) {
                    setBackground(Color.green);
                } else if (code == 43) {
                    setBackground(Color.yellow);
                } else if (code == 44) {
                    setBackground(Color.blue);
                } else if (code == 45) {
                    setBackground(Color.magenta);
                } else if (code == 46) {
                    setBackground(Color.cyan);
                } else if (code == 47) {
                    setBackground(Color.white);
                } else if (code == 49) {
                    setBackground(new JPanel().getBackground());
                } else if (code == 90) {
                    setForeground(TColor.BRIGHT_BLACK);
                } else if (code == 91) {
                    setForeground(TColor.BRIGHT_RED);
                } else if (code == 92) {
                    setForeground(TColor.BRIGHT_GREEN);
                } else if (code == 93) {
                    setForeground(TColor.BRIGHT_YELLOW);
                } else if (code == 94) {
                    setForeground(TColor.BRIGHT_BLUE);
                } else if (code == 95) {
                    setForeground(TColor.BRIGHT_MAGENTA);
                } else if (code == 96) {
                    setForeground(TColor.BRIGHT_CYAN);
                } else if (code == 97) {
                    setForeground(TColor.BRIGHT_WHITE);
                } else if (code == 100) {
                    setBackground(TColor.BRIGHT_BLACK);
                } else if (code == 101) {
                    setBackground(TColor.BRIGHT_RED);
                } else if (code == 102) {
                    setBackground(TColor.BRIGHT_GREEN);
                } else if (code == 103) {
                    setBackground(TColor.BRIGHT_YELLOW);
                } else if (code == 104) {
                    setBackground(TColor.BRIGHT_BLUE);
                } else if (code == 105) {
                    setBackground(TColor.BRIGHT_MAGENTA);
                } else if (code == 106) {
                    setBackground(TColor.BRIGHT_CYAN);
                } else if (code == 107) {
                    setBackground(TColor.BRIGHT_WHITE);
                } else if (code == 38) {
                    // next should be 5 or 2
                    int nextCode = codes.get(i + 1);
                    if (nextCode == 5) {
                        int lookup = codes.get(i + 2);
                        setForeground(ANSILookupResolver.getColor(lookup));
                        i += 2;
                    } else if (nextCode == 2) {
                        setForeground(new Color(codes.get(i + 2), codes.get(i + 3), codes.get(i + 4)));
                    }
                } else if (code == 48) {
                    int nextCode = codes.get(i + 1);
                    if (nextCode == 5) {
                        int lookup = codes.get(i + 2);
                        setBackground(ANSILookupResolver.getColor(lookup));
                        i += 2;
                    } else if (nextCode == 2) {
                        setBackground(new Color(codes.get(i + 2), codes.get(i + 3), codes.get(i + 4)));
                    }
                }
            }
        }
    }
    

    public int basicDrawString(String text, Graphics g, int tx, int ty) {
        int width = 0;
        char ch;
        int charWidth = 0;
        for (int i = 0; i < text.length(); i++) {
            ch = text.charAt(i);
            List<Point> points = this.points.getPoints(ch);
            if (g != null) {
                Point.drawPoints(g, tx + width, ty, points);
            }
            width += this.points.getWidth((int) ch);
        }
        return width;
    }

    public void doPainting(Graphics g) {
        paintAt(g,0,0);
    }
    
    public int drawMonospaced(String text, Graphics g, int tx, int ty) {
        int width = 0;
        char ch;
        int capDubWidth = this.points.getWidth((int) 'W');
        for (int i = 0; i < text.length() - 1; i++) {
            ch = text.charAt(i);
            int xOffset = (capDubWidth - points.getWidth(ch)) / 2;
            if (g != null) {
                Point.drawPoints(g, tx + width + xOffset, ty, points.getPoints(ch));
            }
            width += capDubWidth;
        }
        if (text.length() > 0) {
            ch = text.charAt(text.length() - 1);
            int xOffset = (capDubWidth - points.getWidth(ch)) / 2;
            if (g != null) {
                Point.drawPoints(g, tx + width + xOffset, ty, points.getPoints(ch));
            }
            width += capDubWidth;
        }
        return width;

    }

    public int drawNonMonospaced(String text, Graphics g, int tx, int ty) {
        int width = 0;
        char ch = ' ';
        if (text.length() > 0) {
            for (int i = 0; i < text.length() - 1; i++) {
                ch = text.charAt(i);
                if (g != null) {
                    Point.drawPoints(g, tx + width, ty, points.getPoints(ch));
                }
                width += points.getWidth(ch) + BETWEEN;
            }
            ch = text.charAt(text.length() - 1);
            if (g != null) {
                Point.drawPoints(g, tx + width, ty, points.getPoints(ch));
            }
            width += points.getWidth(ch);
        }
        return width;
    }

    public int drawString(String text, Graphics g, int tx, int ty) {
        if (points.isMonospaced()) {
            return drawMonospaced(text, g, tx, ty);
        } else {
            return drawNonMonospaced(text, g, tx, ty);
        }
    }

    public boolean getBlink() {
        return blink;
    }

    public Dimension getComputedSize() {
        int width = getTextWidth();
        int height = points.getMaxHeight();
        if (doubleUnderline) {
            underline = true;
            height += 2;
        }
        if (underline) {
            height += 2;
        }
        return new Dimension(width, height);
    }

    public int getFontSize() {
        return points.getFontSize();
    }

    public boolean getHighlight() {
        return highlight;
    }

    public BufferedImage getImage() {
        BufferedImage image = ImageTool.getNewBufferedImage(getSize().width, getSize().height);
        Graphics g = image.getGraphics();
        paint(g);
        return image;
    }

    public int getMaxDescent() {
        return points.getMaxDescent();
    }
    
    public int getPaintingWidth() { return getSize().width; }
    
    public String getText() {
        return token.getText();
    }

    public int getTextWidth() {
        int width = 0;
        String text = token.getText();
        if (points.isMonospaced()) {
            return text.length() * points.getWidth('W');
        }
        if (text != null) {
            for (int i = 0; i < text.length(); i++) {
                width += points.getWidth(text.charAt(i)) + BETWEEN;
            }
        }
        return width;
    }

    public int getTextWidth(int pixelsBetweenGlyphs) {
        int width = 0;
        String text = token.getText();
        if (points.isMonospaced()) {
            return text.length() * points.getWidth('W');
        }
        if (text != null) {
            for (int i = 0; i < text.length(); i++) {
                if (i == text.length() - 1) {
                    width += points.getWidth(text.charAt(i));
                } else {
                    width += points.getWidth(text.charAt(i)) + pixelsBetweenGlyphs;
                }
            }
        }
        return width;
    }

    public boolean isSystemErr() {
        if (token instanceof SystemToken) {
            if (((SystemToken) token).getStreamType() == TecData.SYS_ERR) {
                return true;
            }
        }
        return false;
    }

    public boolean isSystemOut() {
        if (token instanceof SystemToken) {
            if (((SystemToken) token).getStreamType() == TecData.SYS_OUT) {
                return true;
            }
        }
        return false;
    }

    public void mouseClicked(MouseEvent e) {
        if (e.isControlDown()) {
            System.out.println("GUITextTokenPainter: " + token.getText() + " Size: " + getSize());
        }
    }

    public void mouseDragged(MouseEvent e) {
        super.mouseDragged(e);
        getParent().paint(getParent().getGraphics());
    }

    public void mouseEntered(MouseEvent e) {
    }

    public void mouseExited(MouseEvent e) {
    }

    public void mousePressed(MouseEvent e) {
    }

    public void mouseReleased(MouseEvent e) {
        super.mouseReleased(e);
        getParent().paint(getParent().getGraphics());
    }

    public void paint(Graphics g) {
        paintAt(g,0,0);
    }

    public void paintAt(Graphics g, int x, int y) {
        //System.out.println("paintAt: bold: " + bold + " italic: " + italic + " Points.font: " + points.toString() + " x: " + x + " y: " + y);
        int width = getSize().width;
        int height = getSize().height;
        Color bg;
        Color fg;
        /**
         * Note that reverse takes highest precedence, then highlight, then
         * blink, and finally, the normal representation
         */
        if (reverse) {
            bg = new TColor(getForeground());
            fg = new TColor(getBackground());
        } else if (highlight) {
            bg = highlightBackground;
            fg = highlightForeground;
        } else if (blink && drawBlink) {
            bg = blinkBackground;
            fg = blinkForeground;
        } else {
            Color jBG = getBackground();
            Color jFG = getForeground();
            if (jBG == null) jBG = TColor.getRandom();
            if (jFG == null) jFG = TColor.getRandom();
            bg = new TColor(jBG);
            fg = new TColor(jFG);
        }
        if (g != null) {
            g.setColor(bg);
            g.fillRect(x, y , getSize().width, getSize().height);

            //g.setColor(fg);
            if (isSystemOut()) {
                g.setColor(Color.black);
            } else if (isSystemErr()) {
                g.setColor(Color.red);
            } else {
                g.setColor(fg);
            }
            if (!hidden) {
                drawString(token.getText(), g, x, y);
            }   

            if (strikethrough && !hidden) {
                g.drawLine(x, y + (height / 2), width, y + (height / 2));
            }   
            
            int baseline = points.getBaseline();
            if (drawBaseline) {
                g.setColor(TColor.TEC_LIGHT_GREEN);
                g.drawLine(x, y + baseline, width, y + baseline);
            }

            if (underline && !hidden) {
                g.setColor(fg);
                g.drawLine(x, points.getSize(), width, points.getSize());
            }
            if (doubleUnderline & !hidden) {
                g.setColor(fg);
                g.drawLine(x, points.getSize() + 2, width, points.getSize() + 2);
            }

            if (xColor != null) {
                g.setColor(xColor);
                g.drawLine(x, y, width - 1, y + height - 1);
                g.drawLine(x, y + height - 1, width - 1, y);
                g.drawLine(0, 0, width - 1, height - 1);
                g.drawLine(0, height - 1, width - 1, y);
            }
            if (outlineColor != null) {
                g.setColor(outlineColor);
                g.drawRect(x, y, width - 1, height - 1);
            }
        }
    }

    public void resetAll() {
        setBackground(defaultBackground);
        setForeground(Color.black);
        bold = false;
        dim = false;
        italic = false;
        underline = false;
        doubleUnderline = false;
        blink = false;
        reverse = false;
        hidden = false;
        strikethrough = false;
    }

    public void resetBoldAndDim() {
        setBold(false);
        setDim(false);
        repaint();
    }

    public void resetSize() {
        setSize(getComputedSize());
    }

    public void setBackground(Color color) {
        Color bg = getBackground();
        if (color == null) color = TecData.DEFAULT_COLOR;
        if (bg == null) {
            oldBackground = TecData.DEFAULT_COLOR;
            super.setBackground(new TColor(color));
        } else {
            oldBackground = new TColor(bg);
            super.setBackground(new TColor(color));
        }
        repaint();
    }

    public synchronized GUITextTokenPainter setBlink(boolean state) {
        blink = state;
        // state set, do the os required functionality, for those that can
        if (blink) {
            blinkTimer = new Timer(blinkSpeed, this);
            blinkTimer.start();
        } else {
            blinkTimer.stop();
            blinkTimer = null; // do the explicit call
            // System.gc();    // run explicitly, if unsure
        }
        return this;
    }

    public GUITextTokenPainter setBlinkBackground(Color bg) {
        blinkBackground = bg;
        // don't repaint, let it change when the cycle restarts
        return this;
    }

    public GUITextTokenPainter setBlinkForeground(Color fg) {
        blinkForeground = fg;
        // don't repaint, let it change when the cycle restarts
        return this;
    }

    public GUITextTokenPainter setBlinkSpeed(int blinkSpeed) {
        this.blinkSpeed = blinkSpeed;
        if (blink) {
            blinkTimer.stop();
            blinkTimer = new Timer(blinkSpeed, this);
            blinkTimer.start();
        }
        return this;
    }

    public void setBold(boolean state) {
        bold = state;
        setTextPointsStyle();
        repaint();
    }

    public void setDim(boolean state) {
        dim = state;
        repaint();
    }

    public void setDoubleUnderline(boolean state) {
        int width = getSize().width;
        int height = getSize().height;
        doubleUnderline = state;
        if (state) {
            underline = true;
            setSize(width, height + 4);
        } else {
            if (underline) {
                setSize(width, height + 2);
            } else {
                setSize(width, height);
            }
        }
        repaint();
    }

    public void setFillColor(Color c) {
        setBackground(c);
    }
    
    public void setFontSize(int size) {
        points = TextPoints.getInstance(points.getFontName(), points.getFontStyle(), size);
        setSize(getComputedSize());
        repaint();
    }

    public void setForeground(Color color) {
        Color fg = getForeground();
        if (fg == null) fg = TColor.getRandom();
        oldForeground = new TColor(fg);
        super.setForeground(color);
        repaint();
    }

    public void setHidden(boolean state) {
        hidden = state;
        repaint();
    }

    public GUITextTokenPainter setHighlight(boolean state) {
        highlight = state;
        repaint();
        return this;
    }

    public GUITextTokenPainter setHighlightBackground(Color bg) {
        highlightBackground = bg;
        repaint(); // repaint immediately, regardless of blink
        return this;
    }

    public GUITextTokenPainter setHighlightForeground(Color fg) {
        highlightForeground = fg;
        repaint(); // repaint immediately, regardless of blink
        return this;
    }

    public void setItalic(boolean state) {
        italic = state;
        setTextPointsStyle();
        repaint();
    }

    public void setLineColor(Color c) {
        setForeground(c);
    }
    
    public void setOutlineColor(Color outline) {
        outlineColor = outline;
        repaint();
    }

    public void setPoints(TextPoints points) {
        this.points = points;
        getComputedSize();
    }
    
    public void setReverse(boolean state) {
        reverse = state;
        repaint();
    }

    public void setStrikethrough(boolean state) {
        strikethrough = state;
        repaint();
    }

    public void setText(String text) {
        token.setText(text);
        setSize(getComputedSize());
        repaint();
    }

    public void setTextPointsStyle() {
        if (bold && italic) {
            points = TextPoints.getInstance(points.getFontName(), Font.BOLD | Font.ITALIC, points.getFontSize());
        } else if (bold && !italic) {
            points = TextPoints.getInstance(points.getFontName(), Font.BOLD, points.getFontSize());
        } else if (!bold && italic) {
            points = TextPoints.getInstance(points.getFontName(), Font.ITALIC, points.getFontSize());
        } else {
            points = TextPoints.getInstance(points.getFontName(), Font.PLAIN, points.getFontSize());
        }
        resetSize();
    }

    public void setToken(TextToken token) {
        setText(token.getText());
        resetSize();
    }
    
    public void setXColor(Color c) {
        this.xColor = c;
        repaint();
    }

    public void setUnderline(boolean state) {
        int width = getSize().width;
        int height = getSize().height;
        underline = state;
        if (state) {
            setSize(width, height + 2);
        } else {
            setSize(width, height);
            doubleUnderline = false;
        }
        repaint();
    }
    
    public char[] toCharArray() {
        String text = token.getText();
        char[] array = new char[text.length()];
        for(int i = 0; i < text.length();i++) {
            array[i] = text.charAt(i);
        }
        return array;
    }
    
    public String[] toStringArray() {
        String text = token.getText();
        String[] array = new String[text.length()];
        for(int i = 0; i < text.length();i++) {
            array[i] = "" + text.charAt(i);
        }
        return array;
    }
    


    public void toggleDrawBaseline() {
        drawBaseline = !drawBaseline;
        repaint();
    }

    public void toggleItalic() {
        italic = !italic;
        setItalic(italic);
    }

}
