package ca.tecreations.graphics;

import ca.tecreations.Point;
import ca.tecreations.TColor;

import java.awt.*;
import java.util.ArrayList;
import java.util.List;
/**
 *
 * @author Tim
 */
public class PieChart extends Circle {
    List<Integer> percents;
    List<TColor> colors;
    List<Line> lines = new ArrayList<>();
    List<List<Point>> arcs = new ArrayList<>();
    List<List<Point>> fills = new ArrayList<>();
    
    public PieChart(int diameter, List<Integer> percents, List<TColor> colors) {
        super(diameter);
        this.diameter = diameter;
        this.percents = percents;
        this.colors = colors;
        super.compute();
        if (colors.size() != percents.size()) {
            throw new IllegalArgumentException("Colors.size: " + colors.size() + " != percents.size: " + percents.size());
        }
        int total = 0;
        for(int i = 0;i < percents.size();i++) {
            total += percents.get(i);
        }
        if (total != 100) {
            throw new IllegalArgumentException("Percents not equal to 100: " + total);
        }
    }

    public void addFirstLine(List<Point> arc) {
        Line line = new Line();
        line.setTranslation(getTXY());
        line.setLineColor(lineColor);
        line.setEndPoint(getPointAtDegree(getNormalAngle(-90)));
        lines.add(line);
    }

    public void addLine(List<Point> arc) {
        Line line = new Line();
        line.setTranslation(getTXY());
        line.setLineColor(lineColor);
        line.setEndPoint(arc.get(0));
        lines.add(line);
    }
    
    public void compute() {
        if (percents != null) {
            int rot = getNormalAngle(-90);
            //System.out.println("Sizes: " + sizes);
            for(int i = 0; i < percents.size();i++) {
                int size = (int)((double)percents.get(i) * 3.6);
                if (i < percents.size() - 1) arcs.add(getArcPoints(rot, rot + size));
                else arcs.add(getLastArc(rot));
                if (lines.size() == 0) addFirstLine(arcs.get(i));
                else addLine(arcs.get(i));
                rot += size;
            }
            Line l1;
            Line l2;
            for(int i = 0;i < percents.size();i++) {
                List<Point> linePoints = new ArrayList<>();
                // set the fillPoints
                if (i < percents.size() - 1) {
                    l1 = lines.get(i);
                    l2 = lines.get(i+1);
                    fills.add(getFillPoints(l1,l2,arcs.get(i)));
                } else {
                    l1 = lines.get(percents.size() - 1);
                    l2 = lines.get(0);
                    fills.add(getFillPoints(l1,l2,arcs.get(i)));
                }
            }
        }
    }
    
    public void draw(Graphics g) {
        draw(g,0,0,lineColor,fillColor);
    }
    public void draw(Graphics g,int x, int y) {
        draw(g,x,y,lineColor,fillColor);
    }

    public void draw(Graphics g, int x, int y, Color lineColor) {
        draw(g,x,y,lineColor,null);
    }

    public void draw(Graphics g, int x, int y, Color lineColor, Color fillColor) {
        if (lines.size() == 0) compute();
        for(int i = 0;i < fills.size();i++) {
            g.setColor(colors.get(i));
            drawPoints(g,getTXY(),fills.get(i));
        }
        for(int i = 0; i < arcs.size();i++) {
            g.setColor(lineColor);
            //g.setColor(colors.get(i));
            drawPoints(g,getTXY(),arcs.get(i));
        }
        g.setColor(lineColor);
        for(int i = 0; i < lines.size();i++) {
            drawPoints(g,getTXY(),lines.get(i).getLinePoints());
        }
    }
    
    public List<Point> getAllPoints(Line l1, Line l2, List<Point> arcPoints) {
        List<Point> points = new ArrayList<>();
        add(l1.getLinePoints(),points);
        add(l2.getLinePoints(),points);
        add(arcPoints,points);
        return points;
    }
    
    public List<Point> getArcPoints(int start, int stop) {
        //System.out.println("Start: " + getNormalAngle(start) + " Stop: " + getNormalAngle(stop));
        List<Point> points = new ArrayList<>();
        List<Point> degree;
        if (start == 270) {
            add(getQ1Of270(),points);
            start++;
        }
        for(int i = start;i < stop;i++) {
            degree = getDegree(getNormalAngle(i));
            if (degree != null) {
                add(degree,points);
            }
        }
        return points;
    }
    
    public String getBlock() {
        String s = "";
        for(int i = 0; i < lines.size();i++) {
            s += lines.get(i).getBlock();
        }
        return s;
    }
    
    public List<Point> getClockwiseWithLine(List<Point> points,int from,Line line) {
        List<Point> list = new ArrayList<>();
        Point p;
        if (from == 0) p = new Point((int)radius,0);
        else if (from == 90) p = new Point(0,(int)radius);
        else if (from == 180) p = new Point((int)-radius,0);
        else p = new Point(0,(int)-radius);
        Point comp;
        int index = 0;
        boolean found = false;
        while (!found && index < points.size()) {
            comp = points.get(index);
            if (comp.x == p.x && comp.y == p.y) found = true;
            else index++;
        }
        for(int i = index;i < points.size();i++) {
            list.add(points.get(i));
        }
        add(line.getLinePoints(),list);
        return list;
    }
    
    public List<Point> getCounterClockwiseWithLine(List<Point> points,int from,Line line) {
        List<Point> list = new ArrayList<>();
        Point p;
        if (from == 0) p = new Point((int)radius,0);
        else if (from == 90) p = new Point(0,(int)radius);
        else if (from == 180) p = new Point((int)-radius,0);
        else p = new Point(0,(int)-radius);
        Point comp;
        int index = points.size() - 1;
        boolean found = false;
        while (!found && index < points.size()) {
            comp = points.get(index);
            if (comp.x == p.x && comp.y == p.y) found = true;
            else index--;
        }
        for(int i = index; i >= 0;i--) {
            list.add(points.get(i));
        }
        add(line.getLinePoints(),list);
        return list;
    }
    
    public List<Point> getFillPoints(Line startLine, Line stopLine, List<Point> arcPoints) {
        List<Point> fillPoints = new ArrayList<>();
        double start = startLine.getAngle();
        double stop = stopLine.getAngle();
        if (inQuadrant1(start) && inQuadrant1(stop)) {
            return getWedgePoints(getAllPoints(startLine,stopLine,arcPoints));
        } else if (inQuadrant2(start) && inQuadrant2(stop)) {
            return getWedgePoints(getAllPoints(startLine,stopLine,arcPoints));
        } else if (inQuadrant3(start) && inQuadrant3(stop)) {
            return getWedgePoints(getAllPoints(startLine,stopLine,arcPoints));
        } else if (inQuadrant4(start) && inQuadrant4(stop)) {
            return getWedgePoints(getAllPoints(startLine,stopLine,arcPoints));
        } else if (start == 270) {
            if (stop == 0) {
                return getQuadrant1FillPoints();
            } else if (stop == 90) {
                add(getQuadrant1FillPoints(),fillPoints);
                add(getQuadrant4FillPoints(),fillPoints);
                return fillPoints;
            } else if (stop == 180) {
                add(getQuadrant1FillPoints(),fillPoints);
                add(getQuadrant4FillPoints(),fillPoints);
                add(getQuadrant3FillPoints(),fillPoints);
                return fillPoints;
            } else if (inQuadrant1(stop)) {
                add(getStopQ1(getClockwiseWithLine(arcPoints,270,stopLine)),fillPoints);
                return fillPoints;
            } else if (inQuadrant4(stop)) {
                add(getQuadrant1FillPoints(),fillPoints);
                add(getStopQ4(getClockwiseWithLine(arcPoints,0,stopLine)),fillPoints);
                return fillPoints;
            } else if (inQuadrant3(stop)) {
                add(getQuadrant1FillPoints(),fillPoints);
                add(getQuadrant4FillPoints(),fillPoints);
                add(getStopQ3(getClockwiseWithLine(arcPoints,90,stopLine)),fillPoints);
                return fillPoints;
            } else if (inQuadrant2(stop)) {
                add(getQuadrant1FillPoints(),fillPoints);
                add(getQuadrant4FillPoints(),fillPoints);
                add(getQuadrant3FillPoints(),fillPoints);
                add(getStopQ2(getClockwiseWithLine(arcPoints,180,stopLine)),fillPoints);
                return fillPoints;
            }
        } else if (start == 0) {
            if (stop == 90) {
                return getQuadrant4FillPoints();
            } else if (stop == 180) {
                add(getQuadrant4FillPoints(),fillPoints);
                add(getQuadrant3FillPoints(),fillPoints);
                return fillPoints;
            } else if (stop == 270) {
                add(getQuadrant4FillPoints(),fillPoints);
                add(getQuadrant3FillPoints(),fillPoints);
                add(getQuadrant2FillPoints(),fillPoints);
                return fillPoints;
            } else if (inQuadrant4(stop)) {
                add(getQuadrant1FillPoints(),fillPoints);
                add(getStopQ4(getClockwiseWithLine(arcPoints,0,stopLine)),fillPoints);
            } else if (inQuadrant3(stop)) {
                add(getQuadrant1FillPoints(),fillPoints);
                add(getQuadrant4FillPoints(),fillPoints);
                add(getStopQ3(getClockwiseWithLine(arcPoints,90,stopLine)),fillPoints);
            } else if (inQuadrant2(stop)) {
                add(getQuadrant4FillPoints(),fillPoints);
                add(getQuadrant3FillPoints(),fillPoints);
                add(getStopQ2(getClockwiseWithLine(arcPoints,180,stopLine)),fillPoints);
            }
        } else if (start == 90) {
            if (stop == 180) {
                return getQuadrant3FillPoints();
            } else if (stop == 270) {
                add(getQuadrant3FillPoints(),fillPoints);
                add(getQuadrant2FillPoints(),fillPoints);
                return fillPoints;
            } else if (inQuadrant3(stop)) {
                add(getStopQ3(getClockwiseWithLine(arcPoints,90,stopLine)),fillPoints);
            } else if (inQuadrant2(stop)) {
                add(getQuadrant3FillPoints(),fillPoints);
                add(getStopQ2(getClockwiseWithLine(arcPoints,180,stopLine)),fillPoints);
            }
        } else if (start == 180) {
            if (stop == 270) {
                return getQuadrant2FillPoints();
            } else if (inQuadrant3(stop)) {
                add(getStopQ3(getClockwiseWithLine(arcPoints,90,stopLine)),fillPoints);
            } else if (inQuadrant2(stop)) {
                add(getStopQ2(getClockwiseWithLine(arcPoints,180,stopLine)),fillPoints);
            }
        } else if (stop == 270) {
            if (start == 0) {
                add(getQuadrant4FillPoints(),fillPoints);
                add(getQuadrant3FillPoints(),fillPoints);
                add(getQuadrant2FillPoints(),fillPoints);
            } else if (start == 90) {
                add(getQuadrant3FillPoints(),fillPoints);
                add(getQuadrant2FillPoints(),fillPoints);
            } else if (start == 180) {
                add(getQuadrant2FillPoints(),fillPoints);
            } else if (inQuadrant2(start)) {
                add(getStartQ2(getCounterClockwiseWithLine(arcPoints,270,startLine)),fillPoints);
            } else if (inQuadrant3(start)) {
                add(getQuadrant2FillPoints(),fillPoints);
                add(getStartQ3(getCounterClockwiseWithLine(arcPoints,180,startLine)),fillPoints);
            } else if (inQuadrant4(start)) {
                add(getQuadrant2FillPoints(),fillPoints);
                add(getQuadrant3FillPoints(),fillPoints);
                add(getStartQ4(getCounterClockwiseWithLine(arcPoints,90,startLine)),fillPoints);
            } else if (inQuadrant1(start)) {
                add(getQuadrant2FillPoints(),fillPoints);
                add(getQuadrant3FillPoints(),fillPoints);
                add(getQuadrant4FillPoints(),fillPoints);
                add(getStartQ1(getCounterClockwiseWithLine(arcPoints,0,startLine)),fillPoints);
            }
            add(getStopQ2(getClockwiseWithLine(arcPoints,180,stopLine)),fillPoints);
        } else if (inQuadrant1(start) && inQuadrant2(stop)) {
            add(getStartQ1(getCounterClockwiseWithLine(arcPoints,0,startLine)),fillPoints);
            add(getQuadrant4FillPoints(),fillPoints);
            add(getQuadrant3FillPoints(),fillPoints);
            add(getStopQ2(getClockwiseWithLine(arcPoints,180,stopLine)),fillPoints);
        } else if (inQuadrant1(start) && inQuadrant3(stop)) {
            add(getStartQ1(getCounterClockwiseWithLine(arcPoints,0,startLine)),fillPoints);
            add(getQuadrant4FillPoints(),fillPoints);
            add(getStopQ3(getClockwiseWithLine(arcPoints,90,stopLine)),fillPoints);
        } else if (inQuadrant1(start) && inQuadrant4(stop)) {
            add(getStartQ1(getCounterClockwiseWithLine(arcPoints,0,startLine)),fillPoints);
            add(getStopQ4(getClockwiseWithLine(arcPoints,0,stopLine)),fillPoints);
        } else if (inQuadrant4(start) && inQuadrant3(stop)) {
            add(getStartQ4(getCounterClockwiseWithLine(arcPoints,90,startLine)),fillPoints);
            add(getStopQ3(getClockwiseWithLine(arcPoints,90,stopLine)),fillPoints);
        } else if (inQuadrant4(start) && inQuadrant2(stop)) {
            add(getStartQ4(getCounterClockwiseWithLine(arcPoints,90,startLine)),fillPoints);
            add(getQuadrant3FillPoints(),fillPoints);
            add(getStopQ2(getClockwiseWithLine(arcPoints,180,stopLine)),fillPoints);
        } else if (inQuadrant3(start) && inQuadrant2(stop)) {
            add(getStartQ3(getCounterClockwiseWithLine(arcPoints,180,startLine)),fillPoints);
            add(getStopQ2(getClockwiseWithLine(arcPoints,180,stopLine)),fillPoints);
        }
        return fillPoints;
    }
    
    public List<Point> getLastArc(int start) {
        //System.out.println("Start: " + getNormalAngle(start));
        List<Point> points = new ArrayList<>();
        List<Point> degree;
        for(int i = getNormalAngle(start); i < 270;i++) {
            degree = getDegree(i);
            if (degree != null) {
                add(degree,points);
            }
        }
        add(getQ2Of270(),points);
        return points;
    }
    
    public Point getPointAtDegree(int angle) {
        if (angle == 0) return new Point((int)radius,0);
        if (angle == 90) return new Point(0,(int)radius);
        if (angle == 180) return new Point((int)-radius,0);
        if (angle == 270) return new Point(0,(int)-radius);
        return null;
    }
    
    public List<Point> getQuadrant1FillPoints() {
        List<Point> points = new ArrayList<>();
        List<Point> q1 = getQuadrant1();
        Point p;
        for(int i = 0; i < q1.size();i++) {
            p = q1.get(i);
            if (p.y <= 0) {
                for(int x = 0; x < p.x;x++) {
                    points.add(new Point(x,p.y));
                }
            }
        }
        return points;
    }
    
    public List<Point> getQuadrant2FillPoints() {
        List<Point> points = new ArrayList<>();
        List<Point> q2 = getQuadrant2();
        Point p;
        for(int i = 0;i < q2.size();i++) {
            p = q2.get(i);
            if (p.y <= 0) {
                for(int x = p.x;x <= 0;x++) {
                    points.add(new Point(x,p.y));
                }
            }
        }
        return points;
    }
    
    public List<Point> getQuadrant3FillPoints() {
        List<Point> points = new ArrayList<>();
        List<Point> q3 = getQuadrant3();
        Point p;
        for(int i = 0; i < q3.size();i++) {
            p = q3.get(i);
            if (p.y >= 0) {
                for(int x = p.x + 1;x <= 0;x++) {
                    points.add(new Point(x,p.y));
                }
            }
        }
        return points;
    }
    
    public List<Point> getQuadrant4FillPoints() {
        List<Point> points = new ArrayList<>();
        List<Point> q4 = getQuadrant4();
        Point p;
        for(int i = 0; i < q4.size();i++) {
            p = q4.get(i);
            for(int x = 0; x < p.x;x++) {
                points.add(new Point(x,p.y));
            }
        }
        return points;
    }

    public List<List<Point>> getSortedByX(List<Point> points) {
        List<List<Point>> exes = new ArrayList<>();
        List<Point> col;
        int start = (int)-radius;
        Point p;
        while (start < radius) {
            col = new ArrayList<>();
            for(int i = 0; i < points.size();i++) {
                p = points.get(i);
                if (p.x == start) col.add(p);
            }
            if (col.size() > 0) {
                exes.add(col);
            }
            start++;
        }
        return exes;
    }
    
    public List<List<Point>> getSortedByY(List<Point> points) {
        List<List<Point>> wyes = new ArrayList<>();
        List<Point> row;
        int start = (int)-radius;
        Point p;
        while (start < radius) {
            row = new ArrayList<>();
            for(int i = 0; i < points.size();i++) {
                p = points.get(i);
                if (p.y == start) row.add(p);
            }
            if (row.size() > 0) {
                wyes.add(row);
            }
            start++;
        }
        return wyes;
    }
    
    public List<Point> getStartQ1(List<Point> linePoints) {
        List<Point> points = new ArrayList<>();
        List<List<Point>> onX = getSortedByX(linePoints);
        List<Point> wyes;
        Point p;
        int maxY;
        for(int i = 0; i < onX.size();i++) {
            wyes = onX.get(i);
            maxY = wyes.get(0).y;
            for(int j = 1;j < wyes.size();j++) {
                p = wyes.get(j);
                if (p.y > maxY) maxY = p.y;
            }
            for(int j = maxY; j < 0;j++) {
                points.add(new Point(wyes.get(0).x,j));
            }
        }
        return points;
    }
    
    public List<Point> getStartQ2(List<Point> linePoints) {
        List<Point> points = new ArrayList<>();
        List<List<Point>> onY = getSortedByY(linePoints);
        List<Point> exes;
        Point p;
        int maxX;
        for(int i = 0; i < onY.size();i++) {
            exes = onY.get(i);
            maxX = exes.get(0).x;
            for(int j = 1;j < exes.size();j++) {
                p = exes.get(j);
                if (p.x > maxX) maxX = p.x;
            }
            for(int j = maxX + 1; j <= 0;j++) {
                points.add(new Point(j,exes.get(0).y));
            }
        }
        return points;
    }
    
    public List<Point> getStartQ3(List<Point> linePoints) {
        List<Point> points = new ArrayList<>();
        List<List<Point>> onX = getSortedByX(linePoints);
        List<Point> exes = null;
        Point p;
        int maxY;
        for(int i = 0; i < onX.size();i++) {
            exes = onX.get(i);
            maxY = exes.get(0).y;
            for(int j = 1;j < exes.size();j++) {
                p = exes.get(j);
                if (p.y < maxY) maxY = p.y;
            }
            for(int j = 0; j < maxY;j++) {
                points.add(new Point(exes.get(0).x,j));
            }
        }
        return points;
    }

    public List<Point> getStartQ4(List<Point> linePoints) {
        List<Point> points = new ArrayList<>();
        List<List<Point>> onY = getSortedByY(linePoints);
        List<Point> exes;
        Point p;
        int minX;
        for(int i = 0; i < onY.size();i++) {
            exes = onY.get(i);
            minX = exes.get(0).x;
            for(int j = 1;j < exes.size();j++) {
                p = exes.get(j);
                if (p.x < minX) minX = p.x;
            }
            for(int j = 0; j < minX;j++) {
                points.add(new Point(j,exes.get(0).y));
            }
        }
        return points;
    }

    public List<Point> getStopQ1(List<Point> linePoints) {
        List<Point> points = new ArrayList<>();
        List<List<Point>> onY = getSortedByY(linePoints);
        //System.out.println("OnY: " + onY.size());
        List<Point> exes;
        Point p;
        int minX;
        for(int i = 0; i < onY.size();i++) {
            exes = onY.get(i);
            minX = exes.get(0).x;
            //System.out.println("Y: " + exes.get(0).y);
            for(int j = 1;j < exes.size();j++) {
                p = exes.get(j);
                if (p.x < minX) minX = p.x;
            }
            for(int j = 0;j < minX;j++) {
                points.add(new Point(j,exes.get(0).y));
            }
        }
        return points;
    }

    public List<Point> getStopQ2(List<Point> linePoints) {
        List<Point> points = new ArrayList<>();
        List<List<Point>> onX = getSortedByX(linePoints);
        List<Point> wyes;
        Point p;
        int minY;
        for(int i = 0; i < onX.size();i++) {
            wyes = onX.get(i);
            minY = wyes.get(0).y;
            for(int j = 1;j < wyes.size();j++) {
                p = wyes.get(j);
                if (p.y > minY) minY = p.y;
            }
            for(int j = minY + 1; j < 0;j++) {
                points.add(new Point(wyes.get(0).x,j));
            }
        }
        return points;
    }
    
    public List<Point> getStopQ3(List<Point> linePoints) {
        List<Point> points = new ArrayList<>();
        List<List<Point>> onY = getSortedByY(linePoints);
        List<Point> exes;
        Point p;
        int maxX;
        for(int i = 0; i < onY.size();i++) {
            exes = onY.get(i);
            maxX = exes.get(0).x;
            for(int j = 1;j < exes.size();j++) {
                p = exes.get(j);
                if (maxX < p.x) maxX = p.x;
            }
            for(int j = maxX + 1; j < 0;j++) {
                points.add(new Point(j,exes.get(0).y));
            }
        }
        return points;
    }
    
    public List<Point> getStopQ4(List<Point> linePoints) {
        List<Point> points = new ArrayList<>();
        List<List<Point>> onX = getSortedByX(linePoints);
        List<Point> wyes;
        Point p;
        int maxY;
        for(int i = 0; i < onX.size();i++) {
            wyes = onX.get(i);
            maxY = wyes.get(0).y;
            for(int j = 1;j < wyes.size();j++) {
                p = wyes.get(j);
                if (maxY < p.y) maxY = p.y;
            }
            for(int j = 0; j < maxY;j++) {
                points.add(new Point(wyes.get(0).x,j));
            }
        }
        return points;
    }

    public List<Point> getWedgePoints(List<Point> points) {
        List<List<Point>> onY = getSortedByY(points);
        List<Point> row;
        Point p;
        Point left;
        Point right;
        for(int i = 0;i < onY.size();i++) {
            row = onY.get(i);
            left = null;
            right = null;
            if (row.size() >= 2) {
                left = row.get(0);
                right = row.get(0);
                for(int j = 1; j < row.size();j++) {
                    p = row.get(j);
                    if (p.x < left.x) {
                        left = p;
                    } else if (p.x > right.x) {
                        right = p;
                    }
                }
                List<Point> lefts = new ArrayList<>();
                List<Point> rights = new ArrayList<>();
                int cx = (right.x - left.x) / 2;
                for(int j = 0; j < row.size();j++) {
                    p = row.get(j);
                    if (p.x < cx) lefts.add(p);
                    else rights.add(p);
                }
                if (lefts.size() > 0 && rights.size() > 0) {
                    left = lefts.get(0);
                    right = rights.get(0);
                    for(int j = 1;j < lefts.size();j++) {
                        p = lefts.get(j);
                        if (p.x > left.x) left = p;
                    }
                    for(int j = 1;j < rights.size();j++) {
                        p = rights.get(j);
                        if (p.x < right.x) right = p;
                    }
                }
            }
            if (left != null && right != null) {
                for(int j = left.x + 1; j < right.x;j++) {
                    points.add(new Point(j,left.y));
                }
            }
        }
        return points;
    }
    
    @Override
    public boolean hasFillPoint(Point p) {
        Point txy = getTXY();
        int cx = p.x - txy.x;
        int cy = p.y - txy.y;
        return Math.sqrt(cx * cx + cy * cy) <= radius;
    }
    
    @Override
    public boolean hasPoint(Point p) {
        Point txy = getTXY();
        int tpx, tpy;
        Point current;
        for(int i = 0; i < linePoints.size();i++) {
            current = linePoints.get(i);
            tpx = txy.x + current.x;
            tpy = txy.y + current.y;
            if (p.x == tpx && p.y == tpy) return true;
        }
        return hasFillPoint(p);
    }
    
    public PieChart setTranslation(int tx, int ty) {
        super.setTranslation(tx,ty);
        return this;
    }
    
    public PieChart setTXY(int tx, int ty) {
        return setTranslation(tx,ty);
    }
}
