package ca.tecreations;

import ca.tecreations.components.event.*;
import ca.tecreations.misc.BitSet;

import java.io.DataInputStream;
import java.io.DataOutputStream; 
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.DosFileAttributes;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.GroupPrincipal;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.nio.file.attribute.UserPrincipal;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 *
 * @author Tim
 */
public class File extends java.io.File {
    public static final String SEPARATOR = java.io.File.separator;
    public static final char SEPARATOR_CHAR = java.io.File.separatorChar;
    public static final String PATH_SEPARATOR = java.io.File.pathSeparator;
    public static final char PATH_SEPARATOR_CHAR = java.io.File.pathSeparatorChar;
    public static final String NOT_SEP = (SEPARATOR.equals("\\") ? "/" : "\\");
    public static final char NOT_SEP_CHAR = NOT_SEP.charAt(0);
    
    public static final String SN = File.class.getSimpleName();
    public static TextFile log = null;
    public static final int DOS = 0;
    public static final int POSIX = 1;
    protected int type;

    public static boolean debug = true;
    public static boolean verbose = true;
    
    List<String> basicStats;
    List<String> basicAttributes;

    private String absPath = "";
    
    public File(String path) {
        super(StringTool.getUnwrapped(path));
        String unwrapped = StringTool.getUnwrapped(path);
        if (super.isDirectory()) {
            if (unwrapped.contains("\\")) {
                if (!unwrapped.endsWith("\\")) unwrapped += "\\";
            } else {
                if (!unwrapped.endsWith("/")) unwrapped += "/";
            }
        }
        absPath = StringTool.getDoubleQuoted(path);
    }
 
    public File(java.io.File file) {
        this(file.getAbsolutePath());
        String unwrapped = StringTool.getUnwrapped(super.getAbsolutePath());
        if (super.isDirectory()) {
            if (unwrapped.contains("\\")) {
                if (!unwrapped.endsWith("\\")) unwrapped += "\\";
            } else {
                if (!unwrapped.endsWith("/")) unwrapped += "/";
            }
        }
        absPath = StringTool.getDoubleQuoted(file.getAbsolutePath());
    }
    
    public File(Path path) {
        this(path.toFile());
    }
    
    public static void copyA(File srcFile, File dstDir, boolean debug, boolean verbose) {
        dstDir.mkdirs();
        File deepest = dstDir.getDeepestDirectoryFile();
        deepest.mkdirs();
        FileInputStream fis = null;
        DataInputStream dis = null;
        FileOutputStream fos = null;
        DataOutputStream dos = null;
        try {
            fis = new FileInputStream(srcFile);
            dis = new DataInputStream(fis);
        } catch (IOException ioe) {
            System.err.println("File.copy: opening input(" + srcFile.getAbsolutePath() + " ): " + ioe);
        }
        String outPath = deepest.getUnwrapped() + srcFile.getName();
        try {
            fos = new FileOutputStream(outPath);
            dos = new DataOutputStream(fos);
        } catch (IOException ioe) {
            System.err.println("File.copy: opening output(" + outPath + " ): " + ioe);
        }
        if (dis == null | dos == null) {
            System.err.println("File.copy: NULL stream: dis: " + dis + " dos: " + dos);
        } else {
            doCopy(dis, dos, debug, verbose);
        }
        try {
            if (dos != null) dos.flush();
            if (dos != null) dos.close();
            if (fos != null) fos.close();
            if (dis != null) dis.close();
            if (fis != null) fis.close();
        } catch (IOException ioe) {
            System.err.println("File.copy: While closing: " + ioe);
        }
    }

    public void copyToFile(File newFile,boolean debug, boolean verbose) {
        copyToFile(newFile.getAbsolutePath(),debug, verbose);
    }
    
    public void copyToFile(String newName,boolean debug, boolean verbose) {
        if (log != null) log.add("File.copyToFile: src: " + getAbsolutePath() + " dst: " + newName);
        if (!isFile()) {
            throw new IllegalArgumentException("File.copyToFile: src must be a file: " + getAbsolutePath());
        }
        File deepest = new File(newName).getDeepestDirectoryFile();
        deepest.mkdirs();
        FileInputStream fis = null;
        DataInputStream dis = null;
        FileOutputStream fos = null;
        DataOutputStream dos = null;
        try {
            fis = new FileInputStream(getUnwrapped());
            dis = new DataInputStream(fis);
        } catch (IOException ioe) {
            doLogAction("File.copyToFile: opening input: " + ioe);
        }
        try {
            if (debug) {
                doLogAction("File.copyToFile: dst: " + deepest.getUnwrapped() + getName());
            }
            fos = new FileOutputStream(StringTool.getUnwrapped(newName));
            dos = new DataOutputStream(fos);
        } catch (IOException ioe) {
            doErrAction("File.copyToFile: opening output: " + ioe);
        }
        if (dis == null | dos == null) {
            doLogAction("File.copyToFile: NULL stream: dis: " + dis + " dos: " + dos);
        } else {
            doCopy(dis, dos, debug, verbose);
        }
        try {
            if (dos != null) {
                dos.flush();
            }
            if (dos != null) {
                dos.close();
            }
            if (fos != null) {
                fos.close();
            }
            if (dis != null) {
                dis.close();
            }
            if (fis != null) {
                fis.close();
            }
        } catch (IOException ioe) {
            doErrAction("File.copyToFile: While closing: " + ioe);
        }
    }
    
    public synchronized void copyToDir(File dstPath, boolean debug, boolean verbose) {
        copyToDir(dstPath.getAbsolutePath(),debug,verbose);
    }
    
    public synchronized void copyToDir(String dstPath, boolean debug, boolean verbose) {
        System.out.println(SN + ".copyToDir(" + debug + ":" + verbose + " : absPath: " + getAbsolutePath() + " dstPath: " + dstPath + ")");
        File dst = new File(dstPath);
        if (!dst.isDirectory()) {
            throw new IllegalArgumentException("File.copyToDir(a1): dst must be a directory: " + dstPath);
        }
        copyA(this, dst, debug, verbose);
    }

    public synchronized void copyToDir(String outPath, ProgressListener pl, boolean debug, boolean verbose) {
        File dst = new File(outPath);
        if (!isFile()) {
            throw new IllegalArgumentException("File.copyToDir: src must be a file: " + getAbsolutePath());
             
        }
        long total = 0;
        long length = length();
        double portion = (double) length / (double) 100;
        int percent = 0;
        File deepest = dst.getDeepestDirectoryFile();
        deepest.mkdirs();
        FileInputStream fis = null;
        DataInputStream dis = null;
        FileOutputStream fos = null;
        DataOutputStream dos = null;
        try {
            fis = new FileInputStream(getUnwrapped());
            dis = new DataInputStream(fis);
        } catch (IOException ioe) {
            System.err.println("File.copyToDir(b1): opening input: " + ioe);
        }
        try {
            fos = new FileOutputStream(StringTool.getUnwrapped(outPath + getName()));
            dos = new DataOutputStream(fos);
        } catch (Exception e) {
            ExceptionHandler.handleIO("File.copyToDir(b2)","Opening", e,true);
        }
        if (dis == null | dos == null) {
            System.out.println("File.copyToDir(b3): NULL stream: dis: " + dis + " dos: " + dos);
        } else {
            byte[] buffer = new byte[64000];
            int noOfBytes = 0;
            try {
                while ((noOfBytes = dis.read(buffer)) != -1) {
                    dos.write(buffer, 0, noOfBytes);
                    dos.flush();
                    total += noOfBytes;
                    percent = (int) ((double) total / (double) portion);
                    if (pl != null) pl.updateItem(percent);
                }
            } catch (IOException ioe) {
                System.err.println("File.copyToDir(b4): While copying: " + ioe);
            }
        }
        try {
            if (dos != null) dos.flush();
            if (dos != null) dos.close();
            if (fos != null) fos.close();
            if (dis != null) dis.close();
            if (fis != null) fis.close();
        } catch (IOException ioe) {
            System.err.println("File.copyTo(b5): While closing: " + ioe);
        }
    }

    public synchronized void copyToDir(String outPath, ProgressListener pl) {
        copyToDir(outPath,pl,false,false);
    }
    
    public synchronized boolean delete(boolean output) {
        boolean deleted = delete();
        if (output) {
            System.out.println(SN + ".delete(" + output + "): " + getAbsolutePath() + " : Deleted: " + deleted);
        }
        return deleted;
    }
    
    public synchronized void deleteAllDirectories() {
        File[] entries = tecListFiles();
        if (entries != null) {
            for (int i = 0; i < entries.length; i++) {
                if (debug) {
                    System.out.println(SN + ".deleteAllDirectories: current: " + entries[i].getAbsolutePath());
                }
                if (entries[i].isDirectory()) {
                    prune(entries[i].getUnwrapped(), false);
                    entries[i].delete();
                }
            }
        }
    }

    private synchronized void deleteAllDirectories(java.io.File parentDir) {
        java.io.File[] entries = parentDir.listFiles();
        if (entries != null) {
            for (int i = 0; i < entries.length; i++) {
                if (debug) {
                    System.out.println(SN + ".deleteAllDirectories: current: " + entries[i].getAbsolutePath());
                }
                if (entries[i].isDirectory()) {
                    deleteAllDirectories(entries[i]);
                    entries[i].delete();
                } else {
                    entries[i].delete();
                }
            }
        }
    }

    public static void doCopy(DataInputStream dis, DataOutputStream dos, boolean debug, boolean verbose) {
        byte[] buffer = new byte[64000];
        int noOfBytes = 0;
        long bytesTotal = 0L;
        try {
            while ((noOfBytes = dis.read(buffer)) != -1) {
                dos.write(buffer, 0, noOfBytes);
                dos.flush();
                bytesTotal += noOfBytes;
                if (debug && verbose) {
                    System.out.println(SN + ".doCopy: Wrote: " + noOfBytes + " of " + bytesTotal + " bytes.");
                }
            }
        } catch (IOException ioe) {
            System.err.println(SN + ".doCopy: While copying: " + ioe);
        }
    }

    public void doErrAction(String msg) {
        if (log != null) log.add(msg);
        System.err.println(msg);
//        if (TecData.tls_tsps != null) TecData.tls_tsps.send(msg);
    }

    public void doLogAction(String msg) {
        if (log != null) log.add(msg);
        System.out.println(SN + ".doLogAction: " + msg);
//        if (TecData.tls_tsps != null) TecData.tls_tsps.send(msg);
    }

    public synchronized void empty() {
        prune(getAbsolutePath(), false);
        mkdir();
    }


    public String getAbsolutePath() {
        String unwrapped = StringTool.getUnwrapped(absPath);
        if (super.isDirectory()) {
            if (unwrapped.contains("\\")) {
                if (!unwrapped.endsWith("\\")) unwrapped += "\\";
            } else {
                if (!unwrapped.endsWith("/")) unwrapped += "/";
            }
        }
        return StringTool.getDoubleQuoted(unwrapped);
    }
    
    public String getAppPermissionsString() {
        String s = (canRead() ? "R" : "-");
        s += (canWrite() ? "W" : "-");
        s += (canExecute() ? "X" : "-");
        return s;
    }

    public String getAsDrive(String unwrapped) {
        char drive = unwrapped.charAt(0);
        if (drive == '/') return "/";
        else if ((drive >= 'A' && drive <= 'Z') | (drive >= 'a' && drive <= 'z')) {
            if (unwrapped.charAt(1) == ':' && unwrapped.charAt(2) == '\\') {
                return StringTool.getDoubleQuoted(unwrapped.substring(0,3));
            }
        }
        return null;
    }
    
    public String getClassPath() {
        if (!is("java")) {
            throw new IllegalArgumentException(SN + ".getClassPath: must be a .java file: " + getAbsolutePath());
        }
        String pkg = "";
        String subPath;
        List<String> lines = new TextFile(getAbsolutePath()).getLines();
        String line;
        for(int i = 0;i < lines.size();i++) {
            line = lines.get(i).trim();
            if (line.startsWith("package")) {
                pkg = line.substring(8,line.indexOf(";")).trim();
                break;
            }
        }
        if (pkg.length() > 0) pkg += ".";
        //System.out.println("Package: " + pkg);
        subPath = StringTool.replaceAll(pkg,".",File.separator);
        //System.out.println("SubPath: " + subPath);
        String unwrapped = getUnwrapped();
        //System.out.println("Unwrapped: " + unwrapped);
        int index = unwrapped.indexOf(subPath);
        return unwrapped.substring(0,index);
    } 
    
    public String getDeepestDirectory() {
        if (isFile()) {
            return StringTool.getDoubleQuoted(getParentFile().getAbsolutePath());
        } else {
            return getAbsolutePath();
        }
    }
    
    public File getDeepestDirectoryFile() {
        String unwrapped = getUnwrapped();
        if (unwrapped.contains("/")) {
            return new File(unwrapped.substring(0,unwrapped.lastIndexOf("/")));
        } else {
            return new File(unwrapped.substring(0,unwrapped.lastIndexOf("\\")));
        }
    }
    
    public static long getDirSize(String path) {
        long size = 0L;
        File[] list = new File(path).tecListFiles();
        if (list != null) {
            for (int i = 0; i < list.length; i++) {
                if (list[i].isDirectory()) {
                    long sizeToAdd = getDirSize(list[i].getAbsolutePath());
                    size += sizeToAdd;
                } else {
                    size += list[i].length();
                }
            }
        }
        return size;
    }

    public static File[] getDirs(String path) {
        File[] entries = new File(path).tecListFiles();
        int count = 0;
        for(int i = 0; i < entries.length;i++) if (entries[i].isDirectory()) count++;
        File[] dirs = new File[count];
        count = 0;
        for(int i = 0;i < entries.length;i++) if (entries[i].isDirectory()) dirs[count++] = entries[i];
        return dirs;
    }
    
    public static List<File> getDirsList(String path) {
        File[] entries = new File(path).tecListFiles();
        List<File> dirs = new ArrayList<>();
        if (entries != null) {
            for(int i = 0; i < entries.length;i++) if (entries[i].isDirectory()) dirs.add(entries[i]);
        }
        return dirs;
    }
    
    public DosFileAttributes getDOSFileAttributes() {
        DosFileAttributes attr = null;
 
        try {
            attr = Files.readAttributes(toPath(), DosFileAttributes.class);
        } catch (IOException ioe) {
            System.out.println("While attempting to get attributes: " + ioe);
        } catch (UnsupportedOperationException y) {
            System.err.println("DOS file attributes not supported:" + y);
        }
        return attr;
    }

    public String getDOSAttributesString() {
        DosFileAttributes attr = getDOSFileAttributes();
        String s = (attr.isHidden() ? "H" : "-");
        s += (attr.isArchive() ? "A" : "-");
        s += (attr.isReadOnly() ? "R" : "-");
        s += (attr.isSystem() ? "S" : "-");
        return s;
    }

    public String getExtension() {
        String name = getName();
        //System.err.println("File.getExtension: " + name);
        if (name.indexOf(".") == -1) {
            return "";
        } else if (name.lastIndexOf(".") > 0) {
            String ext = name.substring(name.lastIndexOf(".") + 1);
            //System.err.println("File.getExtension: name: " + name + " ext: " + ext);
            return ext;
        } else {
            return "";
        }
    }

    public static String getExtension(String path) {
        File f = new File(path);
        return f.getExtension();
    }

    public String getExtensionLower() {
        return getExtension().toLowerCase();
    }

    public static String getExtensionLower(String path) {
        File f = new File(path);
        return f.getExtensionLower();
    }

    public static String getExtensionLower(File file) {
        String unwrapped = file.getUnwrapped();
        return unwrapped.substring(unwrapped.lastIndexOf(".") + 1).toLowerCase();
    }

    public FileInputStream getFileInputStream() {
        FileInputStream in = null;
        try {
            in = new FileInputStream(getUnwrapped());
        } catch (FileNotFoundException fnfe) {
            System.err.println("File not found: " + getAbsolutePath());
        }
        return in;
    }

    public List<String> getFileLines() {
        TextFile file = new TextFile(getAbsolutePath());
        return file.getLines();
    }
    
    /**
     *
     * @return The files name, excluding the file extension.
     */
    public String getFilenameOnly() { // for convenience, make it intuitive, smooth, quick and easy. Help them make it better, quicker.
        return getFileNameOnly();
    }

    /**
     *
     * @return The files name, excluding the file extension.
     */
    public String getFileNameOnly() {
        String name = getName();
        if (name.contains(".")) {
            name = name.substring(0, name.lastIndexOf("."));
        }
        return name;
    }

    public static List<File> getFiles(String absPath) {
        //System.out.println("File.getFiles(" + absPath + ")");
        List<File> files = new ArrayList<>();
        File[] entries = new File(absPath).tecListFiles();
        File entry;
        if (entries != null) {
            for (int i = 0; i < entries.length; i++) {
                if (entries[i].isFile()) {
                    files.add(entries[i]);
                }
            }
        }
        File[] filesArray = new File[files.size()];
        for (int i = 0; i < files.size(); i++) {
            filesArray[i] = files.get(i);
        }
        List<String> sorted = Sort.getSortedByName(filesArray);
        List<File> result = new ArrayList<>();
        for (int i = 0; i < sorted.size(); i++) {
            result.add(new File(sorted.get(i)));
        }
        return result;
    }

    public static List<File> getFiles(String absPath, String ext) {
        //System.out.println("File.getFiles(" + absPath + ")");
        List<File> files = new ArrayList<>();
        File[] entries = new File(absPath).tecListFiles();
        File entry;
        if (entries != null) {
            for (int i = 0; i < entries.length; i++) {
                if (entries[i].isFile() && entries[i].getExtension().equals(ext)) {
                    files.add(entries[i]);
                }
            }
        }
        File[] filesArray = new File[files.size()];
        for (int i = 0; i < files.size(); i++) {
            filesArray[i] = files.get(i);
        }
        List<String> sorted = Sort.getSortedByName(filesArray);
        List<File> result = new ArrayList<>();
        for (int i = 0; i < sorted.size(); i++) {
            result.add(new File(sorted.get(i)));
        }
        return result;
    }

    public String getFQCN() {
        if (!is("java")) {
            throw new IllegalArgumentException(SN + ".getFQCN: must be a .java file: " + getAbsolutePath());
        }
        String pkg = "";
        String subPath;
        List<String> lines = new TextFile(getAbsolutePath()).getLines();
        String line;
        for(int i = 0;i < lines.size();i++) {
            line = lines.get(i).trim();
            if (line.startsWith("package")) {
                pkg = line.substring(8,line.indexOf(";")).trim();
                i = lines.size();
            }
        }
        if (pkg.length() > 0) pkg += ".";
        String fqcn = pkg + getFileNameOnly();
        return fqcn;
    }

    @Override
    public long getFreeSpace() {
        if (hasARoot()) {
            return super.getFreeSpace();
        } else {
            return -1L;
        }
    }
    
    public List<File> getListAll() {
        List<File> files = new ArrayList<>();
        ca.tecreations.File[] entries = tecListFiles();
        if (entries != null) {
            for(int i = 0; i < entries.length;i++) files.add(entries[i]);
        }
        return files;
    }
    
    public List<File> getListDirs() {
        List<File> dirs = new ArrayList<>();
        File[] entries = tecListFiles();
        for(int i = 0; i < entries.length;i++) {
            if (entries[i].isDirectory()) {
                dirs.add(new File(entries[i].getAbsolutePath()));
            }
        }
        dirs = Sort.sortListOfFile(dirs);
        return dirs;
    }
    public List<File> getListFiles() {
        List<File> files = new ArrayList<>();
        File[] entries = tecListFiles();
        for(int i = 0; i < entries.length;i++) {
            if (entries[i].isFile()) {
                files.add(new File(entries[i].getAbsolutePath()));
            }
        }
        files = Sort.sortListOfFile(files);
        return files;
    }
    
    /* Example: Iterating signatures for completion. */
    
    public static File getNewestDir(File file) {
        return file.getNewestDirectory();
    }
    public static File getNewestDir(String path) {
        return new File(path).getNewestDirectory();
    }
    public static File getNewestDirectory(File file) {
        return file.getNewestDirectory();
    }
    public static File getNewestDirectory(String path) {
        return new File(path).getNewestDirectory();
    }
    public File getNewestDirectory() {
        File[] list = tecListFiles();
        if (list != null && list.length > 0) {
            File found = null;
            File target = null;
            for (int i = 0; i < list.length; i++) {
                target = list[i];
                if (target.isDirectory()) {
                    if (found == null) {
                        found = list[i];
                    } else {
                        target = list[i];
                        if (target.lastModified() > found.lastModified()) {
                            found = target;
                        }
                    }
                }
            }
            return found;
        } else {
            return null;
        }
    }

    public static File getNewestFile(File file) {
        return file.getNewestDirectory();
    }
    public static File getNewestFile(String path) {
        return new File(path).getNewestFile();
    }
    public File getNewestFile() {
        File[] list = tecListFiles();
        if (list != null && list.length > 0) {
            File found = null;
            File target = null;
            for (int i = 0; i < list.length; i++) {
                target = list[i];
                if (target.isFile()) {
                    if (found == null) {
                        found = list[i];
                    } else {
                        target = list[i];
                        if (target.lastModified() > found.lastModified()) {
                            found = target;
                        }
                    }
                }
            }
            return found;
        } else {
            return null;
        }
    }

    /**
     * Conform this in such a way that it receives the destination directory and,
     * if necessary, constructs a new directory such that the pattern dir, dir (1), 
     * dir (2), etc is fulfilled.
     * @param path the output path for which we may create a new path name.
     * @return the new path name, if any, otherwise, the non-existent path.
     */
    public static String getNextDirectoryName(String path) {
        File f = new File(path);
        if (f.exists()) {
            String parent = f.getTecParentFile().getUnwrapped();
            String name = f.getFilenameOnly();
            String ext = f.getExtension();
            if (name.contains(" (") && name.contains(")")) {
                int start = name.lastIndexOf(" (") + 2;
                int end = name.lastIndexOf(")");
                String indexString = name.substring(start, end);
                String nameString = name.substring(0, name.lastIndexOf("(") + 1);
                int index = Integer.valueOf(indexString) + 1;
                String newName = parent + nameString + index + ")";
                if (!ext.equals("")) {
                    newName += "." + ext;
                }
                return getNextDirectoryName(newName);
            } else {
                String newName = parent + name + " (1)";
                if (!ext.equals("")) {
                    newName += "." + ext;
                }
                if (new File(newName).exists()) {
                    return getNextDirectoryName(newName);
                } else {
                    return (ext.equals("") ? StringTool.getDoubleQuoted(newName + File.separator) : StringTool.getDoubleQuoted(newName));
                }
            }
        }
        String unwrapped = f.getUnwrapped();
        if (unwrapped.contains("\\") && ! unwrapped.endsWith("\\")) unwrapped += "\\";
        else if (unwrapped.contains("/") && !unwrapped.endsWith("/")) unwrapped += "/";
        return StringTool.getDoubleQuoted(unwrapped);
    }

    /** @see getNextDirectoryName */
    public static String getNextFileName(String path) {
        File f = new File(path);
        if (f.isDirectory()) {
            return getNextDirectoryName(path);
        }
        File deepest = f.getDeepestDirectoryFile();
        String name = f.getFilenameOnly();
        String ext = f.getExtension();
        if (f.exists()) {
            if (name.contains(" (") && name.contains(")")) {
                int start = name.lastIndexOf(" (") + 2;
                int end = name.lastIndexOf(")");
                String indexString = name.substring(start, end);
                //System.out.println("IndexString: '" + indexString + "'");
                String nameString = name.substring(0, name.lastIndexOf("(") + 1);
                //System.out.println("NameString : '" + nameString + "'");
                int index = Integer.valueOf(indexString) + 1;
                String newName = deepest.getUnwrapped() + nameString + index + ")";
                if (!ext.equals("")) {
                    newName += "." + ext;
                }
                return getNextFileName(newName);
            } else {
                String newName = deepest.getUnwrapped() + name + " (1)";
                if (!ext.equals("")) {
                    newName += "." + ext;
                }
                return getNextFileName(newName);
            }
        }
        return f.getAbsolutePath();
    }

    public static File getOldestDir(File file) {
        return file.getOldestDirectory();
    } // group by name, alphabetically...
    public static File getOldestDir(String path) {
        return new File(path).getOldestFile();
    } // Digits ahead of Numbers ahead of underscores[ _ ]. 
    public static File getOldestDirectory(String path) {
        return new File(path).getNewestDirectory();
    } // [ and ] are square brackets.
    public static File getOldestDirectory(File file) {
        return file.getOldestDirectory();
    } // Individual statements must end in ; by end of '{item}'.
    public File getOldestDirectory() {
        FileTime ft1 = null;
        FileTime ft2 = null;
        File[] list = tecListFiles();
        if (list != null && list.length > 0) {
            File found = null;
            File target = null;
            //System.out.println("File: " + found);
            for (int i = 0; i < list.length; i++) {
                //System.out.println("File: " + list[i]);
                target = list[i];
                if (target.isDirectory()) {
                    if (found == null) {
                        found = list[i];
                    } else {
                        try {
                            ft1 = Files.getLastModifiedTime(Paths.get(found.getUnwrapped()), LinkOption.NOFOLLOW_LINKS);
                            ft1.to(TimeUnit.DAYS);
                            ft2 = Files.getLastModifiedTime(Paths.get(target.getUnwrapped()), LinkOption.NOFOLLOW_LINKS);
                            //System.out.println(ft1.toString());
                            //System.out.println(ft2.toString());
                        } catch (IOException ioe) {
                        }
                        if (ft1 != null && ft2 != null) {
                            long l = ft2.to(TimeUnit.DAYS);
                            //System.out.println(l + ": " + (ft2.compareTo(ft1)));
                            if (ft2.compareTo(ft1) == -1) {
                                found = target;
                            }
                        }
                    }
                }
            }
            return found;
        } else {
            return null;
        }
    }  // root code  // I enclose 'parts' inside other 'parts' to separate by style in comments.

    public static File getOldestFile(File file) {
        return file.getOldestFile();
    }
    public static File getOldestFile(String path) {
        return new File(path).getOldestFile();
    }
    public File getOldestFile() {
        File[] list = tecListFiles();
        List<File> files = new ArrayList<>();
        if (list != null && list.length > 0) {
            for (int i = 0; i < list.length; i++) {
                if (list[i].isFile()) {
                    files.add(list[i]);
                }
            }
        }
        File oldest = null;
        if (files.size() == 0) ; else if (files.size() == 1) {
            return files.get(0);
        } else {
            oldest = files.get(0);
            for (int i = 1; i < files.size(); i++) {
                if (files.get(i).lastModified() < oldest.lastModified()) {
                    oldest = files.get(i);
                }

            }
        }
        return oldest;
    }

    
    public String getPackage() {
        if (!is("java")) {
            throw new IllegalArgumentException(SN + ".getPackage: must be a .java file: " + getAbsolutePath());
        }
        String pkg = "";
        String subPath;
        List<String> lines = new TextFile(getAbsolutePath()).getLines();
        String line;
        for(int i = 0;i < lines.size();i++) {
            line = lines.get(i).trim();
            if (line.startsWith("package")) {
                pkg = line.substring(8,line.indexOf(";")).trim();
            }
        }
        return pkg;
    }
    
    public String getPackage_Deriven(String root, String pathObject) {
        String unwrappedRoot = StringTool.getUnwrapped(root);
        String unwrappedPath = StringTool.getUnwrapped(pathObject); // file or directory or root
        System.out.println(SN + ".getPackage_Deriven: root: " + root + " unwrappedPath: " + unwrappedPath);
        String sep = "";
        if (unwrappedRoot.contains("/")) {
            sep = "/";
            if (!unwrappedRoot.endsWith("/")) unwrappedRoot += "/";
        } else {
            sep = "\\";
            if (!unwrappedRoot.endsWith("\\")) unwrappedRoot += "\\";
        }
        String subPath = unwrappedPath.substring(unwrappedRoot.length(),unwrappedPath.lastIndexOf(sep));
        String pkg = StringTool.replaceAll(subPath,sep,".");
        return pkg; 
    }
    
    public static String getPermissionString(int val) {
        String permission = "";
        BitSet bitSet = new BitSet(val);
        permission += (bitSet.isSet(0) ? "R" : "-");
        permission += (bitSet.isSet(1) ? "W" : "-");
        permission += (bitSet.isSet(2) ? "X" : "-");
        return permission;
    }

    public PosixFileAttributes getPOSIXAttributes() {
        Path file = toPath();
        PosixFileAttributes attr = null;

        try {
            attr = Files.readAttributes(file, PosixFileAttributes.class
            );
        } catch (IOException ioe) {
            System.out.println("Couldn't retrieve POSIX attributes for: " + getAbsolutePath());
        }
        return attr;
    }

    public String getPOSIXAttributesCSV() {
        PosixFileAttributes attr = getPOSIXAttributes();
        return attr.owner().getName() + "," + attr.group().getName() + "," + PosixFilePermissions.toString(attr.permissions());
    }

    public String getPOSIXFilePermissions() {
        PosixFileAttributes attrs = getPOSIXAttributes();
        return PosixFilePermissions.toString(attrs.permissions());
    } 
    
    public static String getPOSIXFilePermissionString(int owner, int group, int other) {
        String permissions = "";
        permissions += getPermissionString(owner);
        permissions += getPermissionString(group);
        permissions += getPermissionString(other);
        return permissions;
    }
    
    
    
    public String getPOSIXGroup() {
        PosixFileAttributes attrs = getPOSIXAttributes();
        return attrs.group().getName();
    } 
    
    public String getPOSIXUser() {
        PosixFileAttributes attrs = getPOSIXAttributes();
        return attrs.owner().getName();
    } 
    
    public static String getSimpleName(String path) {
        String name = new File(path).getName();
        if (name.contains(".")) {
            return name.substring(0, name.lastIndexOf("."));
        }
        return name;
    }
    
    public Long getSize() {
        if (isDirectory()) return getDirSize(getAbsolutePath());
        else return length();
    }
    
    public String getSubPackage(String classPath,String unwrapped) {
        return StringTool.replaceAll(getSubPath(classPath,unwrapped),File.separatorChar,'.');
    }
    
    /**
     * gets a subpath component for the current Platform
     * @param src
     * @return anything trailing after /, or \, or a fqcn to File.separator or
     *         the src components full value, minus any leading '/' or '\'
     *         sample: \path\to\dir\ returns path\to\dir\ on windows and path/to/dir/ on nix
     *         sample: someUnknownString returns someUnknownString ?+= File.separator
     *         sample: ca.tecreations.apps returns ca\tecreations\apps\ on Windows, and ca/tecreations/apps/ on nix
     */
    public static String getSubPath(String src) {
        boolean prefixed = false;
        char prefix;
        String separator;
        String subPart = src;
        if (src.startsWith("/") || src.startsWith("\\")) {
            prefixed = true;
            prefix = src.charAt(0);
            subPart = src.substring(1);
        } 
        boolean doForDot = false;
        boolean doForSlash = false;
        boolean doForBackSlash = false;
        if (subPart.contains(".") && !subPart.contains("\\") && !subPart.contains("/")) {
            doForDot = true;
        } else if (subPart.contains("\\")) {
            doForBackSlash = true;
        } else {
            doForSlash = true;
        }
        String nextPart = subPart;
        if (doForDot) {
            nextPart = StringTool.replaceAll(subPart,".",File.separator);
        } else if (doForBackSlash) {
            nextPart = StringTool.replaceAll(subPart,"\\", File.separator);
        } else if (doForSlash) {
            nextPart = StringTool.replaceAll(subPart,"/",File.separator);
        }
        if (!nextPart.endsWith(File.separator)) {
            nextPart += File.separator; // always return a subPath type String
        }
        return nextPart;
    }
    
    public static String getSubPath(File file, String root) {
        if (!file.isFile()) {
            throw new IllegalArgumentException("Specified ca.tecreations.File is a directory -- must be a file in call to getSubPath(File,String root)");
        }
        File rootDir = new File(root);
        if (!rootDir.isDirectory()) {
            throw new IllegalArgumentException("root is not a directory");
        }
        // unknown path, unwrap if necessary
        root = StringTool.getUnwrapped(root);
        // unknown path, so conform to tec style
        if (!root.endsWith(separator)) {
            root += separator;
        }
        String unwrappedFilePath = file.getUnwrapped();
        String fromRoot = unwrappedFilePath.substring(root.length());
        String subPath = "";
        if (fromRoot.length() > 0) {
            if (fromRoot.contains(separator)) {
                subPath = fromRoot.substring(0, fromRoot.lastIndexOf(separator) + 1); // take the separator
            } else {
                subPath = "";
            }
            if (subPath.startsWith(separator)) {
                subPath = subPath.substring(1);
            }
        }
        if (!subPath.endsWith(separator) && subPath.length() > 0) {
            subPath += separator;
        }
        return subPath;
    }

    public static String getSubPath(String root,String targetPath) {
        String unwrappedTarget = StringTool.getUnwrapped(targetPath);
        System.out.println(SN + ".getSubPath: root: " + root + " unwrappedTarget: " + unwrappedTarget);
        String unwrappedRoot = new File(root).getUnwrapped();
        if (unwrappedRoot.contains("\\")) {
            if (!unwrappedRoot.endsWith("\\")) unwrappedRoot += "\\";
        } else {
            if (!unwrappedRoot.endsWith("/")) unwrappedRoot += "/";
        }
        String subPath = unwrappedTarget.substring(unwrappedRoot.length());
        subPath = subPath.substring(0,subPath.lastIndexOf(File.separator) + 1);
        return subPath; 
    }
    
    public static File[] getTecFileArray(java.io.File[] sources) {
        File[] files = new File[sources.length];
        for(int i = 0; i < sources.length;i++) files[i] = new File(sources[i]);
        return files;
    }
    
    public String getTecLastModified() {
        Long dateTime = lastModified();
//        connection.disconnect();
        ZonedDateTime modified = ZonedDateTime.ofInstant(Instant.ofEpochMilli(dateTime), ZoneId.of("GMT"));   

        DateTimeFormatter dateFormatter = 
                DateTimeFormatter.ofPattern("yyyy-MM-dd");
        DateTimeFormatter timeFormatter = 
                DateTimeFormatter.ofPattern("HH:mm:ss:S");
        DateTimeFormatter zoneFormatter = 
                DateTimeFormatter.ofPattern("Z");

        return "D" + modified.format(dateFormatter) + "T" + modified.format(timeFormatter) + "Z" + modified.format(zoneFormatter);
    }
    
    public File getTecParentFile() {
        return new File(getParentFile());
    }
    
    public static List<String> getTextFile(String absPath) {
        TextFile textFile = new TextFile(StringTool.getUnwrapped(absPath)); 
        return textFile.getLines();
    }

    public String getToExtension(boolean includeDot) {
        String unwrapped = getUnwrapped();
        if (unwrapped.indexOf(".") == -1) {
            return unwrapped;
        } else {
            if (includeDot) {
                return unwrapped.substring(0,unwrapped.lastIndexOf(".") + 1);
            } else {
                return unwrapped.substring(0,unwrapped.lastIndexOf("."));
            }
        }
    }
    
    public String getUnwrapped() {
        return StringTool.getUnwrapped(getAbsolutePath());
    }
    
    public String getUpToExtension() { return getToExtension(true); }
    
    @Override
    public long getUsableSpace() {
        if (hasARoot()) {
            return super.getUsableSpace();
        } else {
            return -1L;
        }
    }

    public boolean hasARoot() {
        File[] roots = tecListRoots();
        boolean hasARoot = false;
        for(int i = 0;i < roots.length;i++) {
            if (getUnwrapped().startsWith(roots[i].getUnwrapped())) return true;
        }
        return false;
    }
    
    public boolean hasJava() {
        File[] entries = tecListFiles();
        if (entries != null) {
            for(int i = 0; i < entries.length;i++) {
                if (entries[i].getExtension().equals("java")) {
                    return true;
                }
            }
        }
        return false;
    }
    
    public static boolean hasJava(String path) {
        Pair p = new Pair(path);
        List<File> files = p.getFiles();
        if (files != null) {
            for(int i = 0; i < files.size();i++) {
                if (files.get(i).is("java")) return true;
            }
        }
        return false;
    }
    
    public static boolean hasJava(String path, boolean recurse) {
        Pair p = new Pair(path);
        List<File> dirs = p.getDirs();
        List<File> files = p.getFiles();
        if (files != null) {
            for(int i = 0; i < files.size();i++) {
                if (files.get(i).is("java")) return true;
            }
        }
        if (recurse) {
            if (dirs != null) {
                for(int i = 0; i < dirs.size();i++) {
                    if (hasJava(dirs.get(i).getAbsolutePath(),recurse)) return true;
                }
            }
        }
        return false;
    }
    
    public boolean hasMatchingClass() {
        return new File(getToExtension(false) + ".class").exists();
    }
    
    public boolean is(String type) {
        return getExtension().equals(type);
    }

    public boolean isDir() {
        return isDirectory();
    }
    
    @Override
    public boolean isDirectory() {
        String unwrapped = getUnwrapped();
        if (unwrapped.endsWith("/") | unwrapped.endsWith("\\")) return true;
        else return super.isDirectory();
    }
    
    public String getDrive(String unwrapped) {
        if (unwrapped.contains("/")) {
            if (unwrapped.length() == 1 && unwrapped.charAt(0) == '/') return "/";
        } else {
            if (unwrapped.length() == 2) {
                char letter = unwrapped.charAt(0);
                if (unwrapped.charAt(1) == ':') {
                    if (letter >= 'a' && letter <= 'z' | letter >= 'A' && letter <= 'Z') {
                        return unwrapped + "\\";
                    } else {
                        return null;
                    }
                }
            } else if (unwrapped.length() == 3 &&
                       unwrapped.charAt(1) == ':' && 
                       unwrapped.charAt(2) == '\\'
                      )
            {
                return unwrapped;
            } else if (unwrapped.length() >= 4) {
                return unwrapped.substring(0,3);
            } else {
                throw new IllegalArgumentException(SN + ".getDrive(" + unwrapped + "): not a valid specifier");
            }
        }
        return unwrapped;
    }
    
    public boolean isDriveLetter(char ch) {
        return (ch >= 'a' && ch <= 'z' | ch >= 'A' && ch <= 'Z');
    }
    
    public boolean isRoot() {
        String unwrapped = getUnwrapped();
        char ch = unwrapped.charAt(0);
        if (unwrapped.length() > 0) {
            String drive = getDrive(unwrapped);
            if (drive.equals("/")) {
                return true;
            } else {
                boolean isDriveLetter = isDriveLetter(ch);
                if (isDriveLetter) {
                    String tag = ch + ":" + "\\";
                    File[] roots = tecListRoots();
                    for(int i = 0; i < roots.length;i++) {
                        if (roots[i].getUnwrapped().equals(tag)) return true;
                    }
                }
            }
        }
        return false;
    }
    
    public static List<String> listDirs(String path) {
        List<ca.tecreations.File> dirs = new Pair(path).getDirs();
        List<String> result = new ArrayList<>();
        for(int i = 0; i < dirs.size();i++) {
            result.add(dirs.get(i).getAbsolutePath());
        }
        return result;
    }

    public static List<String> listFiles(String path) {
        List<ca.tecreations.File> dirs = new Pair(path).getFiles();
        List<String> result = new ArrayList<>();
        for(int i = 0; i < dirs.size();i++) {
            result.add(dirs.get(i).getAbsolutePath());
        }
        return result;
    }
    
    public static void main(String[] args) {
        File f = new File("C:\\Users\\");
        System.out.println(f.getSize());
        System.exit(0);
        
        
        File f2 = new File("F:\\projects\\StusTwelve\\ca\\tecreations\\File.java");
        System.out.println("name     : " + f);
        System.out.println("exists   : " + f.exists());
        System.out.println("length   : " + f.length());
        System.out.println("classpath: " + f.getClassPath());
        System.out.println("fqcn     : " + f.getFQCN());
        System.out.println("package  : " + f.getPackage());
        System.out.println("deepest  : " + f.getDeepestDirectory());
        System.out.println("deepestAP: " + f.getDeepestDirectoryFile().getAbsolutePath());
        System.out.println("parent   : " + f.getParent());
        System.out.println("parentFAP: " + f.getParentFile().getAbsolutePath());
        
        FileInputStream fis = f.getFileInputStream();
        try {
            String fisString = (fis == null ? null :  "fis(available): " + fis.available());
            System.out.println(fisString);
        } catch (Exception e) {
            ExceptionHandler.handleIO(SN + ".main", "getting available", e, verbose);
        }
    }
    
    public boolean mkdirs() {
        super.mkdirs();
        return exists();
    }
            
    public String octalToPermission(char d) {
        String result = "";
        if (d == '0') result = "---";
        if (d =='1') result = "--x";
        if (d == '2') result = "-w-";
        if (d =='3') result = "-wx";
        if (d == '4') result = "r--";
        if (d =='5') result = "r-x";
        if (d == '6') result = "rw-";
        if (d =='7') result = "rwx";
        return result;
    }
    
    public String octalToPermission(int i) {
        String result = "";
        if (i == 0) result = "---";
        if (i== 1) result = "--x";
        if (i == 2) result = "-w-";
        if (i == 3) result = "-wx";
        if (i == 4) result = "r--";
        if (i == 5) result = "r-x";
        if (i == 6) result = "rw-";
        if (i == 7) result = "rwx";
        return result;
    }
    
    public String octalTrioToPOSIXPermission(String octal) {
        if (octal.length() != 3) {
            throw new IllegalArgumentException("Not an octal file permission representation: " + octal);
        }
        String perms = "";
        perms += octalToPermission(octal.charAt(0));
        perms += octalToPermission(octal.charAt(1));
        perms += octalToPermission(octal.charAt(2));
        return perms;
    }
    
    public static void prune(String path, boolean output) {
        File target = new File(path);
        File[] list = target.tecListFiles();
        if (list != null) {
            for (int i = 0; i < list.length; i++) {
                if (list[i].isDirectory()) {
                    prune(list[i].getAbsolutePath(), output);
                }
            }
            for (int i = 0; i < list.length; i++) {
                if (list[i].isFile()) {
                    list[i].delete(output);
                }
            }
        }
        target.delete();
    }

    public void setDOSArchive(boolean flag) {
        try {
            Files.setAttribute(this.toPath(), "dos:archive", flag);
        } catch (IOException ioe) {
            System.err.println("Unable to set archive attribute for: " + getAbsolutePath());
        }
    }

    public void setDOSFileAttributes(String attributes) {
        setDOSReadOnly(attributes.contains("R"));
        setDOSHidden(attributes.contains("H"));
        setDOSSystem(attributes.contains("S"));
        setDOSArchive(attributes.contains("A"));
    }

    public void setDOSHidden(boolean flag) {
        try {
            Files.setAttribute(this.toPath(), "dos:hidden", flag);
        } catch (IOException ioe) {
            System.err.println("Unable to set hidden attribute for: " + getAbsolutePath());
        }
    }

    public void setDOSReadOnly(boolean flag) {
        try {
            Files.setAttribute(this.toPath(), "dos:readonly", flag);
        } catch (IOException ioe) {
            System.err.println("Unable to set readonly attribute for: " + getAbsolutePath());
        }
    }

    public void setDOSSystem(boolean flag) {
        try {
            Files.setAttribute(this.toPath(), "dos:system", flag);
        } catch (IOException ioe) {
            System.err.println("Unable to set system attribute for: " + getAbsolutePath());
        }
    }

    public void setPOSIXGroup(String group) {
        GroupPrincipal groupId = null;
        try {
            groupId = toPath().getFileSystem().getUserPrincipalLookupService().lookupPrincipalByGroupName(group);
        } catch (IOException ioe) {
            System.err.println("Unable to lookup up principal (group): " + group + " : " + ioe);

        }
        try {
            Files.getFileAttributeView(toPath(), PosixFileAttributeView.class).setGroup(groupId);
        } catch (IOException ioe) {
            System.err.println("Unable to set group for: " + getAbsolutePath() + " : " + ioe);
        }
    }

    public void setPOSIXUser(String user) {
        UserPrincipal userId = null;
        try {
            userId = toPath().getFileSystem().getUserPrincipalLookupService().lookupPrincipalByName(user);
        } catch (IOException ioe) {
            System.err.println("Unable to look up principal user: " + user + " : " + ioe);
        }
        try {
            Files.setOwner(toPath(), userId);
        } catch (IOException ioe) {
            System.err.println("Unable to set owner for: " + getAbsolutePath() + " : " + ioe);
        }
    }

    public void setPOSIXPermissions(int user, int group, int other) {
        String userStr = getPermissionString(user);
        String groupStr = getPermissionString(group);
        String otherStr = getPermissionString(other);
        Set<PosixFilePermission> perms = PosixFilePermissions.fromString(userStr + groupStr + otherStr);
        //FileAttribute<Set<PosixFilePermission>> attr = PosixFilePermissions.asFileAttribute(perms);
        try {
            Files.setPosixFilePermissions(toPath(), perms);
        } catch (IOException ioe) {
            System.err.println("Unable to set POSIX file permissions: " + ioe);
        }
    }
 
    public void setPOSIXFilePermissions(int permissions) {
        int user = permissions / 100;
        int group = (permissions - (user * 100)) / 10;
        int other = (permissions - (user * 100) - (group * 10));
        String perms = "";
        perms += octalToPermission(user);
        perms += octalToPermission(group);
        perms += octalToPermission(other);
        setPOSIXFilePermissions_do(perms);
    }
    
    
    public void setPOSIXFilePermissions(String permissions) {
        String newPerms = "";
        if (permissions.length() == 3 && StringTool.isOctalDigits(permissions)) {
            int iPermissions = Integer.parseInt(permissions,10);
            int user = iPermissions / 100;
            int group = (iPermissions - (user * 100)) / 10;
            int other = (iPermissions - (user * 100) - (group * 10));
            newPerms += octalToPermission(user);
            newPerms += octalToPermission(group);
            newPerms += octalToPermission(other);
        } else {
            newPerms = permissions;
        }
        setPOSIXFilePermissions_do(newPerms);
    }

    public void setPOSIXFilePermissions_do(String permissions) {
        Set<PosixFilePermission> perms = PosixFilePermissions.fromString(permissions);
        //FileAttribute<Set<PosixFilePermission>> attr = PosixFilePermissions.asFileAttribute(perms);
        try {
            Files.setPosixFilePermissions(toPath(), perms);
        } catch (IOException ioe) {
            System.err.println("Unable to set POSIX file permissions: " + ioe);
        }
    }

    public List<File> tecGetListFiles() {
        java.io.File[] list = listFiles();
        List<File> tecList = new ArrayList<>();
        for(int i = 0;i < list.length;i++) tecList.add(new File(list[i].getAbsolutePath()));
        return tecList;
    }
    
    public static List<File> tecGetListRoots() {
        java.io.File[] roots = java.io.File.listRoots();
        List<File> tecRoots = new ArrayList<>();
        for(int i = 0; i < roots.length;i++) {
            tecRoots.add(new File(roots[i]));
        }
        return tecRoots;
    }
    
    public File[] tecListFiles() {
        java.io.File[] list = listFiles();
        File[] tecList = null;
        if (list != null) {
            tecList = new File[list.length];
            for(int i = 0; i < list.length;i++) tecList[i] = new File(list[i].getAbsolutePath());
        }
        return tecList;
    }
    
    public static List<File> tecListFiles(String path) {
        return new Pair(path).getFiles();
    }
    
    public static File[] tecListRoots() {
        java.io.File[] roots = java.io.File.listRoots();
        File[] tecRoots = new File[roots.length];
        for(int i = 0; i < roots.length;i++) {
            tecRoots[i] = new File(roots[i]);
        }
        return tecRoots;
    }
    
    public java.io.File toJavas(){ 
        return new java.io.File(getUnwrapped());
    }

    public Path toPath() {
        Path path = null;
        try {
            path = Paths.get(StringTool.getUnwrapped(getAbsolutePath()));
        } catch (InvalidPathException ipe) {
            System.err.println("--------->Invalid Path: `" + getAbsolutePath() + "`");
        }
        if (path == null) {
            System.out.println("File.toPath(): " + "Path(java.io.File): " + new java.io.File(getUnwrapped()).toPath());
        }
        return path;
    }

    public String toPathString() {
        return StringTool.getUnwrapped(toPath().toString());
    }

    public File withName(String name) {
        return new File(StringTool.getUnwrapped(getDeepestDirectory()) + name);
    }
    
}
   