package ca.tecreations.apps.capturetool;

import ca.tecreations.ImageTool;
import ca.tecreations.Platform;
import ca.tecreations.Properties;
import ca.tecreations.TColor;
import ca.tecreations.components.BoundingBox;
import ca.tecreations.components.Handle;
import ca.tecreations.components.Magnifier;
import ca.tecreations.components.SizedPanel;
import ca.tecreations.components.event.*;

import ca.tecreations.misc.KeyProcessor;
import ca.tecreations.misc.MouseInterpreter; 
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.image.BufferedImage;
 
import javax.swing.JFrame;
/** 
 *
 * @author Tim 
 */
public class CaptureToolDisplay extends SizedPanel implements BoxSizeSetListener, ComponentListener, KeyListener, MouseListener {
    CaptureTool app;
    Properties properties;
    BufferedImage image;
    BoundingBox box = new BoundingBox(TColor.LIGHT_GREY);
    BufferedImage extracted;
    Magnifier magnifier;

    public CaptureToolDisplay(CaptureTool app) {
        super(Platform.getDesktopSize());
        this.app = app;
        setLayout(null); 
        add(box);
        box.setBoxElementsColor(TColor.red);
        box.setOutline(true);
        box.setLocation(10,10);
        box.setHandlesVisible(true);
        box.addComponentListener(this);
        box.addMouseListener(this);
        addMouseListener(this);
        properties = app.getProperties();
        Integer width = properties.getInt("box.width");
        if (width == null) width = 100;
        Integer height = properties.getInt("box.height");
        if (height == null) height = 100;
        box.setSize(width,height);
        box.addBoxSizeSetListener(this);
    }
    
    public void boxSizeSet(BoxSizeSetEvent e) {
        BoundingBox box = e.getBoundingBox();
        Handle handle = e.getHandle();
    }

    public BufferedImage capture(boolean withApp) {
        if (withApp) {
            app.setVisible(true);
        } else {
            if (app.isVisible()) {
                app.setVisible(false);
                Platform.sleep(1000);
            }
        }
        captureDesktop();
        if (!withApp) app.setVisible(true);
        return image;
    }
    
    public void captureAndExtract() {
        captureDesktop();
        extractPortion();
    }

    public BufferedImage captureDesktop() {
        image = Platform.getScreenCapture();
        return image;
    }
    
    public void componentHidden(ComponentEvent e) {}
    public void componentShown(ComponentEvent e) {}
    public void componentMoved(ComponentEvent e) {
        extractPortion();
        repaint();
    }
    
    public void componentResized(ComponentEvent e) {
        extractPortion();
        repaint();
    } 
      
    public void extractPortion() {
        Point loc = box.getLocation();
        int x = loc.x;
        int y = loc.y;
        int width = box.getWidth();
        int height = box.getHeight();
        if (image != null) {
            extracted = ImageTool.getRegion(image,x,y,width,height);
            box.setImage(extracted);
        }
    }
    
    public BoundingBox getBox() { return box; }
   
    public BufferedImage getBoxContents() {
        extractPortion();
        return extracted;
    }
    
    public BufferedImage getImage() { return image; }
    
    public void keyPressed(KeyEvent e) {
        KeyProcessor kp = new KeyProcessor(e);
        if (kp.isCtrl() && kp.isUp()) {
            box.bumpUp();
            extractPortion();
        } else if (kp.isCtrl() && kp.isDown()) {
            box.bumpDown();
            extractPortion();
        } else if (kp.isCtrl() && kp.isLeft()) {
            box.bumpLeft();
            extractPortion();
        } else if (kp.isCtrl() && kp.isRight()) {
            box.bumpRight();
            extractPortion();
        } else if (kp.isUp()) {
            box.nudgeUp();
            extractPortion();
        } else if (kp.isDown()) {
            box.nudgeDown();
            extractPortion();
        } else if (kp.isLeft()) {
            box.nudgeLeft();
            extractPortion();
        } else if (kp.isRight()) {
            box.nudgeRight();
            extractPortion();
        } else if (kp.isCtrl()) {
            // ctrl by itself, so do nothing
        //} else {
            //kp.printKeyEventData();
        }
    }

    public void keyReleased(KeyEvent e) {
    }
    
    public void keyTyped(KeyEvent e) {
    }
    
    public void mouseEntered(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}
    public void mousePressed(MouseEvent e) {}
    public void mouseReleased(MouseEvent e) {}
    
    public void mouseClicked(MouseEvent e) {
        MouseInterpreter mi = new MouseInterpreter(e);
        if (e.getSource() == box) {
            if (magnifier == null) {
                magnifier = new Magnifier(box.getImage());
                magnifier.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
            } else {
                magnifier.setImage(box.getImage());
            }
            if (magnifier.shouldCenter()) magnifier.centerImage();
        } else if (e.isControlDown()) {
            if (magnifier == null) {
                magnifier = new Magnifier(image);
            } else {
                magnifier.setImage(image);
            }
        }
        magnifier.setVisible(true);
        magnifier.toFront();
    }
    
    public void paint(Graphics g) {
        System.out.println("paint()");
        g.setColor(getBackground());
        g.fillRect(0,0,getSize().width,getSize().height);
        g.drawImage(image,0,0,this);
        extractPortion();
        box.repaint();
    }

    public void paintComponent(Graphics g) {
        System.out.println("paintComponent()");
        g.setColor(getBackground());
        g.fillRect(0,0,getSize().width,getSize().height);
        g.drawImage(image,0,0,this);
        extractPortion();
        box.repaint();
    }

    public void saveDesktop() {
        String path = Platform.getFilename();
        if (!path.endsWith(".png")) path += ".png";
        Platform.saveScreenshot(path);
    }
    
    public void setBoxBounds(int x, int y, int width, int height) {
        box.setLocation(x,y);
        box.setSize(width,height);
        repaint();
    }

    public void setImage(BufferedImage img) {
        image = img;
        repaint();
    }
    
}
