package ca.tecreations.apps._gui;

import ca.tecreations.FileEntry;
import ca.tecreations.StringTool;

import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.table.AbstractTableModel;
/**
 *
 * @author tim
 */
public abstract class FileEntriesTableModel extends AbstractTableModel {
    EntriesPanel panel;
    FileEntriesTable table;
    List<FileEntry> entries = new ArrayList<>();
    
    String side = "";
    boolean debug = false;
    int lastX;
    
    public FileEntriesTableModel(EntriesPanel panel,FileEntriesTable table) {
        this.panel = panel;
        this.table = table;
        
        table.getTableHeader().addMouseListener( new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                lastX = e.getLocationOnScreen().x;
            }
        });
        table.getTableHeader().addMouseMotionListener( new MouseAdapter() {
            public void mouseDragged(MouseEvent e) {
                int x = e.getLocationOnScreen().x;
                int dx = lastX - x;
                if (dx == 0) return;
                int diff = 0;
                if (dx > 0) diff = -1; //0 - Math.abs(dx);
                if (dx < 0) diff = 1; //Math.abs(dx);
                int colIndex = table.getTableHeader().getColumnModel().getColumnIndexAtX(e.getX() - 1);
                if (colIndex == -1) colIndex = 0;
                System.out.println("col: " + colIndex + " diff: " + diff);
                panel.resizeColumn(colIndex,diff);
                lastX = x + diff;
            }
            
            public void mouseMoved(MouseEvent e) {
                lastX = e.getLocationOnScreen().x;
            }
        });
    }
    
    public void addRow(FileEntry entry,boolean update) {
        entries.add(entry);
        if (update) fireTableDataChanged();
        if (debug) System.out.println("Model.addRow: " + side + " : "+ entry.getDisplayName() + " Now: " + entries.size());
    }
    
    public void clearSelection() {
        for(int i = 0; i < entries.size();i++) {
            entries.get(i).setSelected(false);
        }
        fireTableDataChanged();
    }
    
    public void deleteSelection() {
        if (debug) System.out.println("Deleting selection:");
        for(int i = entries.size() - 1;i >= 0;i--) {
            if (entries.get(i).isSelected()) {
                if (debug) System.out.println("FileEntriesTableModel.deleteSelection: " + i + ": " + entries.get(i).getName());
                entries.remove(i);
            }
        }
        fireTableDataChanged();
    }
    
    public void deselect(int index) {
        FileEntry entry = entries.get(index);
        if (entry != null) entry.setSelected(false);
    }
    
    public void deselect(String name) {
        for(int i = 0; i < entries.size();i++) {
            if (entries.get(i).getName().equals(name)) {
                entries.get(i).setSelected(false);
            }
        }
    }
    
    public List<String> getDirs() {
        List<String> list = new ArrayList<>();
        FileEntry entry;
        for(int i = 0; i < entries.size();i++) {
            entry = entries.get(i);
            if (entry.getName().endsWith("\\") | entry.getName().endsWith("/")) {
                list.add(entry.getName());
            }
        }
        return list;
    }
    
    public List<FileEntry> getEntries() { return entries; }
    
    public FileEntry getEntry(int index) {
        return entries.get(index);
    }
    
    public FileEntry getEntryByAbsPath(String absPath) {
        for(int i = 0; i < entries.size();i++) {
            if (entries.get(i).getAbsPath().equals(absPath)) return entries.get(i);
        }
        return null;
    }
    
    public FileEntry getEntryByDisplayName(String displayName) {
        for(int i = 0; i < entries.size();i++) {
            if (entries.get(i).getDisplayName().equals(displayName)) return entries.get(i);
        }
        return null;
    }
    
    @Override
    public int getRowCount() {
        if (debug) System.out.println("Model.getRowCount: " + side + " Now: " + entries.size());
        if (entries == null) return 0;
        return entries.size();
    }
 
    public List<String> getSelection() {
        List<String> selected = new ArrayList<>();
        for(int i = 0; i < entries.size();i++) {
            if (entries.get(i).isSelected()) {
                selected.add(entries.get(i).getName());
            }
        }
        return selected;
    }
    
    public List<FileEntry> getSelectedEntries() {
        List<FileEntry> selected = new ArrayList<>();
        for(int i = 0; i < entries.size();i++) {
            if (entries.get(i).isSelected()) {
                selected.add(entries.get(i));
            }
        }
        return selected;
    }
    
    public List<String> getSelectedPaths() {
        List<String> paths = new ArrayList<>();
        List<FileEntry> entries = getSelectedEntries();
        for(int i = 0; i < entries.size();i++) {
            paths.add(StringTool.getDoubleQuoted(entries.get(i).getName()));
        }
        return paths;
    }
    
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        FileEntry entry = null;
        if (rowIndex < entries.size()) {
            if (rowIndex >= 0) entry = entries.get(rowIndex);
        }
        return entry;
    }    
    
    public int indexOf(FileEntry entry) {
        return entries.indexOf(entry);
    }
    
    public int indexOfName(String name) {
        if (debug) System.out.println("Target: '" + name + "'");
        for(int i = 0; i < entries.size();i++) {
            if (debug) System.out.println("Entry : '" + entries.get(i).getName() + "'");
            if (entries.get(i).getName().equals(name)) return i;
        }
        return -1;
    }
    
    public Boolean isSelected(int index) {
        FileEntry entry = getEntry(index);
        if (entry != null) {
            return entry.isSelected();
        }
        return null;
    }
    
    public void select(int index) {
        entries.get(index).setSelected(true);
        fireTableRowsUpdated(index,index);
    }
    
    public void setEntries(List<FileEntry> entries) {
        this.entries = entries;
        fireTableDataChanged();
    }
    
    public void setEntry(int index, FileEntry entry) {
        entries.set(index,entry);
        fireTableRowsUpdated(index,index);
    }
    
    public void setEntrySize(int index,long size) {
        if (index < entries.size()) {
            FileEntry entry = entries.get(index);
            entry.setSizeLong(size);
            entries.set(index,entry);
        } else {
            throw new IllegalArgumentException("FilEntriesTableModel.setEntrySize: index(" + index + ") >= entries.size(" + entries.size() + ")");
            // entries size mismatch -- Generated by lGetter/rGetter despite change to SizeGetterLauncher.running to false
            // threads complete information processing, so to cancel, would require the ending of the process
            // should this happen, at present, we don't care, just stop processing. We could return a boolean flag for
            // whether the operation was successful to short circuit, SizeGetterLauncher, but, we don't really care,
            // that code can attempt to continue as it would exit on the next loop iteration according to the 
            // SizeGetterLauncher run() method. So, we will stop output and leave as is.
            //System.err.println("setSize: index: " + index + "Entries.size: " + entries.size());
        }
        fireTableCellUpdated(index, 1);
    }

    public void setExecutable(int index, boolean state) {
        FileEntry entry = entries.get(index);
        entry.setExecutable(state);
        entries.set(index,entry);
        fireTableRowsUpdated(index,index);
        
    }
    
    public void setReadable(int index, boolean state) {
        FileEntry entry = entries.get(index);
        entry.setReadable(state);
        entries.set(index,entry);
        fireTableRowsUpdated(index,index);
    }
    
    public void setRowCount(int count,boolean update) {
        if (count == 0) entries = new ArrayList<>();
        else {
            for(int i = entries.size() - 1; i >= count;i--) {
                //System.out.println(i + " : " + entries.get(i).getName());
                entries.remove(i);
            }
        }
        if (debug) System.out.println("Model.setRowCount: " + count + " Now: " + entries.size());
        if (update) fireTableDataChanged();
    }
    
    public void setSelection(int index) {
        for(int i = 0; i < entries.size();i++) {
            entries.get(i).setSelected(false);
        }
        entries.get(index).setSelected(true);
        fireTableDataChanged();
    }
    
    public void setSide(String side) { this.side = side; }

    @Override
    public void setValueAt(Object o, int rowIndex, int columnIndex) {
        if (!(o instanceof FileEntry)) {
            throw new IllegalArgumentException("setValueAt: not a FileEntry");
        }
        entries.set(rowIndex,(FileEntry)o);
    }
    
    public void setWritable(int index, boolean state) {
        FileEntry entry = entries.get(index);
        entry.setWritable(state);
        entries.set(index,entry);
    }
    
    public int size() { return entries.size(); }
    
    public void toggleSelected(int index) {
        entries.get(index).setSelected(!entries.get(index).isSelected());
        this.fireTableRowsUpdated(index,index);
    }
}
