package ca.tecreations.apps.javacompiler;

import ca.tecreations.File;
import ca.tecreations.ImageTool;
import ca.tecreations.LockFile;
import ca.tecreations.Platform;
import ca.tecreations.ProjectPath;
import ca.tecreations.Properties; 
import ca.tecreations.SystemToken;
import ca.tecreations.SystemTool;
import ca.tecreations.TextFile;
import ca.tecreations.components.TFrame;
import ca.tecreations.icons.Associations;
 
import java.awt.AWTException;
import java.awt.BorderLayout;
import java.awt.CheckboxMenuItem;
import java.awt.FlowLayout;
import java.awt.MenuItem;
import java.awt.PopupMenu;
import java.awt.Rectangle;
import java.awt.SystemTray;
import java.awt.TrayIcon;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.List;

//import javax.imageio.ImageIO;
import javax.swing.*;

/**
 *
 * @author Tim
 */
public class JavaCompiler implements ActionListener, ItemListener {
    public static java.awt.SystemTray tray = null;
    public static BufferedImage img = null;
    public static TrayIcon trayIcon = null;

    public static final String MAKE_DELTAS = "make.deltas";
    
    PopupMenu popup;
    MenuItem popPaths = new MenuItem("Paths");
    MenuItem popOutput = new MenuItem("Output");
    MenuItem popStatus = new MenuItem("Status");
    CheckboxMenuItem popStartOpen = new CheckboxMenuItem("Start Open");
    MenuItem popExit = new MenuItem("Exit");
    
    JMenuBar pMenubar = new JMenuBar();
    JMenu pControls = new JMenu("Controls");
    JMenuItem pOutput = new JMenuItem("Output");
    JMenuItem pStatus = new JMenuItem("Status");
    JCheckBoxMenuItem pStartOpen = new JCheckBoxMenuItem("Start Open");
    JMenuItem pExit = new JMenuItem("Exit");
 
    JMenuBar oMenubar = new JMenuBar();
    JMenu oControls = new JMenu("Controls");
    JMenuItem oPaths = new JMenuItem("Paths");
    JMenuItem oStatus = new JMenuItem("Status");
    JCheckBoxMenuItem oStartOpen = new JCheckBoxMenuItem("Start Open");
    JMenuItem oExit = new JMenuItem("Exit");
 
    JMenuBar sMenubar = new JMenuBar();
    JMenu sControls = new JMenu("Controls");
    JMenuItem sPaths = new JMenuItem("Paths");
    JMenuItem sOutput = new JMenuItem("Output");
    JCheckBoxMenuItem sStartOpen = new JCheckBoxMenuItem("Start Open");
    JMenuItem sExit = new JMenuItem("Exit");
 
    private static boolean standalone = false;
    private static final Object lock = new Object();

    public static JavaCompiler instance;

    public static final String LOCK_FILE_PATH = ProjectPath.getTecreationsPath() + "JavaCompiler.lock";
    public static LockFile lockFile = null;

    public static final String COMPILER_PATHS = "compiler.paths";
    public static final String START_OPEN = "start.open";
    public static final String RUNNING = "running";
    public static final String SHUTDOWN = "shutdown";
  
    static Properties properties = new Properties(ProjectPath.getTecPropsPath() + "JavaCompiler.properties");
    TFrame pathsWindow;
    TFrame outputWindow;
    TFrame statusWindow;
 
    List<JavaCompilerThread> compilers = new ArrayList<>();
    JButton add = new JButton("Add");
    JButton remove = new JButton("Remove");
    DefaultListModel<String> pathsModel = new DefaultListModel<String>();
    JList<String> pathsList = new JList<String>(pathsModel);
    JTextArea outputText = new JTextArea(40, 80);
    DefaultListModel<String> statusModel = new DefaultListModel<String>();
    JList<String> statusList = new JList<String>(statusModel);
    JPanel buttons = new JPanel(false);
    JButton clear = new JButton("Clear");

    public static boolean debug = false;
    public static boolean trace = false;
     
    public JavaCompiler() {
        if (SystemTray.isSupported()) {
            tray = SystemTray.getSystemTray();
            img = ImageTool.getNewBufferedImage(
                            Platform.getImageIcon(
                                Associations.class.getResource("coffeebean_16_16.png"),"systray_icon"
                            )
            ); 
            
            // Should your use case require image scaling, one may use this code.
            //if (img != null) {
            //    img = ImageTool.getResized(img, 16, 16);
            //    ImageTool.saveImage(img,ProjectPath.getProjectPath() + ProjectPath.getSubPath("ca.tecreations.apps.systemtray") + "coffeebeen_16_16.png");
            //}
            trayIcon = new TrayIcon(img, "Java Compiler", popup);
            try {
                tray.add(trayIcon);
                popup = new PopupMenu();
                popup.add(popPaths);
                popup.add(popOutput);
                popup.add(popStatus);
                popup.addSeparator();
                popup.add(popStartOpen);
                popup.addSeparator();
                popup.add(popExit);
                trayIcon.setPopupMenu(popup);
            } catch (AWTException awte) {
                System.err.println("JavaCompiler(): unknown: " + awte);
            }
        } else {
            System.out.println("SystemTray is not supported. Using TFrame menus.");
        }

        properties.set(SHUTDOWN,false);
        properties.set(RUNNING,true);
        pathsWindow = new TFrame(properties, "Paths");
        pathsWindow.setName("Paths");
        pathsWindow.setTitle("JavaCompiler: Paths");
        setupPathsGUI();
        if (standalone) pathsWindow.setExitOnClose(true);
        outputWindow = new TFrame(properties, "Output");
        outputWindow.setName("Output");
        outputWindow.setTitle("JavaCompiler: Output");
        setupOutputGUI();
        if (standalone) outputWindow.setExitOnClose(true);
        statusWindow = new TFrame(properties, "Status");
        statusWindow.setName("Status");
        statusWindow.setTitle("JavaCompiler: Status");
        setupStatusGUI();

        setupMenus();
        // check if should "Start Open"
        Boolean startOpen = properties.getBoolean(START_OPEN);
        if (startOpen == null) {
            startOpen = false;
            properties.set(START_OPEN, startOpen);
        }
        popStartOpen.setState(startOpen);
        pStartOpen.setState(startOpen);  
        oStartOpen.setState(startOpen);
        sStartOpen.setState(startOpen); 

        
        // add PROJECTS_HOME to compile paths
        String ppProps = ProjectPath.getTecPropsPath() + "ProjectPath.properties";
        Properties ppProperties = new Properties(ppProps);
        JavaCompilerThread compiler = new JavaCompilerThread(this);
        String projsHome = ppProperties.get("PROJECTS_HOME");
        compiler.start(projsHome);
        compilers.add(compiler);
        pathsModel.addElement(projsHome);
        
        // get the user-added paths to comnpile
        List<String> targets = properties.getList(COMPILER_PATHS);
        if (targets.equals(null)) {
            properties.set(COMPILER_PATHS,"");
        }  
        for (int i = 0; i < targets.size(); i++) {
            pathsModel.addElement(targets.get(i));
            JavaCompilerThread compilerN = new JavaCompilerThread(this);
            compilerN.start(targets.get(i));
            compilers.add(compilerN);
        }  
    
        Rectangle screenSize = Platform.getDesktopSize();
        // always open status
        statusWindow.setVisible(true);
        if (startOpen) {
            pathsWindow.setVisible(true);
            outputWindow.setVisible(true);
            statusWindow.toFront(); 
        } else {
            // we always open status, so if not StartOpen, open to back
            statusWindow.toBack();
        }
        
        // set the initial state for paths and output windows
        if (properties.wasCreated()) {
            pathsWindow.setSize(640, 480); 
            outputWindow.setSize(640, 480);
            int x = screenSize.width - 640;
            int outputY = (screenSize.height - 480) / 2;
            pathsWindow.setLocation(x, 0);
            outputWindow.setLocation(x, outputY);
        } 
        // always set the status window. We let users change it but we always start at bottomRight X 640x480
        // by making the bottom and right of the status window visible, we can see if it is working and if not
        // what the last update was. This means our windows need to only extend so far on the destop, we need
        // about a 32x32 free space at the bottom right of the desktop (above or left of any bars) for this to work.
        statusWindow.setSize(640,480);
        int statusY = screenSize.height - 480 - buttons.getSize().height;
        statusWindow.setLocation(screenSize.width - 640,statusY); 
   
    } 
    
    public void actionPerformed(ActionEvent e) {
        Object itm = e.getSource();
        if (itm == popPaths || itm == oPaths || itm == sPaths) {
            pathsWindow.setVisible(true);
            pathsWindow.toFront();
        } else if (itm == popOutput || itm == pOutput || itm == sOutput) {
            outputWindow.setVisible(true);
            outputWindow.toFront();
        } else if (itm == popStatus || itm == pStatus || itm == oStatus) {
            statusWindow.setVisible(true);
            statusWindow.toFront();
        } else if (itm == popExit || itm == pExit || itm == oExit || itm == sExit) {
            exitProgram();
            System.exit(0);
        } else if (e.getSource() == add) {
            String path = Platform.requestDirectory(null, new File(ProjectPath.getDocumentsPath()), "Select a project dir...");
            if (path != null) {
                addCompilePath(path);
            }
        } else if (e.getSource() == remove) {
            int index = pathsList.getSelectedIndex();
            if (index != -1) {
                compilers.get(index).stopRunning();
                compilers.remove(index);
                pathsModel.removeElementAt(index);
                properties.set(COMPILER_PATHS, pathsModel.elements());
            }
        } else if (e.getSource() == clear) {
            statusModel.removeAllElements();
        }
    } 
 
    public void addActionListeners() {
        if (SystemTray.isSupported()) {
            popPaths.addActionListener(this);
            popOutput.addActionListener(this);
            popStatus.addActionListener(this);
            popStartOpen.addItemListener(this);
            popExit.addActionListener(this);
        }
        pOutput.addActionListener(this);
        pStatus.addActionListener(this);
        pStartOpen.addItemListener(this);
        pExit.addActionListener(this);

        oPaths.addActionListener(this);
        oStatus.addActionListener(this);
        oStartOpen.addItemListener(this);
        oExit.addActionListener(this);

        sPaths.addActionListener(this);
        sOutput.addActionListener(this);
        sStartOpen.addItemListener(this);
        sExit.addActionListener(this);
    }

    public void addCompilePath(String path) {
        JavaCompilerThread compiler = new JavaCompilerThread(this);
        compiler.start(path);
        compilers.add(compiler);
        pathsModel.addElement(path);
        properties.set(COMPILER_PATHS, pathsModel.elements());
    } 

    public void addElement(String s) {
        statusModel.addElement(s);
        statusList.ensureIndexIsVisible(statusModel.getSize() - 1);
        statusList.setSelectedIndex(statusModel.getSize() - 1);
    }

    public static void createAndShowGUI() {
        if (trace) System.out.println("createAndShowGUI()");
        SwingUtilities.invokeLater(() -> {
            instance = new JavaCompiler();
            instance.setStatusVisible(true); // always open status
        });
    }
    
    public void exitProgram() {
        for (int i = compilers.size() - 1; i >= 0; i--) {
            compilers.get(i).stopRunning();
            compilers.remove(i);
        }
        while (new File(LOCK_FILE_PATH).exists()) {
            synchronized (lock) {
                lockFile.unlock();
            } 
            new File(LOCK_FILE_PATH).delete(true);
            if (new File(LOCK_FILE_PATH).exists()) {
                System.out.println("Re-trying deletion in (3s)...");
                Platform.sleep(3000);
            }
        }
        pathsWindow.dispose();
        outputWindow.dispose();
        statusWindow.dispose();
        properties.set(RUNNING,false);
        instance = null;
        if (SystemTray.isSupported()) {
            tray.remove(trayIcon);
            System.gc();
        }
    }

    public void exitSystemAfterShutdown() {
        properties.set(SHUTDOWN,false);
        exitProgram();
        Integer pid = properties.getIntOrZero("pid");
        if (pid != null) {
            if (pid > 0) {
                String cmd = "taskkill /F /PID " + pid;
                if (Platform.isNix()) {
                    cmd = "sudo kill -9 " + pid;
                }
                new SystemTool().runAndGet(cmd);
            } else {
                System.exit(0);
            }
        }
    } 

    public static File getLockFile() {
        return new File(JavaCompiler.LOCK_FILE_PATH);
    }
    
    public TFrame getOutput() { return outputWindow; }
    
    public JTextArea getOutputText() {
        return outputText;
    }

    public boolean getOutputVisible() { 
        return outputWindow.isVisible();
    }
    
    public TFrame getPaths() { return pathsWindow; }
    
    public boolean getPathsVisible() {
        return pathsWindow.isVisible();
    }
    
    public int getPID() { 
        properties.read(false);
        return properties.getInt("pid"); 
    }
     
    public static String getRuntimePath() { return ProjectPath.getRuntimePath(JavaCompiler.class.getProtectionDomain()); }
    
    public static Properties getProperties() {
        return properties;
    }

    public TFrame getStatus() { return statusWindow; }
    
    public boolean getStatusVisible() { 
        return statusWindow.isVisible();
    }
    
    public static boolean gotLockFile() {
        if (!new File(ProjectPath.getTecreationsPath()).exists()) {
            new File(ProjectPath.getTecreationsPath()).mkdirs();
        }
        String path = LOCK_FILE_PATH;
        if (!new File(path).exists()) {
            TextFile file = new TextFile(path);
            file.write();
            file.close();
        }
        lockFile = new LockFile(LOCK_FILE_PATH);
        lockFile.lock();
        return lockFile.isLocked();
    }

    public void itemStateChanged(ItemEvent e) {
        Object itm = e.getSource();
        boolean state = false;
        if (itm == popStartOpen) state = popStartOpen.getState();
        if (itm == pStartOpen) state = pStartOpen.getState();
        if (itm == oStartOpen) state = oStartOpen.getState();
        else state = true;           //sStartOpen
        setStartOpen(state);
    }
    
    public static void launch() {
        try {
            if (trace) System.out.println(JavaCompiler.class.getSimpleName() + ".launch()");
            if (lockFile == null && gotLockFile()) {
                createAndShowGUI();
            } else {
                System.err.println("Lockfile in use.");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    } 
  
    public static void main(String[] args) {
        launch();
    }
   
   public void setErrorOutput(List<SystemToken> tokens) {
        outputText.setText("");
        for (int i = 0; i < tokens.size(); i++) {
            outputText.append(tokens.get(i).getText() + "\n");
        } 
    } 

    public void setOutputVisible(boolean state) {
        outputWindow.setVisible(state);
    }
    
    public void setPathsVisible(boolean state) {
        pathsWindow.setVisible(state);
    }

    public void setStartOpen(boolean state) {
        popStartOpen.setState(state);
        pStartOpen.setState(state);
        oStartOpen.setState(state);
        sStartOpen.setState(state);
        properties.set(START_OPEN, state);
    }

    public void setStatusVisible(boolean state) {
        statusWindow.setVisible(state);
    } 
     
//    public void setSystemErr(PrintStream err) {
//        System.setErr(err);
//    }

//    public void setSystemOut(PrintStream out) {
//        System.setOut(out);
//    }

    public void setupMenus() {
        pControls.add(pOutput);
        pControls.add(pStatus); 
        pControls.addSeparator();
        pControls.add(pStartOpen);
        pControls.addSeparator();
        pControls.add(pExit);
        pMenubar.add(pControls);

        oControls.add(oPaths);
        oControls.add(oStatus); 
        oControls.addSeparator();
        oControls.add(oStartOpen);
        oControls.addSeparator();
        oControls.add(oExit);
        oMenubar.add(oControls);

        sControls.add(sPaths);
        sControls.add(sOutput);
        sControls.addSeparator();
        sControls.add(sStartOpen);
        sControls.addSeparator();
        sControls.add(sExit);
        sMenubar.add(sControls);

        pathsWindow.setJMenuBar(pMenubar);
        outputWindow.setJMenuBar(oMenubar);
        statusWindow.setJMenuBar(sMenubar);
        addActionListeners();
    }
    
    public void setupOutputGUI() {
        outputWindow.setLayout(new BorderLayout());
        outputText.setEditable(false);
        
        JScrollPane scrollPane = new JScrollPane(outputText, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        outputWindow.add(scrollPane, BorderLayout.CENTER);
        outputWindow.validate();
        outputWindow.setDefaultCloseOperation(JFrame.HIDE_ON_CLOSE);
    }

    public void setupPathsGUI() {
        pathsWindow.setLayout(new BorderLayout());

        JPanel buttons = new JPanel(new FlowLayout(FlowLayout.LEFT));
        buttons.add(add);
        buttons.add(remove);
        add.addActionListener(this);
        remove.addActionListener(this);
        pathsWindow.add(buttons, BorderLayout.NORTH);
 
        JScrollPane scrollPane = new JScrollPane(pathsList, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        pathsWindow.add(scrollPane, BorderLayout.CENTER);
        pathsWindow.validate();
        pathsWindow.setDefaultCloseOperation(JFrame.HIDE_ON_CLOSE);
    } 
  
    public void setupStatusGUI() {
        statusWindow.setLayout(new BorderLayout());

        JScrollPane scrollPane = new JScrollPane(statusList, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        statusWindow.add(scrollPane, BorderLayout.CENTER);
        buttons.add(clear);
        clear.addActionListener(this);
        statusWindow.add(buttons, BorderLayout.SOUTH);
        statusWindow.validate();
        statusWindow.setDefaultCloseOperation(JFrame.HIDE_ON_CLOSE);
    }
}
