package ca.tecreations;
// https://stackoverflow.com/questions/14416107/int-array-to-bufferedimage 
    
import ca.tecreations.TColor;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.Transparency;
import java.awt.image.BufferedImage;
import java.awt.image.*;
import java.io.File;
import java.io.InputStream;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.imageio.*;
import javax.swing.ImageIcon;
/**
 *
 * @author timothy
 */
public class ImageTool {
    public static boolean debug = true;
    public static final int TOP = 0;
    public static final int BOTTOM = 1;
    public static final int LEFT = 2;
    public static final int RIGHT = 3;
    static BufferedImage image = null;
    
    // stack: https://stackoverflow.com/questions/14416107/int-array-to-bufferedimage 
    static int[] bitMasks = new int[] { 0xFF0000, 0xFF00, 0xFF, 0xFF000000 };
    
    TColor bg = TColor.TRANSLUCENT;
    
    public BufferedImage copy(BufferedImage image) {
        BufferedImage copy = getNewBufferedImage(image.getWidth(),image.getHeight());
        Graphics g = copy.getGraphics();
        g.drawImage(image,0,0,null);
        return copy;
    }
    
    public static ImageIcon createImageIcon(String path) {
        return new ImageIcon(getImage(path),"No alt text specified.");
    }
    
    public static ImageIcon createImageIcon(String path, String description) {
        return new ImageIcon(getImage(path),description);
    }
    
    public static ImageIcon createImageIcon(Class _class, String path, String description) {
        URL imgURL = _class.getResource(path);
        if (imgURL != null) {
            return new ImageIcon(imgURL, description);
        } else {
            System.err.println("ImageTool.createImageIcon: Couldn't find file: " + path);
            return null;
        }
    }
    
    public static BufferedImage crop(BufferedImage src, int direction, int pixelsCount) {
        int x = 0;
        int y = 0;
        if (direction == TOP) {
            return getRegion(src, 0,pixelsCount,src.getWidth(),src.getHeight());
        } else if (direction == BOTTOM) {
            return getRegion(src, 0,0,src.getWidth(),src.getHeight() - pixelsCount);
        } else if (direction == LEFT) {
            return getRegion(src, pixelsCount,0,src.getWidth() - pixelsCount,src.getHeight());
        } else if (direction == RIGHT) {
            return getRegion(src, 0,0,src.getWidth() - pixelsCount,src.getHeight());
        } else {
            return image;
        }
    }
    
    public static int[] extract(Image src, int x, int y, int width, int height) {
       if (width <= 0 || height <= 0) return null;
       int[] pixels = new int[width * height];
        java.awt.image.PixelGrabber pg = new java.awt.image.PixelGrabber(src,x,y,width,height,pixels,0,width);
        try {
            pg.grabPixels();
        } catch (InterruptedException ie) {
            System.out.println("Interrupted getting pixel data");
            return pixels;
        }
        if ((pg.getStatus() & ImageObserver.ABORT) != 0) {
            System.err.println("Image fetch aborted or errored");
            return pixels;
        }
        return pixels;
    }
    
    public static List<Pixel> getPixels(Image src) {
        int width = src.getWidth(null);
        int height = src.getHeight(null);
        if (width == 0 || height == 0) return null;
        List<Pixel> pixels = new ArrayList<>();
        int[] raw = extract(src,0,0,width,height);
        if (raw.length != width * height) {
            System.err.println("Invalid reception raw.length: " + raw.length + " width: " + width + " height: " + height);
        }
        System.err.println("ImageTool.getPixels: width: " + width + " height: " + height);
        for(int y = 0; y < height;y++) {
            for(int x = 0; x < width;x++) {
                pixels.add(new Pixel(new TColor(raw[y*width + x]),x,y));
            }
        }
        return pixels;
    }
    
    public static BufferedImage fade(BufferedImage dst, BufferedImage src1, BufferedImage src2, int index) {
        Graphics2D g2d = (Graphics2D)dst.getGraphics();
        float alpha = (index / 100) * 255;
        g2d.setComposite(AlphaComposite.SrcOver.derive(alpha));
        g2d.drawImage(src1, 0, 0, null);
        g2d.setComposite(AlphaComposite.SrcOver.derive(1f - alpha));
        g2d.drawImage(src2, 0, 0, null);
        return dst;
    }
    
    public static void fill(Image img, Color c) {
        Graphics g = img.getGraphics();
        int w = img.getWidth(null);
        int h = img.getHeight(null);
        g.setColor(c);
        g.fillRect(0,0,w,h);
    }
    
    public static Image getAsResource(String name) {
        Image image = null;
        try {
            URL url = ClassLoader.getSystemResource(name);
            try {
                image = ImageIO.read(url );
            } catch (IllegalArgumentException iae) {
                System.err.println("ImageTool.getAsResource: illegal argument: " + name);
            }
        } catch (IOException ioe) {
            System.err.println("ImageTool.getAsResource(" + name + "): " + ioe);
        }
        return image;
    }
    
    public static Color getBottomLeftColor(BufferedImage image) {
        return new Color(extract(image,0,image.getHeight() - 1,1,1)[0]);
    }
    
    public static Color getBottomRightColor(BufferedImage image) {
        return new Color(extract(image,image.getWidth() - 1,image.getHeight() - 1,1,1)[0]);
    }

    public static BufferedImage getImage(String path) {
        BufferedImage img = null;
        if (!new File(path).exists()) throw new IllegalArgumentException("ImageTool.getImage: path does not exist: " + path);
        try {
            img = ImageIO.read(new File(path));
            return img;
        } catch (IOException ioe) {
            System.out.println("IO Exception: " + ioe + " Reading: " + path);
            return null;
        }        
    }

    public static BufferedImage getImage(List<Pixel> pixels) {
        int maxX = 0;
        int maxY = 0;
        Pixel pixel;
        for(int i = 0; i < pixels.size();i++) {
            pixel = pixels.get(i);
            maxX = Math.max(maxX,pixel.getX());
            maxY = Math.max(maxY,pixel.getY());
        }
        BufferedImage img = ImageTool.getNewBufferedImage(maxX,maxY);
        Graphics ig = img.getGraphics();
        Pixel p;
        for(int i = 0;i < pixels.size();i++) {
            p = pixels.get(i);
            ig.setColor(p.getTColor());
            ig.drawLine(p.getX(),p.getY(),p.getX(),p.getY());
        }
        return img;
    }

    public static Image getImage(Class _class, String path) {
        URL imgURL = _class.getResource(path);
        if (imgURL != null) {
            return new ImageIcon(imgURL, "").getImage();
        } else {
            System.err.println("ImageTool.getImage(" + _class + "," + path + "): path does not exist.");
            return null;
        }
    }
     
    public static TColor getTColor(BufferedImage img, int x, int y) {
        int[] color = extract(img, x, y, 1, 1);
        return new TColor(color[0]);
    }
     
    public static BufferedImage getImageAsResource(URL url) {
        try {
            return ImageIO.read(url);
        } catch (IOException ioe) {
            System.err.println("Unable to retrieve resource: " + url.toString());
        }
        return null;
    }

    public static BufferedImage getImageFromJar(String runtimePath,String path) {
        try {
            InputStream inputStream = ImageTool.class.getResourceAsStream(path);
            if (inputStream == null) {
                throw new IOException("Resource not found: " + path);
            }
            return ImageIO.read(inputStream);
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    
    /**
     * https://stackoverflow.com/questions/14416107/int-array-to-bufferedimage 
     * @param pixels
     * @param width
     * @param height
     * @return sets and returns the BufferedImage
     */
    public static BufferedImage getImageFromPixels(int[] pixels,int width, int height) {
        if (width <= 0 || height <= 0) return null;
        SinglePixelPackedSampleModel sm = new SinglePixelPackedSampleModel(
        DataBuffer.TYPE_INT, width, height, bitMasks);
        DataBufferInt db = new DataBufferInt(pixels, pixels.length);
        WritableRaster wr = Raster.createWritableRaster(sm, db, new Point());   
        image = new BufferedImage(ColorModel.getRGBdefault(), wr, false, null);    
        return image;
    }

    public static Dimension getImageSize(String path) {
        return getImageSize(getImage(path));
    }
    
    public static Dimension getImageSize(BufferedImage image) {
        return new Dimension(image.getWidth(),image.getHeight());
    }
    
    public static BufferedImage getNewBufferedImage(Dimension size) {
        return getNewBufferedImage(size.width,size.height);
    }
    
    public static BufferedImage getNewBufferedImage(Image image) {
        BufferedImage bufferedImage = new BufferedImage(image.getWidth(null), image.getHeight(null), BufferedImage.TYPE_INT_ARGB);
        Graphics2D g2 = bufferedImage.createGraphics();
        g2.drawImage(image, 0, 0, null);
        g2.dispose();
        return bufferedImage;
    }

    public static BufferedImage getNewBufferedImage(ImageIcon ii) {
        Image img = ii.getImage();
        BufferedImage bi = getNewBufferedImage(img.getWidth(null),img.getHeight(null));
        Graphics g = bi.getGraphics();
        g.drawImage(img,0,0,null);
        return bi;
    }
    

    public static BufferedImage getNewBufferedImage(ImageProducer ip) {
        return getNewBufferedImage(Toolkit.getDefaultToolkit().createImage(ip));
    }
    
    public static BufferedImage getNewBufferedImage(List<Pixel> pixels) {
        int maxWidth = 0;
        int maxHeight = 0;
        Pixel p;
        for(int i = 0; i < pixels.size();i++) {
            p = pixels.get(i);
            maxWidth = Math.max(p.getX(),maxWidth);
            maxHeight = Math.max(p.getY(),maxHeight);
        }
        BufferedImage result = getNewBufferedImage(maxWidth,maxHeight);
        Graphics g = result.getGraphics();
        for(int i = 0;i < pixels.size();i++) {
            p = pixels.get(i);
            g.setColor(p.getTColor());
            g.drawLine(p.getX(),p.getY(),p.getX(),p.getY());
        }
        return result;
    }
    
    public static BufferedImage getNewBufferedImage(int width, int height) {
        if (width == 0 || height == 0) {
            System.out.println("Width: " + width + " Height: " + height);
            return null;
        }
        Color c = new Color(0f,0f,0f,1.0f);
        BufferedImage result = new BufferedImage(width,height,BufferedImage.TYPE_INT_RGB);
        Graphics g = result.getGraphics();
        g.setColor(c);
        g.fillRect(0,0,width,height);
        return result;
    }
    
    public Color getPixel(Image img, int x, int y) {
        int[] pixel = new int[1];
        java.awt.image.PixelGrabber pg = new java.awt.image.PixelGrabber(img, x, y, 1, 1, pixel, 0, 1);
        try {
            pg.grabPixels();
        } catch (InterruptedException e) {
            System.err.println("interrupted waiting for pixels!");
            return null;
        }
        if ((pg.getStatus() & ImageObserver.ABORT) != 0) {
            System.err.println("image fetch aborted or errored");
            return null;
        }
        int alpha = (pixel[0] >> 24) & 0xff;
        int red   = (pixel[0] >> 16) & 0xff;
        int green = (pixel[0] >>  8) & 0xff;
        int blue  = (pixel[0]      ) & 0xff;
        return new Color(red,green,blue,alpha);
    }

    public static BufferedImage getRegion(BufferedImage img,int x, int y, int width,int height) {
        int[] pixels = extract(img,x,y,width,height);
        if (width == 0 || height == 0) return null;
        return getImageFromPixels(pixels,width,height);
    }

    public static BufferedImage getResized(BufferedImage originalImage, int targetWidth, int targetHeight) {
        BufferedImage resizedImage = new BufferedImage(targetWidth, targetHeight, BufferedImage.TYPE_INT_RGB);
        Graphics2D graphics2D = resizedImage.createGraphics();
        if (targetWidth > 0 && targetHeight > 0) {
            graphics2D.drawImage(originalImage, 0, 0, targetWidth, targetHeight, null);
            graphics2D.dispose();
        }
        return resizedImage;
    }   
    
    public static BufferedImage getResized2(BufferedImage originalImage, int targetWidth, int targetHeight) {
        BufferedImage outputImage = null;
        if (targetWidth > 0 && targetHeight > 0) {
            Image resultingImage = originalImage.getScaledInstance(targetWidth, targetHeight, Image.SCALE_DEFAULT);
            outputImage = new BufferedImage(targetWidth, targetHeight, BufferedImage.TYPE_INT_RGB);
            outputImage.getGraphics().drawImage(resultingImage, 0, 0, null);
        }
        return outputImage;
    }
    
    public static Color getTopLeftColor(BufferedImage image) {
        return new Color(extract(image,0,0,1,1)[0]);
    }
    
    public static Color getTopRightColor(BufferedImage image) {
        return new Color(extract(image,image.getWidth() - 1,0,1,1)[0]);
    }

    
    /**
     * Gets the transparency of an image.
     *
     * @param image the image
     * @return OPAQUE, BITMASK or TRANSLUCENT (see java.awt.Transparency)
     */
    public static int getTransparency(Image image ) {
        // If buffered image, the color model is readily available
        if ( image instanceof BufferedImage ) {
            BufferedImage bimage = (BufferedImage) image;
            return bimage.getColorModel().getTransparency();
        }
        // Use a pixel grabber to retrieve the image's color model;
        // grabbing a single pixel is usually sufficient
        java.awt.image.PixelGrabber pg = new java.awt.image.PixelGrabber( image, 0, 0, 1, 1, false );
        try {
            pg.grabPixels();
        } catch ( InterruptedException e ) {
        }

        // Get the image's color model
        ColorModel cm = pg.getColorModel();

        int transparency = Transparency.OPAQUE;
        if ( cm != null ) {
            transparency = cm.getTransparency();
        }
        return transparency;
    }

    public static BufferedImage getWithoutBackground(BufferedImage image,Color bg) {
        int[] pixels = extract(image,0,0,image.getWidth(),image.getHeight());
        BufferedImage dstImage = new BufferedImage(image.getWidth(), image.getHeight(), BufferedImage.TYPE_INT_ARGB);
        Graphics2D g = dstImage.createGraphics();    
        AlphaComposite composite = AlphaComposite.getInstance(AlphaComposite.CLEAR, 0.0f);
        g.setComposite(composite);
        g.setColor(new Color(0, 0, 0, 0));
        g.fillRect(0, 0, dstImage.getWidth(),dstImage.getHeight());
        int pixel;
        int x = 0;
        int y = 0;
        for(int i = 0; i < pixels.length;i++) {
            //System.out.println(i + ": x: " + x + " y: " + y);
            pixel = pixels[i];
            //int alpha = (pixel >> 24) & 0xff;
            int red   = (pixel >> 16) & 0xff;
            int green = (pixel >>  8) & 0xff;
            int blue  = (pixel      ) & 0xff;
            if (debug) {
                System.out.println(i + ": " + x + "," + y + " : Comparing: bg: " + bg + "{" + bg.getRed() + "," + bg.getGreen() + "," + bg.getBlue() + "} target: raw[" + red + "," + green + "," + blue + "]");
            }
            if (red == bg.getRed() && green == bg.getGreen() && blue == bg.getBlue()) {
                //dstImage.setRGB(x,y,new Color(red,green,blue,255).toIntARGB());
            } else {
                dstImage.setRGB(x,y,pixel);
            }
            x++;
            if (x >= image.getWidth()) {
                x = 0;
                y++;
                if (debug) System.out.println();
            }
        }
        return dstImage;
    }
    
    // https://www.programcreek.com/java-api-examples/?api=java.awt.image.PixelGrabber
    public static final boolean hasAlpha(Image image) {
        // if BufferedImage, the color model is readily available
        if (image instanceof BufferedImage) {
            return ((BufferedImage)image).getColorModel().hasAlpha();
        }
        // Use a pixel grabber to retrieve the Image's ColorModel
        // grabbing a single pixel is usually sufficient
        java.awt.image.PixelGrabber pg = new java.awt.image.PixelGrabber(image,0,0,1,1,false);
        try {
            pg.grabPixels();
        } catch (InterruptedException ie) {
            
        }
        // get the images color model
        ColorModel cm = pg.getColorModel();
        return cm.hasAlpha();
    }
    
    public static final boolean isGreyscaleImage(java.awt.image.PixelGrabber pg) {
        return pg.getPixels() instanceof byte[];
    }
    
    public static boolean isHorizontalLineAllSame(BufferedImage image, int scanLine, Color target) {
        int[] pixels = extract(image,0,scanLine,image.getWidth(),1);
        for(int i = 0; i < pixels.length;i++) {
            if (!new Color(pixels[i]).equals(target)) {
                return false;
            }
        }
        return true;
    }
            
    public static boolean isVerticalLineAllSame(BufferedImage image, int scanLine, Color target) {
        int[] pixels = extract(image,scanLine,0,1,image.getHeight());
        
        for(int i = 0; i < pixels.length;i++) {
            if (!new Color(pixels[i]).equals(target)) {
                return false;
            }
        }
        return true;
    }
            
    public void setImage(BufferedImage image) {
        this.image = image;
    }
    
    //https://stackoverflow.com/questions/665406/how-to-make-a-color-transparent-in-a-bufferedimage-and-save-as-png
    public static BufferedImage setTransparent(BufferedImage im, final Color color) {
        ImageFilter filter = new RGBImageFilter() {

            // the color we are looking for... Alpha bits are set to opaque
            public int markerRGB = color.getRGB() | 0xFF000000;

            public final int filterRGB(int x, int y, int rgb) {
                if ((rgb | 0xFF000000) == markerRGB) {
                    // Mark the alpha bits as zero - transparent
                    return 0x00FFFFFF & rgb;
                } else {
                    // nothing to do
                    return rgb;
                }
            }
        };

        ImageProducer ip = new FilteredImageSource(im.getSource(), filter);
        return getNewBufferedImage(ip);
    }
    
    public static BufferedImage setTransparent(BufferedImage img, int x, int y) {
        img.setRGB(x,y,img.getRGB(x,y) | 0xFF000000);
        return img;
    }
    
    public static BufferedImage setTransparentBorder(BufferedImage img, int topMax, int leftMax, int bottomMax,int rightMax) {
        for(int y = 0; y < topMax;y++) {
            for(int x = 0; x < img.getWidth();x++) {
                img = setTransparent(img,x,y);
            }
        }
        for(int y = 0; y < img.getHeight();y++) {
            for(int x = 0; x < leftMax;x++) {
                img = setTransparent(img,x,y);
            }
        }
        for(int y = img.getHeight() - bottomMax; y < img.getHeight();y++) {
            for(int x = 0; x < img.getWidth();x++) {
                img = setTransparent(img,x,y);
            }
        }
        for(int y = 0; y < img.getHeight();y++) {
            for(int x = img.getWidth() - rightMax; x < img.getWidth();x++) {
                img = setTransparent(img,x,y);
            }
        }
        
        return img;
    }
    
    public static void saveImage(BufferedImage img, String path) {
        String lpath = path.toLowerCase();
        try {
            if (lpath.endsWith("png")) {
                ImageIO.write(img,"png",new File(path));
            } else if (lpath.endsWith("gif")) {
                ImageIO.write(img,"gif",new File(path));
            } else if (lpath.endsWith("jpg")) {
                ImageIO.write(img,"jpg",new File(path));
            } else if (lpath.endsWith("jpeg")) {
                ImageIO.write(img,"jpg",new File(path));
            } else {
                ImageIO.write(img,"png",new File(path + ".png"));
            }
        } catch (IOException ioe) {
            System.out.println("IO Exception: " + ioe + " Saving: " + path);
        }
    }
    
}
 
