package ca.tecreations.apps.deploy;

import ca.tecreations.File;
import ca.tecreations.Platform;
import ca.tecreations.ProjectPath;
import ca.tecreations.Properties;
import ca.tecreations.StringTool;
import ca.tecreations.components.TFrame;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.event.*;

/** 
 *
 * @author Tim de Vries
 */
public class Deploy extends JPanel implements ActionListener, ItemListener, WindowListener {
    static String className = "Deploy";
    Properties properties;
    TFrame frame;
    public static Deploy instance;
    JTabbedPane tabbedPane;

    JMenuBar menubar = new JMenuBar();
    JMenu settings = new JMenu("Settings");
    JCheckBoxMenuItem secure = new JCheckBoxMenuItem("Make Secure");
    JMenuItem open = new JMenuItem("Open...");
    JMenuItem saveAs = new JMenuItem("Save As...");
    JMenuItem delete = new JMenuItem("Delete...");
    JMenuItem recon = new JMenuItem("Reconstruct");
    
    Border padding = BorderFactory.createEmptyBorder(10, 20, 20, 20);
    JButton deployLocal = new JButton("Deploy Local");
    JButton deployRemote = new JButton("Deploy Remote");
    JavaProjectOptions project;
    OutputOptions output;
    CodeSigning signing;
    ManifestDetail manifest;
    RemoteDeployment remote;
    Distribution distribute;

    public static boolean debug = false; 
    
    public Deploy(TFrame frame) {
        super(new BorderLayout());
        this.frame = frame;
        instance = this;
        properties = frame.getProperties();
        project = new JavaProjectOptions(this);
        output = new OutputOptions(this);
        signing = new CodeSigning(this);
        manifest = new ManifestDetail(this);
        remote = new RemoteDeployment(this);
        distribute = new Distribution(this);
        setupGUI();
        if (properties.wasCreated()) {
            frame.setLocationRelativeTo(null);
            doInitialSetup();
            properties.set(Data.PROJECT_PATH, ProjectPath.getProjectPath());
        } else {
            loadProperties();
        }
        Boolean bool = properties.getBoolean(Data.PROJECT_CREATE_JAR);
        if (bool == null) {
            bool = false;
        }
        setProjectEnabled(bool);
        Boolean makeSecure = properties.getBoolean(Data.MAKE_SECURE);
        if (makeSecure == null || makeSecure == false) {
            secure.setSelected(false);
        } else {
            secure.setSelected(makeSecure);
        }
    }

    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == deployLocal) {
            remote.makeAsDir();
            commitLocal();
        } else if (e.getSource() == deployRemote) {
            remote.makeAsDir();
            commitRemote();
        } else if (e.getSource() == recon) {
            construct(properties.get(Data.PROJECT_PATH));
        }
    }

    public void commit() {
        new ActionProcess(this);
        Boolean makeSecure = properties.getBoolean(Data.MAKE_SECURE);
        if (makeSecure != null && makeSecure) {
            if (signing != null) {
                signing.clear();
            }
            if (remote != null) {
                remote.clear();
            }
        }
    }

    public void commitLocal() {
        boolean active = properties.getBoolean("remote.active");
        properties.set("remote.active", false);
        commit();
        properties.set("remote.active", active);
    }

    public void commitRemote() {
        Boolean active = properties.getBoolean("remote.active");
        if (active == null) {
            System.err.println("remote.active is null in call to commitRemote() -- probably invalid settings");
        }
        properties.set("remote.active", true);
        commit();
        properties.set("remote.active", active);
    }

    public void construct(String path) {
        if (!StringTool.getUnwrapped(path).endsWith(File.separator)) {
            path += File.separator;
        }
        properties.setDelayWrite();
        project.doInitialSetup(path);
        output.doInitialSetup();
        signing.doInitialSetup();
        manifest.doInitialSetup();
        remote.doInitialSetup(); 
        distribute.doInitialSetup();
        setProperties();
    }

    public static void createAndShowGUI(ProjectPath pp) {
        System.err.println("Deploy.createAndShowGUI: ProjectPath: " + ProjectPath.getProjectPath());

        //Create and set up the window.
        TFrame frame = new TFrame(pp.getPropertiesPath() + "Deploy.properties", "Deploy");
        frame.setExitOnClose(true);
        //Create and set up the content pane.
        Deploy newContentPane = new Deploy(frame);
        newContentPane.setOpaque(true);
        frame.setContentPane(newContentPane);
        
        //Display the window.
        //frame.pack();
        frame.setVisible(true);
    }

    public void delete() {
        String startPath = ProjectPath.getProjectPath();
        if (Platform.isWin()) {
            startPath += "ca\\tecreations\\deploy\\saved\\";
        } else {
            startPath += "ca/tecreations/deploy/saved/";
        }
        String target = Platform.requestFile(frame, new File(startPath), "Delete Settings");
        if (target != null) {
            new File(target).delete();
        }
    }

    public void doInitialSetup() {
        properties.setDelayWrite(true);
        secure.setSelected(true);
        properties.set(Data.MAKE_SECURE, true);
        project.doInitialSetup(ProjectPath.getProjectPath());
        manifest.doInitialSetup();
        setProperties();
    }

    public void filenameUpdated() {
        output.filenameUpdated();
        signing.filenameUpdated();
    }

    public boolean getCreateJarCheckBoxValue() {
        return true; //createJar.isSelected();
    }

    public CodeSigning getCodeSigning() {
        return signing;
    }

    public String getCurrentFilename() {
        if (properties.getBoolean(Data.SIGNING_ACTIVE)) {
            return signing.getSignedFilename();
        } else {
            return output.getCurrentFilename();
        }
    }

    public static boolean getDebug() { return debug; }
    
    public String getFilenamePrefix() {
        return output.getFilenamePrefix();
    }
    
    public Boolean getIncludeClasses() {
        return project.includeClasses();
    }
    
    public Boolean getIncludeUnspecified() {
        return project.includeUnspecified();
    } 
    
    public Boolean getIncludeJars() {
        return project.includeJars();
    }
    
    public Boolean getIncludeSources() {
        return project.includeSources();
    }
    
    public JavaProjectOptions getJPO() { return project; }
    
    public char[] getKeyStorePass() {
        return remote.getKeystorePassword();
    }
    
    public String getMainClass() {
        return manifest.getMainClass();
    }
    
    public ManifestDetail getManifest() {
        return manifest;
    }

    public OutputOptions getOutput() {
        return output;
    }

    public Properties getProperties() {
        return properties;
    }

    public String getPropertiesFilename() {
        return properties.getFilename();
    }

    public RemoteDeployment getRemote() {
        return remote;
    }

    public String getRemoteHost() {
        return remote.getHost();
    }
    
    public String getRemotePath() {
        return remote.getPath();
    }

    public String getRemotePort() {
        return remote.getHost();
    }
    
    public String getRemoteServerPath() {
        return remote.getPath();
    }
    public TFrame getTFrame() {
        return frame;
    }

    public void itemStateChanged(ItemEvent e) {
        if (e.getSource() == secure) {
            properties.set(Data.MAKE_SECURE, secure.isSelected());
            if (secure.isSelected()) {
                remote.secure();
            }
        }
    }

    public static Deploy launch(ProjectPath pp) {
        javax.swing.SwingUtilities.invokeLater(() -> {
            createAndShowGUI(pp);
        });
        while (instance == null) {
            Platform.sleep(125);
        }
        return instance;
    }

    public void loadProperties() {
        while (project == null) Platform.sleep(125);
        while (output == null) Platform.sleep(125);
        while (signing == null) Platform.sleep(125);
        while (manifest == null) Platform.sleep(125);
        while (remote == null) Platform.sleep(125);
        while (distribute == null) Platform.sleep(125);
        project.getProperties(properties);
        output.getProperties(properties);
        signing.getProperties(properties);
        manifest.getProperties(properties);
        remote.getProperties(properties);
        distribute.getProperties(properties);
    }

    public static void main(String[] args) {
        launch(ProjectPath.instance);
    }
    
    public void setAppendDate(boolean state) {
        output.setAppendDate(state);
    }
    
    public void setAppendTime(boolean state) {
        output.setAppendTime(state);
    }
    
    public void setEnabled(boolean state) {
        project.setEnabled(state);
        output.setEnabled(state);
        signing.setEnabled(state);
        manifest.setEnabled(state);
        remote.setEnabled(state);
        distribute.setEnabled(state);
    }
    
    public void setFilenamePrefix(String prefix) {
        output.setFilenamePrefix(prefix);
    }

    public void setIncludeClasses(boolean state) {
        project.setIncludeClasses(state);
    }
    
    public void setIncludeUnspecified(boolean state) {
        project.setIncludeUnspecified(state);
    }
    
    public void setIncludeJars(boolean state) {
        project.setIncludeJars(state);
    }
    
    public void setIncludeJava(boolean state) {
        project.setIncludeSources(state);
    }
    
    public void setIncludeSources(boolean state) {
        project.setIncludeSources(state);
    }
    
    public void setMainClass(String fqcn) {
        manifest.setMainClass(fqcn);
    }
    
    public void setMakeSecure(boolean state) {
        secure.setSelected(state);
        properties.set(Data.MAKE_SECURE,state);
    }
    
    public void setProjectEnabled(boolean state) {
        if (state) {
            signing.enableComponent();
            manifest.enableComponent();
        } else {
            signing.disableComponent();
            manifest.disableComponent();
        }
    }

    public void setProperties() {
        properties.setDelayWrite(true);
        properties.set(Data.MAKE_SECURE,secure.isSelected());
        project.setProperties();
        output.setProperties();
        signing.setProperties();
        manifest.setProperties();
        remote.setProperties();
        distribute.setProperties();
        properties.write();
    }
    
    public void setPropertiesPath(String path) {
        frame.setTitle(path);
        properties = new Properties(path);
        loadProperties();
    }

    public void setRemoteActive(boolean state) {
        remote.setActive(state);
    }
    
    public void setRemoteHost(String host) {
        remote.setHost(host);
    }
    
    public void setRemoteKeyStore(String path) {
        remote.setKeyStore(path);
    }
        
    public void setRemoteKeyStorePass(String keystore_pass) {
        remote.setKeyStorePass(keystore_pass);
    }
    
    public void setRemoteKeyStorePass(char[] keystore_pass) {
        remote.setKeyStorePass(keystore_pass);
    }
    
    public void setRemotePath(String path) {
        remote.setPath(path);
    }
    
    public void setRemotePort(String port) {
        remote.setPort(port);
    }

    public void setRemoteServerPath(String path) {
        setRemotePath(path);
    }
    
    public void setRemoteTrustStore(String path) {
        remote.setTrustStore(path);
    }
    
    public void setupGUI() {
        frame.setJMenuBar(menubar);
        frame.addWindowListener(this);
        menubar.add(settings);
        settings.add(secure);
        settings.addSeparator();
        settings.addSeparator();
        settings.add(open);
        settings.add(saveAs);
        settings.addSeparator();
        settings.add(delete);
        settings.addSeparator();
        settings.add(recon);
        secure.addItemListener(this);
        open.addActionListener(this);
        saveAs.addActionListener(this);
        delete.addActionListener(this);
        recon.addActionListener(this);

        project.setBorder(padding);
        JPanel outputHolder = new JPanel(new BorderLayout());
        outputHolder.add(output, BorderLayout.NORTH);
        outputHolder.add(signing, BorderLayout.SOUTH);
        outputHolder.setBorder(padding);
        manifest.setBorder(padding);
        remote.setBorder(padding);

        tabbedPane = new JTabbedPane();
        tabbedPane.addTab("Project Detail", null,
                project,
                "The targets for this deployment.");
        tabbedPane.addTab("Output Options", null,
                outputHolder,
                "Output related options.");
        tabbedPane.addTab("Manifest Detail", null,
                manifest,
                "The manifest attributes and main class for the jar file.");
        tabbedPane.addTab("Remote Host", null,
                remote,
                "The remote host on which to place this deployment.");
        tabbedPane.addTab("Distribution", null,
                distribute,
                "Distribution");
        add(tabbedPane, BorderLayout.CENTER);
        JPanel holder = new JPanel(new GridLayout(1, 2));
        holder.add(deployLocal);
        holder.add(deployRemote);
        add(holder, BorderLayout.SOUTH);
        tabbedPane.addChangeListener(new ChangeListener() {
            public void stateChanged(ChangeEvent e) {
                if (tabbedPane.getSelectedComponent().equals(manifest)) {
                    manifest.retrieveMainClass();
                } else if (tabbedPane.getSelectedComponent().equals(remote)) {
                    remote.setHostnameBackground();
                }
            }
        });
        deployLocal.addActionListener(this);
        deployRemote.addActionListener(this);
        frame.setSize(640, 600);
        frame.setResizable(false);
        frame.setExitOnClose(true);
    }

    public void windowClosed(WindowEvent e) {
        System.exit(0);
    }

    public void windowClosing(WindowEvent e) {
        System.exit(0);
    }

    public void windowOpened(WindowEvent e) {
    }

    public void windowDeiconified(WindowEvent e) {
    }

    public void windowIconified(WindowEvent e) {
    }

    public void windowDeactivated(WindowEvent e) {
    }

    public void windowActivated(WindowEvent e) {
    }

}
