package ca.tecreations.apps.deploy;

import ca.tecreations.File;
import ca.tecreations.JarWriter;
import ca.tecreations.Pair;
import ca.tecreations.Platform;
import ca.tecreations.ProjectPath;
import ca.tecreations.Properties;
import ca.tecreations.StringTool;
import ca.tecreations.SystemToken;
import ca.tecreations.SystemTool;
import ca.tecreations.TypeToType;
import ca.tecreations.components.ProgressDialog;
import ca.tecreations.components.event.ProgressListener;
import ca.tecreations.net.NoTLSConnectionException;
import ca.tecreations.net.PKIData;
import ca.tecreations.net.Client;
 
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.jar.JarOutputStream;

import javax.swing.*;

/**
 *
 * @author Tim de Vries
 */
public class Process implements ProgressListener {
    public static final String SN = Process.class.getSimpleName();
    boolean debug = true;
    boolean verbose = true;
    Deploy deployment;
    Properties properties;
    ProgressDialog progress;
    boolean attempt = true;
    String failure = "";
    int progressDots = 0;
    Client client;
    
    public Process(Deploy deployment) {
        this.deployment = deployment;
        progress = new ProgressDialog(deployment.frame);
        progress.setLocationRelativeTo(deployment.frame);
        properties = deployment.getProperties();
        properties.read(false);
 
        System.err.println("Process():");
        
        // do a code sanity check. Are we closing all our jar files correctly?
        int found = deleteZipFSTMPFiles(ProjectPath.getJarsPath());
        if (found > 0) {
            System.out.println("Removed: " + found + " : " + ProjectPath.PROJECT_DIR + "/jars/zipfstmp?????.tmp files.");
        }
        found = deleteZipFSTMPFiles(ProjectPath.getDownloadsPath());
        if (found > 0) {
            System.out.println("Removed: downloads: " + found + " : zipfstmp?????.tmp files.");
        }
        String outputPathVar = properties.get(Data.PROJECT_NAME);
        String distPath = StringTool.getUnwrapped(ProjectPath.getDistPath());
        //System.err.println("Dist. Path: " + distPath);
        if (!new File(distPath).exists()) {
            new File(distPath).mkdirs();
        }
        while (deployment.getOutput() == null) {
            System.err.println("Waiting To Execute: Process(): getOutput() == null");
        }
        String normalJarPath = distPath + deployment.getOutput().getCurrentFilename();
        String signedJarPath = "";
        // delay just slightly so the user can track what we are doing
        Platform.sleep(250);
        buildDeployment(normalJarPath);
        Platform.sleep(250);
        boolean signWhenDone = properties.getBoolean(Data.SIGNING_ACTIVE);
        if (signWhenDone) {
            signedJarPath = ProjectPath.getDistPath() + deployment.getCodeSigning().getSignedFilename();
            // see comments on delay to follow
            Platform.sleep(250);
            signJar(normalJarPath, signedJarPath);
            Platform.sleep(250);
        }
        if (!failure.equals("")) {
            System.err.println("Failure: '" + failure + "'");
        } else {
            System.out.print("Distributing: ");
            if (signedJarPath.equals("")) {
                System.out.println(" normal: " + normalJarPath);
                distribute(normalJarPath);
            } else {
                System.out.println(" signed: " + signedJarPath);
                distribute(signedJarPath);
            }
            Boolean active = properties.getBoolean(Data.REMOTE_ACTIVE);
            if (active != null && active && attempt) {
                deployToRemoteHost();
            } else {
                System.err.println("Not deploying: active: " + active + " attempt: " + attempt);
                        
            }
        }
        if (signedJarPath.equals("")) {
            System.out.println("Done: " + normalJarPath);
        } else {
            System.out.println("Done: " + signedJarPath);
        }
    }
    
    public void addItem(String s) {
        progress.setItem(s);
    }

    public void addToProject(JarWriter jarWriter, String rootPath, List<File> list) {
        progress.setTitle("Packaging: 0 %");
        progress.setPercent(0);
        progress.setVisible(true);
        for (int i = 0; i < list.size(); i++) {
            jarWriter.addFile(rootPath, list.get(i));
            int percent = (int)((double)i / (double)list.size() * (double)100);
            progress.setTitle("Packaging: " + percent + " %");
            progress.setPercent(percent);
        }
        progress.setTitle("Packaging: 100 %");
        progress.setPercent(100);
    }

    public static void backup(ProgressDialog progress, String filename, List<File> list) {
        System.out.println("Backing up...");
        JarWriter jarWriter;
        JarOutputStream target = null;
        jarWriter = new JarWriter(filename);
        String parentPath;
        if (target != null) {
            for (int i = 0; i < list.size(); i++) {
                parentPath = list.get(i).getParent();
                //System.out.println("ParentPath: " + parentPath);
                jarWriter.addFile(parentPath, list.get(i));
                int percent = (int)((double)i / (double)list.size() * (double)100);
                progress.setTitle("Packaging: " + percent + " %");
                progress.setPercent(percent);
            }
        }
        if (progress != null) {
            progress.setTitle("Packaging: 100 %");
            progress.setPercent(100);
        }
        try {
            if (target != null) {
                target.close();
            }
        } catch (IOException ioe) {
            System.err.println("Unable to close jar.");
        }
    }

    public void buildDeployment(String outFilename) {
        outFilename = StringTool.getUnwrapped(outFilename);
        boolean createJar = properties.getBoolean(Data.PROJECT_CREATE_JAR);
        String csvPaths = properties.get(Data.PROJECT_PATHS_INCLUDES);
        System.out.println("BuildDeployment: " + outFilename);
        
        new File(outFilename).getDeepestDirectoryFile().mkdirs();
        
        List<String> includeList = properties.getList(Data.PROJECT_PATHS_INCLUDES);
        List<String> excludeList = properties.getList(Data.PROJECT_PATHS_EXCLUDES);
        includeList = preprocess(includeList);
        excludeList = preprocess(excludeList);
        
        
        List<File> list = new GetPackingList(new File(ProjectPath.getProjectPath(properties.get(Data.PROJECT_NAME))),
                                             includeList,
                                             excludeList,
                                             deployment.getIncludeSources(),
                                             deployment.getIncludeClasses(),
                                             deployment.getIncludeUnspecified(),
                                             deployment.getIncludeJars()
                                            ).getPackingList();
        // if debugging, print trace
        System.out.println("Building target: " + outFilename);
        if (createJar) {
            buildJar(outFilename, list);
        } else {
            if (csvPaths != null && !csvPaths.equals("")) {
                System.out.println("backup: " + outFilename);
                backup(progress, outFilename, list);
            }
        }
        if (progress != null) {
            progress.dispose();
        }
    }

    public void buildJar(String filename, List<File> list) {
        System.out.println("BuildJar: " + filename);
        JarWriter jarWriter = new JarWriter(filename);
        if (jarWriter != null) {
            String mainClassVar = properties.get("main.class");
            if (debug) {
                System.out.println("Using main class: " + mainClassVar);
            }
            if (!mainClassVar.equals("None")) {
                jarWriter.setMainClass(mainClassVar);
            }
            List<String> keys = new ArrayList<>();
            List<String> values = new ArrayList<>();
            getManifest(keys, values);
            for (int i = 0; i < keys.size(); i++) {
                jarWriter.addToManifest(keys.get(i), values.get(i));
            }
            String prjName = properties.get(Data.PROJECT_NAME);
            System.out.println("Adding: from: " + prjName);
            addToProject(jarWriter, ProjectPath.getProjectPath(prjName), list);
            jarWriter.close();
        }
    }

    /** clean up zipfstmp jar files -- if there are any, probably we have a problem with our code not
     * properly closing resources. I've checked the Jar/JarReader/JarWriter. I think they look good.
     * so this will only do the project path directory, it now recurses to hit the full list
     */   
    public static int deleteZipFSTMPFiles(String absPath) {
        Pair pair = new Pair(absPath);
        List<File> dirs = pair.getDirs();
        List<File> files = pair.getFiles();
        int found = 0;
        File f;
        for(int i = 0; i < files.size();i++) {
            f = files.get(i);
            if (f.getFilenameOnly().toLowerCase().startsWith("zipfstmp") &&
                    // this skips any middle section, ie: zipfstmp109$13#477.tmp
                f.getExtension().toLowerCase().equals("tmp")) {
                found++;
                f.delete();
            }
        }
        if (dirs != null) {
            for(int i = 0; i < dirs.size();i++) {
                found += deleteZipFSTMPFiles(dirs.get(i).getAbsolutePath());
            }
        }
        return found;
    }
 
    public void deployToRemoteHost() {
        //TFrame frame = deployment.getTFrame();
        String clientKeystore = properties.get(PKIData.REMOTE_KEYSTORE);
        String clientTruststore = properties.get(PKIData.REMOTE_TRUSTSTORE);
        
        String lower = clientKeystore.toLowerCase();
        if (lower.startsWith("$user_home$")) {
            clientKeystore = ProjectPath.getUserHome() + clientKeystore.substring("$USER_HOME$".length() + 1);
        } else if (lower.startsWith("$documents_path$")) {
            clientKeystore = ProjectPath.getDocumentsPath() + clientKeystore.substring("$DOCUMENTS_PATH$".length() + 1);
        } 
        lower = clientTruststore.toLowerCase();
        if (lower.startsWith("$user_home$")) {
            clientTruststore = ProjectPath.getUserHome() + clientTruststore.substring("$USER_HOME$".length() + 1);
        } else if (lower.startsWith("$documents_path$")) {
            clientTruststore = ProjectPath.getDocumentsPath() + clientTruststore.substring("$DOCUMENTS_PATH$".length() + 1);
        }  
        
        if (clientKeystore != null && !new File(clientKeystore).exists()) {
            System.err.println(SN + ".deployToRemoteHost: Client keystore does not exist: " + clientKeystore);
            failure = SN + ".deployToRemoteHost:  No Client KeyStore";
        } 
        if (clientTruststore != null && !new File(clientTruststore).exists()) {
            System.err.println(SN + ".deployToRemoteHost: Client truststore does not exist: " + clientTruststore);
            failure = SN + ".deployToRemoteHost:  No Client TrustStore";
        }
        char[] ksPass = deployment.getRemote().getKeystorePassword();
        if (ksPass.length == 0) {
            System.err.println(SN + ".deployToRemoteHost:  Empty keystore_pass.");
            failure = SN + ".deployToRemoteHost:  Missing keystore Password";
        }
        try {
            client = new Client(properties,true,ksPass);
            //client.setDebug(true);
            //client.setVerbose(true);
            //client.addProgressListener(progress);
        } catch (NoTLSConnectionException e) {
            System.out.println(SN + ".deployToRemoteHost:  " + e.getPropertiesFilename());
        }
        if (client == null) attempt = false;
        if (debug && verbose) {
            System.out.println(SN + ".deployToRemoteHost:  Using Properties: " + properties.getFilename());
        }
        if (attempt) {
            String distPath = ProjectPath.getDistPath();
            String src;
            String dst;
            if (properties.getBoolean(Data.REMOTE_ACTIVE)) {
                Platform.sleep(250); // delay just slightly, so the user knows what we are doing
                String name;
                if (properties.getBoolean(Data.SIGNING_ACTIVE)) {
                    name = deployment.getCodeSigning().getSignedFilename();
                } else {
                    name = deployment.getOutput().getCurrentFilename();
                }
                src = distPath + name;
                dst = properties.get(Data.REMOTE_PATH) + name;
                src = StringTool.getDoubleQuoted(src);
                dst = StringTool.getDoubleQuoted(dst);
                System.out.println(SN + ".deployToRemoteHost:  Uploading: " + dst);
                //client.startJob(new File(src).length());
                progress.setPercent(0);
                progress.setVisible(true);
                progress.setTitle("Copying to remote...");
                client.processPUT(src,dst,new File(src).length());
                Platform.sleep(250); // delay, just slightly so the user can see the progress
                progress.setVisible(false);
                //client.removeProgressListener(this);
                System.out.println(SN + ".deployToRemoteHost:  Done.");
            } else {
                System.err.println(SN + ".deployToRemoteHost:  remote.active: false");
            }
        } else {
            System.err.println(SN + ".deployToRemoteHost:  attempt: " + attempt);
        }
        //client.removeProgressListener(progress);
    } 

    public void distribute(String target) {
        if (progress != null) progress.setTitle("Copying...");
        //System.err.println("Process.distribute: target: " + target);
        File dist = new File(target);
        Boolean copyToDownloads = properties.getBoolean("copy.to.downloads");
        if (copyToDownloads != null && copyToDownloads) {
            System.out.println("ProjectPath.getDownloadsPath(): " + ProjectPath.getDownloadsPath());
            String pathToDelete = ProjectPath.getDownloadsPath() + dist.getName();
            new File(pathToDelete).delete(true);
            dist.copyToDir(ProjectPath.getDownloadsPath(),progress);
        }
        List<String> destinations = properties.getList("copy.to.paths");
        for (int i = 0; i < destinations.size(); i++) {
            String pathToDelete = destinations.get(i) + dist.getName();
            new File(pathToDelete).delete(true);
            try {
                dist.copyToDir(destinations.get(i),progress);
            } catch (IllegalArgumentException iae) {
                System.err.println("Process.distribute: " + iae);
                iae.printStackTrace();
                System.out.println("Process.distribute: dst: " + destinations.get(i));
            }
        }
    } 

    public void getManifest(List<String> keys, List<String> values) {
        String compact = properties.get(Data.MANIFEST_ATTRIBUTES);
        if (compact != null && compact.length() > 0) {
            StringBuffer buf = new StringBuffer();
            List<String> parts = new ArrayList<>();
            for (int i = 0; i < compact.length(); i++) {
                if (compact.charAt(i) == ',') {
                    parts.add(buf.toString());
                    buf = new StringBuffer();
                } else {
                    buf.append(compact.charAt(i));
                }
            }
            parts.add(buf.toString());
            for (int i = 0; i < parts.size(); i++) {
                int index = parts.get(i).indexOf(":");
                keys.add(parts.get(i).substring(0, index));
                values.add(parts.get(i).substring(index + 1, parts.get(i).length()));
            }
        }
    }
    
    public List<File> getPackingList() {
        boolean sources = properties.getBoolean(Data.PROJECT_INCLUDE_SOURCES);
        boolean classes = properties.getBoolean(Data.PROJECT_INCLUDE_CLASSES);
        boolean unspecified = properties.getBoolean(Data.PROJECT_INCLUDE_UNSPECIFIED);
        boolean jars = properties.getBoolean(Data.PROJECT_INCLUDE_JARS);
        List<String> includes = properties.getList(Data.PROJECT_PATHS_INCLUDES);
        List<String> excludes = properties.getList(Data.PROJECT_PATHS_EXCLUDES);
        List<File> packingList = new GetPackingList(new File(deployment.getJPO().getProjectPath()),includes,excludes,sources,classes,unspecified,jars).getPackingList();
        return packingList;
    }

    public List<String> getPackingList(String path, List<String> includes, List<String> excludes, boolean sources, boolean classes, boolean jars, boolean images) {
        Pair pair = new Pair(path);
        List<File> files = pair.getFiles();
        List<File> dirs = pair.getDirs();
        List<String> packingList = new ArrayList<>();
        String extension;
        //System.err.println("Excludes: " + excludes);
        if (files != null) {
            for (File file : files) {
                String unwrapped = file.getUnwrapped();
                boolean exclude = false;
                for(int i = 0; i < excludes.size();i++) {
                    exclude = false;
                    String exclusion = excludes.get(i);
                    if (exclusion.endsWith("*")) {
                        String start = exclusion.substring(0,exclusion.length() - 1);
                        exclude = unwrapped.startsWith(start);
                    }
                    if (exclude) break; // break from here and exclude
                }
                if (!exclude) {
                    if (!inList(excludes, unwrapped)) {
                        extension = file.getExtension().toLowerCase();
                        if (sources && extension.equals("java")) {
                            packingList.add(unwrapped);
                        } else if (classes && extension.equals("class")) {
                            packingList.add(unwrapped);
                        } else if (!jars && extension.equals("jar")) {
                        } else if (jars && extension.equals("jar")) {
                            packingList.add(unwrapped);
                        } else if (images && isImage(extension)) {
                            packingList.add(unwrapped);
                        } else if (!inList(excludes,unwrapped)) {
                            packingList.add(unwrapped);
                        }
                    }
                }
            }
        }
        if (dirs != null) {
            for (File dir : dirs) {
                String unwrapped = dir.getUnwrapped();
                if (!inList(excludes, unwrapped)) {
                    List<String> children = getPackingList(unwrapped, includes, excludes, sources, classes, jars,images);
                    for (int j = 0; j < children.size(); j++) {
                        packingList.add(children.get(j));
                    }
                }
            }
        }
        return packingList;
    }
    
    

    public ProgressDialog getProgressBarDialog() {
        return progress;
    }

    public static boolean inList(List<String> list, String target) {
        String current;
        for (int i = 0; i < list.size(); i++) {
            current = list.get(i);
            if (current.startsWith("*.")) {
                String extension = current.substring(2);
                if (target.endsWith(extension)) return true;
            } else if (list.get(i).endsWith("*")) {
                String start = current.substring(0,current.length() - 1);
                if (target.startsWith(start)) return true;
            } else if (list.get(i).startsWith("*") && list.get(i).endsWith("*")) {
                if (target.contains(current.substring(1,current.length() - 1))) return true;
            } else if (list.get(i).equalsIgnoreCase(target)) {
                return true;
            }
        }
        return false;
    }
    
    public static boolean isImage(String extension) {
        String lower = extension.toLowerCase();
        if (lower.startsWith(".")) {
            lower = lower.substring(1);
        }
        return lower.equals("png") || 
               lower.equals("jpg") || 
               lower.equals("gif");
    }
    
    public static void main(String[] args) {
        System.out.println("test: " + "*test*".substring(1,"*test*".length() - 1));
    }
    
    public List<String> preprocess(List<String> paths) {
        List<String> result = new ArrayList<>();
        String path;
        for(int i = 0; i < paths.size();i++) {
            path = ProjectPath.getActual(paths.get(i));
            result.add(path);
        }
        return result;
    }
    
    public void setItem(String item) {
        progress.setTitle(item);
    }
    
    public void setJob(String job) {
    }
    
    public void setTitle(String title) {
        progress.setTitle(title);
    }
    
    public void signJar(String src, String dst) {
        System.out.println("Signing jar...");
        JOptionPane msg = new JOptionPane("Signing Jar... Please wait.", JOptionPane.INFORMATION_MESSAGE);
        JDialog dialog = msg.createDialog(deployment.getTFrame(), "Signing...");
        dialog.setModal(false);
        dialog.setVisible(true);
        String signingPrefix = properties.get(Data.SIGNING_PREFIX);
        String tsa = properties.get(Data.SIGNING_TSA);
        String keystore = properties.get(Data.SIGNING_KEYSTORE);

        char[] storepass = deployment.getCodeSigning().getPassword();
        if (storepass == null || storepass.length == 0) {
            System.err.println("Signing password is empty.");
            dialog.setVisible(false);
            return;
        } else if (!new File(keystore).exists()) {
            System.err.println("Signing keystore does not exist: " + keystore);
            dialog.setVisible(false);
            return;
        } else {
            String alias = properties.get(Data.SIGNING_ALIAS);
            char[] keyPass = deployment.getCodeSigning().getKeyPassword();
            if (keyPass == null || keyPass.length == 0) {
                System.err.println("Signing key password is empty.");
                dialog.setVisible(false);
                return;
            } else {
                sign(src, dst, tsa, keystore, storepass, alias, keyPass);
                for (int i = 0; i < storepass.length; i++) {
                    storepass[i] = '\0';
                }
                for (int i = 0; i < keyPass.length; i++) {
                    keyPass[i] = '\0';
                }
                dialog.setVisible(false);
            }
        }
    }

    public void toFront() {
        progress.toFront();
    }
    
    public void sign(String src, String dst, String tsa, String keystore, char[] storepass, String alias, char[] keyPass) {
        if (src.contains(" ")) {
            src = "\"" + src + "\"";
        }
        if (dst.contains(" ")) {
            dst = "\"" + dst + "\"";
        }
        if (keystore.contains(" ")) {
            keystore = "\"" + keystore + "\"";
        }
        System.out.println("SRC: " + src);
        System.out.println("DST: " + dst);
        String[] command;
        if (src.equals(dst)) {
            command = new String[11];
        } else {
            command = new String[13];
        }
        command[0] = "jarsigner";
        command[1] = "-tsa";
        command[2] = tsa;
        command[3] = "-keystore";
        command[4] = keystore;
        command[5] = "-storepass";
        StringBuffer buf = new StringBuffer();
        for (int i = 0; i < storepass.length; i++) {
            buf.append(storepass[i]);
        }
        command[6] = buf.toString();
        buf = new StringBuffer();
        for (int i = 0; i < keyPass.length; i++) {
            buf.append(keyPass[i]);
        }
        command[7] = "-keypass";
        command[8] = buf.toString();
        buf = null;
        System.gc();
        if (src.equals(dst)) {
            command[9] = src;
            command[10] = alias;
        } else {
            command[9] = "-signedjar";
            command[10] = dst;
            command[11] = src;
            command[12] = alias;
        }
        SystemTool tool = new SystemTool();
        List<SystemToken> output = tool.runForOutput(TypeToType.toString(command), debug);
        for (int i = 0; i < output.size(); i++) {
            failure += output.get(i).getText();
        }
        if (failure.startsWith("jar signed")) {
            failure = "";
            attempt = true;
        }
    }

    public void update(String msg, int job, int item) {
        progress.setTitle(msg);
        updateJob(job);
        updateItem(item);
    }
    
    public void update(int job, int item) {
        updateJob(job);
        updateItem(item);
    }

    public void updateItem(int item) {
        progress.setTitle("Copying... " + item + "%");
        progress.setPercent(item);
    }

    public void updateJob(int job) {
    }
}
