package ca.tecreations;

import java.io.IOException;
import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarInputStream;

/**
 *
 * @author Tim
 */
public class Jar {
    public static final String SN = Jar.class.getSimpleName(); 
    File file;
    JarFile jar;
    Long unpackedSize; 
    List<JarEntry> entries;
    List<String> crc_s = new ArrayList<>();
    List<String> classes = new ArrayList<>();
    
    public Jar(File f) {
        this(f.getUnwrapped());
    }
    
    public Jar(String absPath) {
        file = new File(absPath);
        if (!file.getExtension().equals("jar")) {
            throw new IllegalArgumentException("Jar(): must have a .jar extension");
        }
        try {
            jar = new JarFile(file.getUnwrapped());
        } catch (IOException ioe) {
            System.err.println("Jar(): Unable to open jarfile: " + absPath + " ioe: " + ioe);
        }
        FileInputStream fis = null;
        JarInputStream jarInput = null;
        try {
            fis = new FileInputStream(file.getUnwrapped());
            jarInput = new JarInputStream(fis);
            JarEntry jarEntry;
            while ((jarEntry = jarInput.getNextJarEntry()) != null) {
                unpackedSize += jarEntry.getSize();
                entries.add(jarEntry);
                String newName = jarEntry.getName().replace('/','.');
                if (newName.toLowerCase().endsWith(".class")) {
                    classes.add(newName.substring(0,newName.lastIndexOf(".")));
                }
            }
        } catch (IOException ioe) {
            System.err.println(SN + ".getJarEntries: IOE: " + ioe);
        } finally {
            try {
                if (jarInput != null) jarInput.close();
                if (fis != null) fis.close();
            } catch (IOException ioe) {
                System.err.println(SN + ".___getJarEntries: Unable to close: fis: " + fis + " jarInput: " + jarInput);
            }
        }
        classes = getSortedByPackage(classes);
    }
    
    public String getAbsolutePath() { return file.getAbsolutePath(); }
    
    public List<JarEntry> getClassEntries() {
        List<JarEntry> entries = new ArrayList<>();
        FileInputStream fis = null;
        JarInputStream jarInput = null;
        try {
            fis = new FileInputStream(file.getUnwrapped());
            jarInput = new JarInputStream(fis);
            JarEntry jarEntry;
            while ((jarEntry = jarInput.getNextJarEntry()) != null) {
                if (jarEntry.getName().toLowerCase().endsWith(".class")) entries.add(jarEntry);
            }
        } catch (IOException ioe) {
            System.err.println("Jar.getClassEntries: IOE: " + ioe);
        } finally {
            try {
                if (jarInput != null) jarInput.close();
                if (fis != null) fis.close();
            } catch (IOException ioe) {
                System.err.println("Unable to close: fis: " + fis + " jarInput: " + jarInput);
            }
        }
        return entries;
    }

    /**
     * Get the list of fqcn class names.
     * @return 
     */
    
    public List<String> getClassNames() { return classes; }
    
    public static int getDotCount(String name) {
        int dotCount = 0;
        for(int i = 0; i < name.length();i++) {
            if (name.charAt(i) == '.') dotCount++;
        }
        return dotCount;
    }
    
    public List<JarEntry> getEntries() { return entries; }
    
    public File getFile() { return file; }
    
    public String getFileNameFromJarEntry(JarEntry entry) {
        String name = entry.getName();
        name = name.substring(name.lastIndexOf("/") + 1); // jars/zip files use foreslash as File.separator
        return name;
    }

    public String getFileNameOnlyFromJarEntry(JarEntry entry) {
        String name = getFileNameFromJarEntry(entry);
        name = name.substring(0,name.lastIndexOf("."));
        return name;
    }

    public List<String> getJarEntryNames(File f) {
        List<String> names = new ArrayList<>();
        for(int i = 0; i < entries.size();i++) {
            names.add(entries.get(i).getName());
        }
        return names;
    }

    public List<JarEntry> getJarEntries(File f) { return entries; }

    public String getJarName() { return file.getFileNameOnly(); }
    
    public List<String> getJarsPaths() {
        List<String> jarsPaths = new ArrayList<>();
        String name;
        String subPath;
        for(int i = 0; i < entries.size();i++) {
            name = entries.get(i).getName();
            subPath = name.replace("/",File.separator);
            jarsPaths.add(subPath);
        }
        return jarsPaths;
    }
    
    public static List<JarEntry> getJarsEntries(File f) { 
        Jar jar = new Jar(f);
        List<JarEntry> jars = new ArrayList<>();
        List<JarEntry> entries = jar.getEntries();
        for(int i = 0; i < entries.size();i++) {
            JarEntry entry = entries.get(i);
            String name = entry.getName();
            if (name.toLowerCase().startsWith("jars/")) {
                jars.add(entry);
            }
        }
        return jars;
    }

    public List<String> getSortedByPackage(List<String> classes) {
        List<String> defaultPkg = new ArrayList<>();
        List<String> packages = new ArrayList<>();
        for(int i = classes.size() - 1; i >= 0; i--) {
            if (getDotCount(classes.get(i)) == 0) {
                defaultPkg.add(classes.get(i));
                classes.remove(i);
            } else {
                packages.add(classes.get(i));
                classes.remove(i);
            }
        }
        defaultPkg = Sort.sort(defaultPkg);
        List<String> sorted = new ArrayList<>();
        for(int i = 0; i < defaultPkg.size();i++) {
            sorted.add(defaultPkg.get(i));
        }
        packages = sortByPackage(packages);
        //System.out.println("Packages: " + packages);
        for(int i = 0; i < packages.size();i++) {
            sorted.add(packages.get(i));
        }
        return sorted;
    }
    
    public String getTecreationsUnpackPath() {
        String path =  ProjectPath.getTecreationsPath() + "jars" + File.separator;
        String jarName = getJarName();
        if (jarName.contains("_jars")) {
            jarName = jarName.substring(0,jarName.indexOf("_jars"));
        }
        path += jarName + File.separator + "jars" + File.separator;
        return path;
    }
    
    public Long getUnpackedSize() { return unpackedSize; }

    public String getUnwrapped() { return file.getUnwrapped(); }
    
    public static boolean isJarRuntime() {
        return ProjectPath.getRuntimePath(Jar.class.getProtectionDomain()).endsWith(".jar");
    }
    
    public boolean isUnpacked() {
        String outPath = getTecreationsUnpackPath();
        List<JarEntry> jarEntries = getEntries();
        for(int i = 0; i < jarEntries.size();i++) {
            //System.out.println("JarEntry[" + i + "]: " + jarEntries.get(i).getName());
            String entry = jarEntries.get(i).getName();
            String name = entry;
            if (name.contains("/")) name = name.substring(name.lastIndexOf("/") + 1);
            if (name.toLowerCase().endsWith(".jar")) {
                name = name.substring(0,name.lastIndexOf("."));
                if (!new File(outPath + name).exists()) return false;
                
            }
        }
        return false;
    }
    
    public static void list(String path) {
        File[] list = new File(path).tecListFiles();
        for(int i = 0; i < list.length;i++) {
            if (list[i].isDirectory()) list(list[i].getAbsolutePath());
            else {
                System.out.println("list: " + list[i].getAbsolutePath());
            }
        }
    }
    
    public List<String> sortByPackage(List<String> classes) {
        List<List<String>> newList = new ArrayList<>();
        int maxDots = 0;
        for(int i = 0; i < classes.size();i++) {
            maxDots = Math.max(maxDots,getDotCount(classes.get(i)));
        }
        List<String> sortedByDots;
        for(int i = 1; i < maxDots;i++) {
            sortedByDots = new ArrayList<>();
            for(int j = classes.size() - 1;j >= 0; j--) {
                if (getDotCount(classes.get(j)) == i) {
                    sortedByDots.add(classes.get(j));
                    classes.remove(j);
                }
            }
            newList.add(sortedByDots);
        }
        List<String> sorted = new ArrayList<>();
        List<List<String>> sortedPkgs = new ArrayList<>();
        for(int i = 0; i < newList.size();i++) {
            List<String> sortedPkg = Sort.sort(newList.get(i));
            sortedPkgs.add(sortedPkg);
        }
        for(int i = 0; i < sortedPkgs.size();i++) {
            List<String> list = sortedPkgs.get(i);
            for(int j = 0; j < list.size();j++) {
                sorted.add(list.get(j));
            }
        }
        return sorted;
    }
}
 