package ca.tecreations.graphics;

import ca.tecreations.TColor;
import ca.tecreations.Point;

import java.awt.Color;
import java.awt.Graphics;
import java.util.ArrayList;
import java.util.List;
import ca.tecreations.interfaces.Paintable;

/**
 *
 * @author Tim
 */
public abstract class DrawObject implements Paintable {
    
    public static int PICK_SIZE_TO_MOVE = 3;
    public static int PICK_SIZE_TO_SELECT = 5;
    public String name;
    List<Integer> rotations = new ArrayList<>();
    List<Point> translations = new ArrayList<>();
    public int width;
    public int height;
    TColor lineColor = TColor.black;
    TColor fillColor = null;
    protected List<Point> linePoints = null;
    protected List<Point> fillPoints = null;

    public static List<Point> add(List<Point> src, List<Point> dst) {
        for(int i = 0; i < src.size();i++) {
            dst.add(src.get(i));
        }
        return dst;
    }

    
    public DrawObject addRotation(int i) {
        rotations.add(getNormalAngle(i));
        return this;
    }
    
    public void addTranslation(int x, int y) { translations.add(new Point(x,y)); }
    public void addTranslation(Point p) { translations.add(p); }

    public abstract void draw(Graphics g);
    public abstract void draw(Graphics g,int offsetX, int offsetY);

    public void draw(Graphics g, Point offsetXY) {
        draw(g,offsetXY.x,offsetXY.y);
    }

    public abstract void draw(Graphics g, int offsetX, int offsetY, Color lineColor);
    public abstract void draw(Graphics g, int offsetX, int offsetY, Color lineColor,Color fillColor);
    
    public void drawPoint(Graphics g, Point txy,Point p) {
        setPixel(g, txy.x + p.x, txy.y + p.y);
    }

    public void drawPoints(Graphics g, Point txy, List<Point> list) {
        Point p;
        if (list != null) {
            for (int i = 0; i < list.size(); i++) {
                p = list.get(i);
                setPixel(g, txy.x + p.x, txy.y + p.y);
            }
        }
    }
 
//    public abstract String getBlock();
    
    public Color getFillColor() { return fillColor; }
    
    public Color getLineColor() { return lineColor; }
    
    public List<Point> getFillPoints() { return fillPoints; }
    
    public List<Point> getLinePoints() { return linePoints; }
    
    public String getName() { return name; }
    
    public static int getNormalAngle(int angle) { 
//        System.out.println("receiving: " + angle);
        if (angle >= 360) {
            angle -= 360;
            return getNormalAngle(angle);
        } else {
            if (angle < 0) return getNormalAngle(360 + angle);
        }
//        System.out.println("sending  : " + angle);
        return angle;
    }
    
    public static double getNormalAngle(double angle) {
        if (angle >= 360) {
            angle -= 360;
            return getNormalAngle(angle);
        } else {
            if (angle < 0) return getNormalAngle(360 + angle);
        }
        return angle;
    }
    
    public int getRotation() {
        int rotation = 0;
        for(int i = 0; i < rotations.size();i++) {
            rotation += rotations.get(i);
        }
        int val = getNormalAngle(rotation);
        return val;
    }
    
    public int getTranslationsX() {
        int tx = 0;
        for(int i = 0; i < translations.size();i++) {
            tx += translations.get(i).x;
        }
        return tx;
    }
    
    public int getTranslationsY() {
        int ty = 0;
        for(int i = 0; i < translations.size();i++) {
            ty += translations.get(i).y;
        }
        return ty;
    }

    public int getTX() { return getTranslationsX(); }
    public int getTY() { return getTranslationsY(); }
    
    public Point getTXY() {
        return new Point(getTX(),getTY());
    }
    
    public int getXOffset(Point p1, Point p2) {
        if (p1.x < p2.x) {
            return p2.x - p1.x;
        } else {
            return -(p1.x - p2.x);
        }
    }
    
    public int getYOffset(Point p1, Point p2) {
        if (p1.y < p2.y) {
            return p2.y - p1.y;
        } else {
            return -(p1.y - p2.y);
        }
    }
    
    public abstract boolean hasFillPoint(Point p);
     
    public abstract boolean hasPoint(Point p);
    
    public void moveBy(int dx, int dy) {
        addTranslation(dx,dy);
    }
    
    public void setFillColor(Color fill) {
        if (fill == null) fill = TColor.getRandom();
        fillColor = new TColor(fill);
    }
    
    public void setLineColor(Color line) { lineColor = new TColor(line); }
    
    public DrawObject setName(String name) { this.name = name; return this;  }
    
    public DrawObject setPixel(Graphics g, int x, int y) {
        g.drawLine(x,y,x,y);
        return this;
    }
    
    public DrawObject setRotation(int angle) {
        rotations = new ArrayList<>();
        return addRotation(angle);
    }
    
    public DrawObject setTranslation(int x, int y) {
        translations = new ArrayList<>();
        translations.add(new Point(x,y));
        return this;
    }
    
    public DrawObject setTranslation(Point p) {
        setTranslation(p.x,p.y);
        return this;
    }
    
    public DrawObject setTXY(Point p) {
        setTranslation(p);
        return this;
    }
    
    public DrawObject setTXY(int x, int y) {
        setTranslation(x,y);
        return this;
    }
    
    public void stroke(Graphics g, DrawObject o, int xOff, int yOff) {
        // save the original txy
        Point oldTXY = o.getTXY();
        // offset the object
        o.setTranslation(getTranslationsX() + xOff,getTranslationsY() + yOff);
        // and draw it
        o.draw(g,0,0,o.getLineColor(),o.getFillColor());
        // return to previous state
        o.setTranslation(oldTXY.x,oldTXY.y);
    }

    public static void main(String[] args) {
        Circle circle = new Circle(25);
        System.out.println(circle);
        circle.addRotation(100);
        System.out.println(circle);
        circle.addRotation(-20);
        System.out.println(circle);
        circle.addRotation(-360);
        System.out.println(circle);
        circle.addRotation(360);
        System.out.println(circle);
        
        System.out.println("getNormalAngle(-1)   : " + circle.getNormalAngle(-1));
        System.out.println("getNormalAngle(-89)  : " + circle.getNormalAngle(-89));
        System.out.println("getNormalAngle(-179) : " + circle.getNormalAngle(-179));
        System.out.println("getNormalAngle(-269) : " + circle.getNormalAngle(-269));
        System.out.println("getNormalAngle(-359) : " + circle.getNormalAngle(-359));
        System.out.println("getNormalAngle(-360) : " + circle.getNormalAngle(360));
        System.out.println("getNormalAngle(360)  : " + circle.getNormalAngle(360));
        System.out.println("getNormalAngle(0)    : " + circle.getNormalAngle(0));
        System.out.println("getNormalAngle(720)  : " + circle.getNormalAngle(720));
        System.out.println("getNormalAngle(1440) : " + circle.getNormalAngle(1440));
        System.out.println("getNormalAngle(-720) : " + circle.getNormalAngle(-720));
        System.out.println("getNormalAngle(-1440): " + circle.getNormalAngle(-1440));
    }
}
