package ca.tecreations.text;

import ca.tecreations.Statics;
import ca.tecreations.TecData;
import ca.tecreations.File;
import ca.tecreations.Font;
import ca.tecreations.Point;
import ca.tecreations.ProjectPath;
import ca.tecreations.Properties;
import ca.tecreations.TextToken;
import ca.tecreations.components.Magnifier;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.font.FontRenderContext;
import java.awt.font.LineMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.ImageObserver;
import java.awt.image.PixelGrabber;
import java.util.ArrayList;
import java.util.List;

/**
 *
 * @author Tim
 */
public class TextPoints {

    // so we can avoid re-creating and re-loading
    public static List<String> fontList = new ArrayList<>();
    public static List<TextPoints> textPoints = new ArrayList<>();

    public Font tecFont;

    public static String ascii = "";

    static {
        for (int i = 0; i < 256; i++) {
            ascii += (char) i;
        }
    }
    private BufferedImage temp = null;
    protected List<List<Point>> points = new ArrayList<>();
    private int[] widths; // configured in doWidthCalculations();
    private List<Point> pList; // temp variable used when computing points list
    protected LineMetrics lm = null;
    int maxWidth = 0;
    int maxHeight = 0; 
    int ascent;
    int baseline;
    int descent;
    boolean monospaced = false;

    boolean debug = false;

    //--------------------------------------------------------------------------
    
    public static TextPoints getInstance(String name, int size, String style) {
        if (style.equalsIgnoreCase("b")) {
            return getInstance(name,Font.BOLD,size);
        } else if (style.equalsIgnoreCase("i")) {
            return getInstance(name,Font.ITALIC,size);
        } else if (style.equalsIgnoreCase("bi")) {
            return getInstance(name,Font.BOLD & Font.ITALIC,size);
        } else if (style.equalsIgnoreCase("ib")) {
            return getInstance(name,Font.BOLD & Font.ITALIC,size);
        } else if (style.equalsIgnoreCase("bold")) {
            return getInstance(name,Font.BOLD,size);
        } else if (style.equalsIgnoreCase("italic")) {
            return getInstance(name,Font.ITALIC,size);
        } else if (style.equalsIgnoreCase("bold+italic")) {
            return getInstance(name,Font.BOLD & Font.ITALIC,size);
        } else if (style.equalsIgnoreCase("italic+bold")) {
            return getInstance(name,Font.BOLD & Font.ITALIC,size);
        }
        return getInstance(name,Font.PLAIN,size);
    }

    public static TextPoints getInstance(String name, String style, int size) {
        return getInstance(name,size,style);
    }

    public static TextPoints getInstance(String name, int style, int size) {
        return getInstance(new Font(name, style, size));
    }

    public static TextPoints getInstance(String name, int style, int size, boolean monospaced) {
        return getInstance(new Font(name, style, size), monospaced);
    }

    public static TextPoints getInstance(java.awt.Font font) {
        return getInstance(new Font(font));
    }

    public static TextPoints getInstance(java.awt.Font font, boolean monospaced) {
        return getInstance(new Font(font),monospaced);
    }

    public static TextPoints getInstance(ca.tecreations.Font font) {
        String fontFile = getFontFilename(font,false);
        if (!fontList.contains(fontFile)) {
            return new TextPoints(font,false);
        }
        return textPoints.get(fontList.indexOf(fontFile));
    }

    public static TextPoints getInstance(ca.tecreations.Font font, boolean monospaced) {
        String fontFile = getFontFilename(font,monospaced);
        if (!fontList.contains(fontFile)) {
            new TextPoints(font,monospaced);
        }
        return textPoints.get(fontList.indexOf(fontFile));
    }

    private TextPoints(Font font,boolean monospaced) {
        tecFont = font;
        String fontFile = getFontFilename(font,monospaced);
        if (!new File(ProjectPath.getFontsPath() + fontFile).exists()) {
            if (debug) {
                System.out.println("TextPoints(): Creating: " + fontFile);
            }
            doProcessing();
            doWidthCalculations();
            
            // it looks like underscore isn't saving any points
            pList = new ArrayList<>();
            for(int i = 0; i < maxWidth;i++) {
                pList.add(new Point(i,getFontSize() - 1)); // we'll actually draw in
                                                       // bottom-most line
            }
            points.set(95,pList);
            
            save(fontFile);
            pList = null;  // don't need this anymore, used in process() and addIfBlack();
        } else {
            if (debug) {
                System.out.println("TextPoints() Loading: " + fontFile);
            }
            points = load(fontFile);
            doWidthCalculations();
        }
        fontList.add(fontFile);
        textPoints.add(this);
    }

    // this is the next separator, for the next block, which incidentally, is in
    // the second position.
    //==========================================================================

    
    // next, add methods classified by Alphabetical, Public &| Protected &| Private
    // Put return type as the next priority, so you would arrive at something like
    // this classes definition.
    public TextPoints addBold() {
        return getInstance(getFontName(), getFontStyle() | Font.BOLD, getFontSize(),monospaced);
    }

    public TextPoints addItalic() {
        return getInstance(getFontName(), getFontStyle() | Font.ITALIC, getFontSize(),monospaced);
    }

    // process once, after that, look it up
    public void doProcessing() {
        for (int i = 0; i < ascii.length(); i++) {
            process(ascii.charAt(i));
        }
    }

    /**
     * Assigns the widths of the individual characters Assigns maxWidth
     *
     * @see Font.addPoints(List<Point>)
     */
    public void doWidthCalculations() {
        List<Integer> zeros = new ArrayList<>();
        List<Point> list;
        int width;
        widths = new int[256];
        if (monospaced) {
            list = points.get((int) 'W');
            int capDubWidth = getMaxX(list);
            for (int i = 0; i < widths.length; i++) {
                widths[i] = capDubWidth;
            }
            maxWidth = capDubWidth;
        } else {
            for (int i = 0; i < 256; i++) {
                list = points.get(i);
                if (list.size() > 0) {
                    width = getMaxX(list);
                    widths[i] = width; // here is the width for the particular char;
                    maxWidth = Math.max(width, maxWidth);
                    //if (debugSizes) System.out.println(i + ", '" + (char)i + "' :, Width: " + width + ", MaxWidth: " + maxWidth);
                } else {
                    // probably this should be all non-printable and cursor controls
                    zeros.add(i);
                }
            }
            for (int i = 0; i < zeros.size(); i++) {
                if (debug) {
                    System.out.println("Zero Width Char: [" + i + "]: '" + (char) i + "'");
                }
                widths[zeros.get(i)] = maxWidth;
            }

            // set our unknown character widths
            int single = widths[ascii.indexOf('W')];
            widths[ascii.indexOf('-')] = widths[ascii.indexOf('P')];
            widths[ascii.indexOf('_')] = widths[ascii.indexOf('P')];
            widths[ascii.indexOf(' ')] = widths[ascii.indexOf('P')];
            widths[ascii.indexOf('|')] = single;
            widths[ascii.indexOf('\t')] = single * TecData.TAB_SIZE;
        }
    }

    public boolean equals(TextPoints tp2) {
        boolean equal = true;
        List<Point> char1;
        List<Point> char2;
        Point char1Target;
        Point char2Target;
        for (int i = 0; i < 256; i++) {
            char1 = getPoints(i);
            char2 = tp2.getPoints(i);
            if (char1.size() == char2.size()) {
                for (int j = 0; j < char1.size(); j++) {
                    char1Target = char1.get(j);
                    boolean present = false;
                    for (int k = 0; k < char2.size(); k++) {
                        char2Target = char2.get(k);
                        if (char2Target.equals(char1Target)) {
                            present = true;
                            k = char2.size();
                        }
                    }
                    if (!present) {
                        return false;
                    }
                }
            } else {
                return false;
            }
        }
        return true;
    }

    public int getAscent() {
        return ascent;
    }

    public int getBaseline() {
        return baseline;
    }

    public TextPoints getBold() {
        return getInstance(getFontName(), Font.BOLD, getFontSize(),monospaced);
    }

    public TextPoints getBoldItalic() {
        return getInstance(getFontName(), Font.BOLD | Font.ITALIC, getFontSize(),monospaced);
    }

    public static TextPoints getCodePoints() {
        TextPoints points = getInstance("Courier New", Font.BOLD, 10);
        points.setMonospaced(true);
        return points;
    }

    public static TextPoints getCodeBoldPoints14() {
        TextPoints points = getInstance("Courier New", Font.BOLD, 14);
        points.setMonospaced(true);
        return points;
    }

    public static TextPoints getCodeBoldPoints18() {
        TextPoints points = getInstance("Courier New", Font.BOLD, 18);
        points.setMonospaced(true);
        return points;
    }

    public static TextPoints getCodePlainPoints() {
        return getInstance("Courier New", Font.PLAIN, 14);
    }

    public static TextPoints getCodePlainPoints18() {
        return getInstance("Courier New", Font.PLAIN, 18);
    }

    public static TextPoints getCodePoints12() {
        return getInstance("Courier New", Font.PLAIN, 12);
    }

    public int getMaxDescent() { return descent; }
    
    public static String getDefaultString() {
        return ascii;
    }

    public int getDescent() {
        return descent;
    }

    public String getFilename() {
        return tecFont.getName() + "." + tecFont.getStyleCode() + "." + tecFont.getSize() + ".font";
    }

    public Font getFont() {
        return tecFont;
    }

    public static String getFontFilename(ca.tecreations.Font font,boolean monospaced) {
        String name = font.getNameForPath();
        name += "." + font.getSize() + "." + getFontStyleCode(font);
        if (monospaced) name += ".mono";
        name += ".font";
        return name;
    }

    public String getFontFilename(java.awt.Font font) {
        return getFontFilename(new Font(font.getName(), font.getStyle(), font.getSize()),monospaced);
    }
    
    public String getFontName() {
        return tecFont.getName();
    }

    public int getFontSize() {
        return tecFont.getSize();
    }

    public int getFontStyle() {
        return tecFont.getStyle();
    }

    public static String getFontStyleCode(Font font) {
        if (font.getStyle() == (Font.BOLD + Font.ITALIC)) {
            return "BI";
        }
        if (font.getStyle() == Font.BOLD) {
            return "B";
        }
        if (font.getStyle() == Font.ITALIC) {
            return "I";
        }
        return "P";
    }

    public String getFontStyleString() {
        if (tecFont.getStyle() == (Font.BOLD + Font.ITALIC)) {
            return "BOLD+ITALIC";
        }
        if (tecFont.getStyle() == Font.BOLD) {
            return "BOLD";
        }
        if (tecFont.getStyle() == Font.ITALIC) {
            return "ITALIC";
        }
        return "PLAIN";
    }

    public TextPoints getItalic() {
        return getInstance(getFontName(), Font.ITALIC, getFontSize());
    }

    public java.awt.Font getJavaFont() {
        return tecFont.getJavaFont();
    }
    
    public int getMaxHeight() {
        return maxHeight;
    }

    public int getMaxWidth() {
        return maxWidth;
    }

    public int getMaxX(List<Point> list) {
        int maxX = 0;
        for (int i = 0; i < list.size(); i++) {
            maxX = Math.max(list.get(i).x, maxX);
        }
        return maxX;
    }

    public int getMaxY(List<Point> list) {
        int maxY = 0;
        for (int i = 0; i < list.size(); i++) {
            maxY = Math.max(list.get(i).y, maxY);
        }
        return maxY;
    }

    public TextPoints getMonospaced() {
        TextPoints instance = getInstance(getFontName(),getFontStyle(),getFontSize(),true);
        instance.setMonospaced(true);
        return instance;
    }
    
    public String getName() {
        return getFontName();
    }

    public TextPoints getNamed(int index) {
        return getInstance(Font.getName(index), getFontStyle(), getFontSize(),monospaced);
    }

    public TextPoints getPlain() {
        return getInstance(getFontName(), Font.PLAIN, getFontSize(),monospaced);
    }

    public String getPointList(List<Point> points) {
        String s = "";
        for (int i = 0; i < points.size(); i++) {
            s += points.get(i).getAsCSV() + ";";
        }
        return s;
    }

    public List<Point> getPoints(char c) {
        return points.get(ascii.indexOf(c));
    }

    public List<Point> getPoints(int i) {
        return points.get(i);
    }

    public Dimension getSampleWidthAndHeight() {
        int width = 0;
        for (int i = 0; i < widths.length; i++) {
            width += widths[i] + 1; // java pen is down and to the right, so add 1
        }
        return new Dimension(width, getFontSize());
    }

    public int getSize() {
        return getFontSize();
    }

    public TextPoints getSized(int size) {
        return getInstance(getFontName(), getFontStyle(), size, monospaced);
    }

    public static String getString() {
        return ascii;
    }

    public int getStyle() {
        return getFontStyle();
    }

    public int getTextWidth(String text) {
        int width = 0;
        for (int i = 0; i < text.length(); i++) {
            width += getWidth(text.charAt(i));
        }
        return width;
    }

    public int getWidth(int i) {
        if (monospaced) {
            return widths[(int) 'W'];
        } else {
            return widths[i];
        }
    }

    public int getWidth(String s) {
        if (s.length() == 0 | s.length() > 1) throw new IllegalArgumentException("String 's' must be exactly 1 character long: " + s);
        return widths[(int)s.charAt(0)];
    }
    
    public boolean isBlack(int x, int y, int pixel) {
        int alpha = (pixel >> 24) & 0xff;
        int maxX = 0;
        int red = (pixel >> 16) & 0xff;
        int green = (pixel >> 8) & 0xff;
        int blue = (pixel) & 0xff;
        if (red == 0 && green == 0 && blue == 0) {
            return true;
        }
        return false;
    }

    public boolean isMonospaced() {
        return monospaced;
    }

    private List<List<Point>> load(String filename) {
        Properties properties = new Properties(ProjectPath.getFontsPath() + filename);
        maxHeight = properties.getInt("max.height");
        ascent = properties.getInt("ascent");
        baseline = properties.getInt("baseline");
        descent = properties.getInt("descent");
        List<List<Point>> points = new ArrayList<>();
        for (int i = 0; i < 256; i++) {
            List<Point> indexPoints = Statics.getListOfPoint(properties.get(i));
            points.add(indexPoints);
        }
        return points;
    }

    public static void main(String[] args) {
        new TextPoints_DeleteTecreationsFonts();
        //TextPointsMono points = new TextPointsMono("Courier New",Font.BOLD,18);
        TextPoints points = getInstance(Font.MONOSPACED, Font.BOLD, 18);
        points.setMonospaced(true);
        System.out.println("Default: '" + getDefaultString() + "'");
        SystemTokenPainter painter = new SystemTokenPainter(points, new TextToken(getDefaultString()));
        BufferedImage image = painter.getImage();
        Magnifier magnifier = new Magnifier(image);
        magnifier.setVisible(true);
        String ascii = points.getDefaultString();
        for(int i = 0; i < ascii.length();i++) {
            System.out.println("'" + ascii.charAt(i) + "' : " + i);
        }
        System.out.println("Currently: " + points.getFont().toString());
        System.out.println("Font.size: " + points.getFontSize());
        System.out.println("MaxHeight: " + points.getMaxHeight());
    }

    private void process(char c) {
        int[] pix = null;
        pList = new ArrayList<>();
        // so first create a Graphics context, we'll do this using BufferedImage

        // we want it 2x2 where each cell is the height and height of the font, so that's wider and taller
        int side = tecFont.getSize() * 2;
        temp = new BufferedImage(side, side, BufferedImage.TYPE_INT_RGB);

        // draw the backgrounnd, to white for contrast against textColor
        Graphics g = temp.getGraphics();
        g.setFont(tecFont.getJavaFont());
        g.fillRect(0, 0, side, side);

        Graphics2D g2d = (Graphics2D) g;
        FontRenderContext context = g2d.getFontRenderContext();
        Rectangle bounds = tecFont.getJavaFont().getStringBounds("" + c, context).getBounds();
        if (lm == null) {
            lm = tecFont.getJavaFont().getLineMetrics("" + c, context);
            ascent = (int) Math.abs(lm.getAscent());
            descent = (int) Math.abs(lm.getDescent());
            baseline = maxHeight - descent;
        }
        int width = (int) bounds.getWidth();
        int height = (int) bounds.getHeight();

        // draw the points...

        /*
        BufferedImage image = ImageTool.getNewBufferedImage(side,side);
        Graphics ig = image.getGraphics();
        ig.setColor(Color.black);
        ig.setFont(tecFont.getJavaFont());
         */
        g.setColor(Color.black);
        g.setFont(tecFont.getJavaFont());

        int y = (int) (lm.getHeight() - lm.getDescent());
        g.drawString("" + c, 0, y);
        /*            ig.drawString("" + c,0,y);
            Magnifier2 magnifier = new Magnifier2(image);
            magnifier.setVisible(true);
         */
        // and determine which pixels were actually painted
        pix = new int[side * side];
        PixelGrabber pixelGrabber = new PixelGrabber(temp, 0, 0, side, side, pix, 0, side);
        try {
            pixelGrabber.grabPixels();
        } catch (InterruptedException e) {
            System.err.println("interrupted waiting for pixels!");
            return;
        }
        if ((pixelGrabber.getStatus() & ImageObserver.ABORT) != 0) {
            System.err.println("image fetch errored or aborted");
            return;
        }
        if (c == '|') {
            for (int j = 0; j < side; j++) {
                for (int i = 0; i < side; i++) {
                    if (isBlack(i, j, pix[j * side + i])) {
                        pList.add(new Point(i, j));
                        maxHeight = Math.max(maxHeight, j);
                    }
                }
            }
        } else {
            for (int j = 0; j < side; j++) {
                for (int i = 0; i < side; i++) {
                    if (isBlack(i, j, pix[j * side + i])) {
                        pList.add(new Point(i, j));
                        maxHeight = Math.max(maxHeight, j);
                    }
                }
            }
        }
        points.add(pList);
    }

    public void save(String filename) {
        Properties properties = new Properties(ProjectPath.getFontsPath() + filename);
        properties.set("max.height", maxHeight);
        properties.set("ascent", ascent);
        properties.set("baseline", baseline);
        properties.set("descent", descent);
        for (int i = 0; i < 256; i++) {
            // are you saying we should prefix with width for character?
            List<Point> points = getPoints(i);
            String list = getPointList(points);
            properties.set("" + i, list);
        }
    }

    public void setMonospaced(boolean state) {
        monospaced = state;
//        doWidthCalculations();
    }

    public String toString() {
        return "Font: " + tecFont.getName() + "," + tecFont.getStyleCode() + ", Height: " + maxHeight + ", maxWidth: " + maxWidth;
    }
}
